<?php
/**
 * EnrollmentGroups class checking if a contact is enrolled in a specific group or not.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;


/**
 * EnrollmentGroups class checking if a contact is enrolled in a specific group or not.
 * 
 * @since 1.17.2
 */
class EnrollmentGroups {

	/**
	 * Checks if any group ID from the given groups array matches any group ID in the automation data.
	 *
	 * @param array $groups The array of groups to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a group ID matches a group ID, false otherwise.
	 * @since 1.17.2
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		$group_ids = array_values( array_column( $value, 'value' ) );

		if ( empty( $group_ids ) ) {
			return false;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$groups  = learndash_get_users_group_ids($user_id);

		if (!$groups) {
			return false;
		}

		$intersect = array_intersect( $groups, $group_ids );

		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if any group ID from the given groups array does not match any group ID in the automation data.
	 *
	 * @param array $groups The array of groups to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a group ID does not match a group ID, false otherwise.
	 * @since 1.17.2
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email)) {
			return true;
		}

		$group_ids = array_values(array_column($value, 'value'));
		if (empty($group_ids)) {
			return true;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$groups  = learndash_get_users_group_ids($user_id);

		if (!$groups) {
			return true;
		}

		$intersect = array_intersect($groups, $group_ids);

		if (empty($intersect)) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if all group IDs from the given groups array match all group IDs in the automation data.
	 *
	 * @param array $groups The array of groups to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if all group IDs match all group IDs, false otherwise.
	 * @since 1.17.2
	 */
	public function includes_all_of( array $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email)) {
			return false;
		}

		$group_ids = array_values(array_column($value, 'value'));
		if (empty($group_ids)) {
			return false;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$groups  = learndash_get_users_group_ids($user_id);

		if (!$groups) {
			return false;
		}

		$diff = array_diff($group_ids, $groups);

		if (empty($diff)) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the group IDs from the given groups array match any group ID in the automation data.
	 *
	 * @param array $groups The array of groups to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if none of the group IDs match any group ID, false otherwise.
	 * @since 1.17.2
	 */
	public function includes_none_of( array $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email)) {
			return true;
		}

		$group_ids = array_values(array_column($value, 'value'));
		if (empty($group_ids)) {
			return true;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$groups  = learndash_get_users_group_ids($user_id);

		if (!$groups) {
			return true;
		}

		$intersect = array_intersect($groups, $group_ids);

		if (empty($intersect)) {
			return true;
		}

		return false;
	}
}
