<?php

/**
 * EnrollmentCourses class checking if a contact is enrolled in a specific course or not.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;


/**
 * EnrollmentCourses class checking if a contact is enrolled in a specific course or not.
 * 
 * @since 1.17.2
 */
class EnrollmentCourses {

	/**
	 * Checks if any course ID from the given courses array matches any course ID in the automation data.
	 *
	 * @param array $courses The array of courses to check for matching course IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a course ID matches a course ID, false otherwise.
	 * @since 1.17.2
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';

		if ( empty( $user_email ) ) {
			return false;
		}

		$course_ids = array_values( array_column( $value, 'value' ) );
		if ( empty( $course_ids ) ) {
			return false;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$courses = learndash_user_get_enrolled_courses($user_id);

		if (!$courses) {
			return false;
		}

		$intersect = array_intersect( $courses, $course_ids );

		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if any course ID from the given courses array does not match any course ID in the automation data.
	 *
	 * @param array $courses The array of courses to check for matching course IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a course ID does not match a course ID, false otherwise.
	 * @since 1.17.2
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email)) {
			return true;
		}

		$course_ids = array_values(array_column($value, 'value'));
		if (empty($course_ids)) {
			return true;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$courses = learndash_user_get_enrolled_courses($user_id);

		if (!$courses) {
			return true;
		}

		$intersect = array_intersect($courses, $course_ids);

		if (empty($intersect)) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if all course IDs from the given courses array match all course IDs in the automation data.
	 *
	 * @param array $courses The array of courses to check for matching course IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if all course IDs match all course IDs, false otherwise.
	 * @since 1.17.2
	 */
	public function includes_all_of( array $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email)) {
			return false;
		}

		$course_ids = array_values(array_column($value, 'value'));
		if (empty($course_ids)) {
			return false;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$courses = learndash_user_get_enrolled_courses($user_id);

		if (!$courses) {
			return false;
		}

		$diff = array_diff($course_ids, $courses);

		if (empty($diff)) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the course IDs from the given courses array match any course ID in the automation data.
	 *
	 * @param array $courses The array of courses to check for matching course IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if none of the course IDs match any course ID, false otherwise.
	 * @since 1.17.2
	 */
	public function includes_none_of( array $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email)) {
			return true;
		}

		$course_ids = array_values(array_column($value, 'value'));
		if (empty($course_ids)) {
			return true;
		}

		$user    = get_user_by('email', $user_email);
		$user_id = $user->ID;
		$courses = learndash_user_get_enrolled_courses($user_id);

		if (!$courses) {
			return true;
		}

		$intersect = array_intersect($courses, $course_ids);

		if (empty($intersect)) {
			return true;
		}

		return false;
	}
}
