<?php
/**
 * Email Click  Class for Checking Email open
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2022-08-09 11:03:17
 * @modify date 2022-08-09 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use DateTime;
use Mint\MRM\DataBase\Tables\EmailMetaSchema;
use Mint\MRM\DataBase\Tables\EmailSchema;
use Mint\Mrm\Internal\Traits\Singleton;

use MintMail\App\Internal\Automation\ActionScheduler;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Email Click  Class for Checking Email open
 */
class EmailClicked {
	use Singleton;

	/**
	 * Action scheduler.
	 *
	 * @var object
	 */
	private $action_scheduler;

	/**
	 * Broadcast email table name
	 *
	 * @var string
	 */
	private $broadcast_email_table;

	/**
	 * Broadcast email meta table name
	 *
	 * @var string
	 */
	private $broadcast_email_meta_table;

	/**
	 * Initialization
	 */
	public function __construct() {
		global $wpdb;
		$this->action_scheduler           = new ActionScheduler();
		$this->broadcast_email_table      = $wpdb->prefix . EmailSchema::$table_name;
		$this->broadcast_email_meta_table = $wpdb->prefix . EmailMetaSchema::$table_name;
	}

	/**
	 * Get action name
	 *
	 * @return String
	 * @since  1.0.0
	 */
	public function get_name() {
		return __( 'User Role', 'mailmint-pro' );
	}

	/**
	 * Check if a broadcast email was clicked after a certain time.
	 *
	 * @param int|string $value The time value to check against (in MySQL datetime format).
	 * @param array      $autamation_data The automation data that contains the previous step and automation IDs and the user's email address.
	 * @return bool
	 * @since 1.1.2
	 */
	public function after( $value, $autamation_data ) {
		global $wpdb;
		$send_email_step = HelperFunctions::find_send_mail_action( $autamation_data );
		$autamation_id   = isset( $send_email_step['automation_id'] ) ? $send_email_step['automation_id'] : '';
		$step_id         = isset( $send_email_step['step_id'] ) ? $send_email_step['step_id'] : '';
		$email           = isset( $autamation_data['data']['user_email'] ) ? $autamation_data['data']['user_email'] : '';

        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT m.id FROM {$this->broadcast_email_meta_table} m 
                                                LEFT JOIN {$this->broadcast_email_table} e ON m.mint_email_id = e.id
                                                WHERE e.email_address = %s
                                                AND e.automation_id = %d
                                                AND e.step_id = %s
                                                AND m.meta_key = %s
                                                AND m.created_at > %s",
				$email,
				$autamation_id,
				$step_id,
				'is_click',
				$value
			),
			ARRAY_A
		); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		if ( empty( $results ) ) {
			return false;
		}
		return true;
	}

	/**
	 * Check if a broadcast email was clicked before a certain time.
	 *
	 * @param int|string $value The time value to check against (in MySQL datetime format).
	 * @param array      $autamation_data The automation data that contains the previous step and automation IDs and the user's email address.
	 * @return bool
	 * @since 1.1.2
	 */
	public function before( $value, $autamation_data ) {
		global $wpdb;
		$send_email_step = HelperFunctions::find_send_mail_action( $autamation_data );
		$autamation_id   = isset( $send_email_step['automation_id'] ) ? $send_email_step['automation_id'] : '';
		$step_id         = isset( $send_email_step['step_id'] ) ? $send_email_step['step_id'] : '';
		$email           = isset( $autamation_data['data']['user_email'] ) ? $autamation_data['data']['user_email'] : '';
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT m.id FROM {$this->broadcast_email_meta_table} m
                                                LEFT JOIN {$this->broadcast_email_table} e ON m.mint_email_id = e.id
                                                WHERE e.email_address = %s
                                                AND e.automation_id = %d
                                                AND e.step_id = %s
                                                AND m.meta_key = %s
                                                AND m.created_at < %s",
				$email,
				$autamation_id,
				$step_id,
				'is_click',
				$value
			),
			ARRAY_A
		); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		if ( empty( $results ) ) {
			return false;
		}
		return true;
	}

	/**
	 * Check if a broadcast email was clicked in a certain time.
	 *
	 * @param int|string $value The time value to check against (in MySQL datetime format).
	 * @param array      $autamation_data The automation data that contains the previous step and automation IDs and the user's email address.
	 * @return bool
	 * @since 1.1.2
	 */
	public function in_the_date( $value, $autamation_data ) {
		global $wpdb;
		$date_time       = new DateTime( $value );
		$formatted_date  = $date_time->format( 'Y-m-d' );
		$send_email_step = HelperFunctions::find_send_mail_action( $autamation_data );
		$autamation_id   = isset( $send_email_step['automation_id'] ) ? $send_email_step['automation_id'] : '';
		$step_id         = isset( $send_email_step['step_id'] ) ? $send_email_step['step_id'] : '';
		$email           = isset( $autamation_data['data']['user_email'] ) ? $autamation_data['data']['user_email'] : '';

        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT m.id FROM {$this->broadcast_email_meta_table} m
                                                LEFT JOIN {$this->broadcast_email_table} e ON m.mint_email_id = e.id
                                                WHERE e.email_address = %s
                                                AND e.automation_id = %d
                                                AND e.step_id = %s
                                                AND m.meta_key = %s
                                                AND DATE(m.created_at) = %s",
				$email,
				$autamation_id,
				$step_id,
				'is_click',
				$formatted_date
			),
			ARRAY_A
		); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		if ( empty( $results ) ) {
			return false;
		}
		return true;
	}

}
