<?php
/**
 * EddPurchasedCategories class for checking if a contact has purchased a specific product or not.
 *
 * @author [WPFunnels Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * EddPurchasedCategories class for checking 
 * if a contact has purchased a specific product or not.
 * 
 * @since 1.2.7
 */
class EddPurchasedCategories {

	/**
	 * Checks if any segment ID from the given segments array matches any group ID in the automation data.
	 *
	 * @param array $segments The array of segments to check for matching group IDs.
	 * @param array $automation_data The automation data containing the relevant information.
	 *
	 * @return bool Returns true if a segment ID matches a group ID, false otherwise.
	 * @since 1.2.7
	 */
	public function included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';
		$customer   = edd_get_customer_by( 'email', $user_email );

		// Check if the customer exists.
		if ( ! $customer ) {
			return false;
		}

		// Fetch all completed orders for the customer.
		$orders = edd_get_orders( [
			'customer_id' => $customer->id,
			'status'      => 'complete',
			'number'      => -1,
		] );
	
		// Check if there are any orders.
		if ( empty( $orders ) || ! is_array( $orders ) ) {
			return false;
		}

		// Initialize an array to store purchased product IDs.
		$purchased_products = [];

		// Loop through each order
		foreach ( $orders as $order ) {
			$order_items = $order->get_items();
	
			// Extract product IDs and merge into the array.
			foreach ( $order_items as $item ) {
				$purchased_products[] = $item->product_id;
			}
		}
	
		// Remove duplicate product IDs.
		$purchased_products = array_unique( $purchased_products );
        if (empty($purchased_products)) {
            return false;
        }

		$purchased_category_ids = [];

		// Loop through each purchased product
		foreach ( $purchased_products as $product_id ) {
			// Get the category IDs (term IDs) associated with the product
			$category_ids = wp_get_post_terms( $product_id, 'download_category', [ 'fields' => 'ids' ] );

			// Merge category IDs into the main array
			$purchased_category_ids = array_merge( $purchased_category_ids, $category_ids );
		}

		// Remove duplicate category IDs
		$purchased_category_ids = array_unique( $purchased_category_ids );
		$product_cats           = array_values( array_column( $value, 'value' ) );

		// Find the intersection between group IDs and segment IDs.
		$intersect = array_intersect($purchased_category_ids, $product_cats);
		// If there is at least one common ID, return true.
		if ( !empty( $intersect ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
	 *
	 * @param array $segments           The value to check if it does not exist within the first name.
	 * @param array $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function not_included_in( array $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';
		$customer   = edd_get_customer_by( 'email', $user_email );

		// Check if the customer exists.
		if ( ! $customer ) {
			return false;
		}

		// Fetch all completed orders for the customer.
		$orders = edd_get_orders( [
			'customer_id' => $customer->id,
			'status'      => 'complete',
			'number'      => -1,
		] );
	
		// Check if there are any orders.
		if ( empty( $orders ) || ! is_array( $orders ) ) {
			return false;
		}

		// Initialize an array to store purchased product IDs.
		$purchased_products = [];

		// Loop through each order
		foreach ( $orders as $order ) {
			$order_items = $order->get_items();
	
			// Extract product IDs and merge into the array.
			foreach ( $order_items as $item ) {
				$purchased_products[] = $item->product_id;
			}
		}
	
		// Remove duplicate product IDs.
		$purchased_products = array_unique( $purchased_products );
        if (empty($purchased_products)) {
            return false;
        }

		$purchased_category_ids = [];

		// Loop through each purchased product
		foreach ( $purchased_products as $product_id ) {
			// Get the category IDs (term IDs) associated with the product
			$category_ids = wp_get_post_terms( $product_id, 'download_category', [ 'fields' => 'ids' ] );

			// Merge category IDs into the main array
			$purchased_category_ids = array_merge( $purchased_category_ids, $category_ids );
		}

		// Remove duplicate category IDs
		$purchased_category_ids = array_unique( $purchased_category_ids );
		$product_cats           = array_values( array_column( $value, 'value' ) );

		// Find the segment IDs that are not present in the group IDs.
		$diff = array_diff( $purchased_category_ids, $product_cats );

		if ( !empty( $diff ) ) {
			return true;
		}

		return false;
	}
}
