<?php
/**
 * State class for checking contact State
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use Mint\MRM\DataBase\Tables\ContactSchema;
use MintMailPro\Mint_Pro_Helper;
use Mint\MRM\DataBase\Models\ContactModel;


/**
 * State class for checking contact State
 */
class State {

	/**
	 * Checks if a given value is included in the State
	 *
	 * @param string $value The value to search for.
	 * @param array  $automation_data The automation data containing the State
	 *
	 * @return bool True if the value is included in the State, false otherwise.
	 * @since 1.2.7
	 */
	public function includes( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$postal = ContactModel::get_contact_meta_value_by_key($user_email, 'state');

		if (false !== stripos($postal, $value)) {
			return true;
		}
		return false;
	}

	/**
	 * Checks if the specified value does not exist within the given automation data's first name.
	 *
	 * @param string $value           The value to check if it does not exist within the first name.
	 * @param array  $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the State, false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_includes( string $value, array $automation_data ): bool {
		return !$this->includes( $value, $automation_data );
	}

	/**
	 * Checks if the State matches the specified value.
	 *
	 * @param string $value           The value to compare with the State.
	 * @param array  $automation_data The automation data containing the State.
	 *
	 * @return bool                   True if the State matches the value, false otherwise.
	 * @since 1.2.7
	 */
	public function equal( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$postal = ContactModel::get_contact_meta_value_by_key($user_email, 'state');
		return strcasecmp($value, $postal) === 0;
	}

	/**
	 * Checks if the Stateis not equal to the specified value (case-insensitive).
	 *
	 * @param string $value      The value to compare with the State.
	 * @param array  $automation_data The automation data containing the State.
	 *
	 * @return bool              True if the State is not equal to the value (case-insensitive), false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_equal( string $value, array $automation_data ): bool {

		return !$this->equal( $value, $automation_data );
	}

	/**
	 * Checks if the postal is empty.
	 *
	 * @param array $automation_data The automation data containing the postal.
	 *
	 * @return bool True if the postal is empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_empty(string $value, array $automation_data): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return true;
		}

		$postal = ContactModel::get_contact_meta_value_by_key($user_email, 'state');
		return empty($postal);
	}

	/**
	 * Checks if the postal is not empty.
	 *
	 * @param array $automation_data The automation data containing the postal.
	 *
	 * @return bool True if the postal is not empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_not_empty(string $value, array $automation_data): bool{
		return !$this->is_empty($value, $automation_data);
	}
}
