<?php

/**
 * Phone Number Condition Class
 *
 * Provides various conditions to evaluate phone number strings for automation logic.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-12-04 09:03:17
 * @modify date 2024-12-10 09:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;

use Mint\MRM\DataBase\Models\ContactModel;

/**
 * Class PhoneNumber
 *
 * Provides various conditions to evaluate phone number strings for automation logic.
 * Includes methods to check if a phone number contains, equals, starts with, or ends with a given value.
 * 
 * @since 1.17.1
 */
class PhoneNumber {

	/**
	 * Check if the phone number includes a given value (case-insensitive).
	 *
	 * @param string $value The substring to check within the phone number.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the phone number includes the value, false otherwise.
	 * @since 1.17.1
	 */
	public function includes( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$phone_number = ContactModel::get_contact_meta_value_by_key( $user_email, 'phone_number' );

		if ( false !== stripos( $phone_number, $value ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Check if the phone number does not include a given value (case-insensitive).
	 *
	 * @param string $value The substring to check within the phone number.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the phone number does not include the value, false otherwise.
	 * @since 1.17.1
	 */
	public function does_not_includes( string $value, array $automation_data ): bool {
		return !$this->includes( $value, $automation_data );
	}

	/**
	 * Check if the phone number equals a given value (case-insensitive).
	 *
	 * @param string $value The value to compare with the phone number.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the phone number equals the value, false otherwise.
	 * @since 1.17.1
	 */
	public function equal( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$phone_number = ContactModel::get_contact_meta_value_by_key($user_email, 'phone_number');

		return strcasecmp( $value, $phone_number ) === 0;
	}

	/**
	 * Check if the phone number does not equal a given value (case-insensitive).
	 *
	 * @param string $value The value to compare with the phone number.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the phone number does not equal the value, false otherwise.
	 * @since 1.17.1
	 */
	public function does_not_equal( string $value, array $automation_data ): bool {
		// Reuse the equal function by negating its result.
		return !$this->equal( $value, $automation_data );
	}

	/**
	 * Check if the phone number is empty.
	 *
	 * @param string $value Unused parameter for the logic but kept for consistency.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the phone number is empty, false otherwise.
	 * @since 1.17.1
	 */
	public function is_empty( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$phone_number = ContactModel::get_contact_meta_value_by_key($user_email, 'phone_number');
		return empty($phone_number);
	}

	/**
	 * Check if the phone number is not empty.
	 *
	 * @param string $value Unused parameter for the logic but kept for consistency.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the phone number is not empty, false otherwise.
	 * @since 1.17.1
	 */
	public function is_not_empty( string $value, array $automation_data ): bool {
		return !$this->is_empty($value, $automation_data);
	}
}
