<?php

/**
 * Gender Condition Class
 *
 * Provides various conditions to evaluate gender strings for automation logic.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-12-04 09:03:17
 * @modify date 2024-12-10 09:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;

use Mint\MRM\DataBase\Models\ContactModel;

/**
 * Gender class for checking contact Gender
 */
class Gender {

	/**
	 * Checks if the Gender matches the specified value.
	 *
	 * @param string $value           The value to compare with the Gender.
	 * @param array  $automation_data The automation data containing the Gender.
	 *
	 * @return bool                   True if the Gender matches the value, false otherwise.
	 * @since 1.17.0
	 */
	public function equal( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$gender = ContactModel::get_contact_meta_value_by_key($user_email, 'gender');
		return strcasecmp( $value, $gender ) === 0;
	}

	/**
	 * Checks if the Gender is not equal to the specified value (case-insensitive).
	 *
	 * @param string $value      The value to compare with the Gender.
	 * @param array  $automation_data The automation data containing the Gender.
	 *
	 * @return bool              True if the Gender is not equal to the value (case-insensitive), false otherwise.
	 * @since 1.17.0
	 */
	public function does_not_equal( string $value, array $automation_data ): bool {
		// Reuse the equal function by negating its result.
		return !$this->equal( $value, $automation_data );
	}

	/**
	 * Checks if the Gender is empty.
	 *
	 * @param array $automation_data The automation data containing the Gender.
	 *
	 * @return bool True if the Gender is empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_empty( string $value, array $automation_data): bool{
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return true;
		}

		$gender = ContactModel::get_contact_meta_value_by_key($user_email, 'gender');
		return empty($gender);
	}

	/**
	 * Checks if the Gender is not empty.
	 *
	 * @param array $automation_data The automation data containing the Gender.
	 *
	 * @return bool True if the Gender is not empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_not_empty( string $value, array $automation_data ): bool{
		return !$this->is_empty( $value, $automation_data );
	}
}
