<?php
/**
 * FirstName class for checking contact first name
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use Mint\MRM\DataBase\Tables\ContactSchema;
use MintMailPro\Mint_Pro_Helper;

/**
 * FirstName class for checking contact first name
 */
class FirstName {

	/**
	 * Checks if a given value is included in the first name.
	 *
	 * @param string $value The value to search for.
	 * @param array  $automation_data The automation data containing the first name.
	 *
	 * @return bool True if the value is included in the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function includes( string $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';
		$first_name = Mint_Pro_Helper::get_name_by_email( $user_email, 'first_name' );

		if ( empty( $first_name ) || empty( $value ) ) {
			return false;
		}

		if ( false !== stripos( $first_name, $value ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Checks if the specified value does not exist within the given automation data's first name.
	 *
	 * @param string $value           The value to check if it does not exist within the first name.
	 * @param array  $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the first name, false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_includes( string $value, array $automation_data ): bool {
		return !$this->includes( $value, $automation_data );
	}

	/**
	 * Checks if the first name matches the specified value.
	 *
	 * @param string $value           The value to compare with the first name.
	 * @param array  $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the first name matches the value, false otherwise.
	 * @since 1.2.7
	 */
	public function equal( string $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';
		$first_name = Mint_Pro_Helper::get_name_by_email( $user_email, 'first_name' );
		return strcasecmp( $value, $first_name ) === 0;
	}

	/**
	 * Checks if the first name is not equal to the specified value (case-insensitive).
	 *
	 * @param string $value      The value to compare with the first name.
	 * @param array  $automation_data The automation data containing the first name.
	 *
	 * @return bool              True if the first name is not equal to the value (case-insensitive), false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_equal( string $value, array $automation_data ): bool {
		// Reuse the equal function by negating its result.
		return !$this->equal( $value, $automation_data );
	}
}
