<?php

/**
 * Email Condition Class
 *
 * Provides various conditions to evaluate email strings for automation logic.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-12-04 09:03:17
 * @modify date 2024-12-10 09:03:17
 * @package MintMail\App\Internal\Automation\Condition
 */

namespace MintMail\App\Internal\Automation\Condition;

/**
 * Class Email
 *
 * Provides various conditions to evaluate email strings for automation logic.
 * Includes methods to check if an email contains, equals, starts with, or ends with a given value.
 * 
 * @since 1.17.1
 */
class Email {

	/**
	 * Check if the user's email includes a given value (case-insensitive).
	 *
	 * @param string $value The substring to check within the user's email.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the email includes the value, false otherwise.
	 * @since 1.17.1
	 */
	public function includes( string $value, array $automation_data ): bool {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';
		if ( empty( $user_email ) || empty( $value ) ) {
			return false;
		}

		if ( false !== stripos( $user_email, $value ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Check if the user's email does not include a given value (case-insensitive).
	 *
	 * @param string $value The substring to check within the user's email.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the email does not include the value, false otherwise.
	 * @since 1.17.1
	 */
	public function does_not_includes( string $value, array $automation_data ): bool {
		return !$this->includes( $value, $automation_data );
	}

	/**
	 * Check if the user's email equals a given value (case-insensitive).
	 *
	 * @param string $value The value to compare with the user's email.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the email equals the value, false otherwise.
	 * @since 1.17.1
	 */
	public function equal( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
		if (empty($user_email) || empty($value)) {
			return false;
		}

		return strcasecmp( $value, $user_email ) === 0;
	}

	/**
	 * Check if the user's email does not equal a given value (case-insensitive).
	 *
	 * @param string $value The value to compare with the user's email.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the email does not equal the value, false otherwise.
	 * @since 1.17.1
	 */
	public function does_not_equal( string $value, array $automation_data ): bool {
		// Reuse the equal function by negating its result.
		return !$this->equal( $value, $automation_data );
	}

	/**
	 * Check if the user's email starts with a given value.
	 *
	 * @param string $value The value to check as the prefix of the user's email.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the email starts with the value, false otherwise.
	 * @since 1.17.1
	 */
	public function starts_with(string $value, array $automation_data): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
		if (empty($user_email) || empty($value)) {
			return false;
		}

		return strpos($user_email, $value) === 0;
	}

	/**
	 * Check if the user's email ends with a given value.
	 *
	 * @param string $value The value to check as the suffix of the user's email.
	 * @param array $automation_data The automation data containing the user's email.
	 *
	 * @return bool Returns true if the email ends with the value, false otherwise.
	 * @since 1.17.1
	 */
	public function ends_with(string $value, array $automation_data): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
		if (empty($user_email) || empty($value)) {
			return false;
		}

		return substr($user_email, -strlen($value)) === $value;
	}
}
