<?php
/**
 * DateOfBirth Class for Checking contacts' birth dates.
 *
 * @package MintMail\App\Internal\Automation\Condition
 * @author MRM Team
 * @email support@getwpfunnels.com
 * @create date 2023-09-08 11:03:17
 * @modify date 2023-09-08 11:03:17
 */

namespace MintMail\App\Internal\Automation\Condition;

use DateTime;
use Mint\MRM\DataBase\Models\ContactModel;

/**
 * DateOfBirth  Class for Checking contacts birth date
 */
class DateOfBirth {

	/**
	 * Checks if the contact's date of birth is after a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's date of birth is after the specified date, false otherwise.
	 * @since 1.5.6
	 */
	public function after( $value, $automation_data ) {
		$dob = $this->get_contact_dob( $automation_data );
		return $this->compare_dob( $dob, $value, '>' );
	}

	/**
	 * Checks if the contact's date of birth is before a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's date of birth is before the specified date, false otherwise.
	 * @since 1.5.6
	 */
	public function before( $value, $automation_data ) {
		$dob = $this->get_contact_dob( $automation_data );
		return $this->compare_dob( $dob, $value, '<' );
	}

	/**
	 * Checks if the contact's date of birth matches a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's date of birth matches the specified date, false otherwise.
	 * @since 1.5.6
	 */
	public function in_the_date( $value, $automation_data ) {
		$dob = $this->get_contact_dob( $automation_data );
		return $this->compare_dob( $dob, $value, '=' );
	}

	/**
	 * Gets the contact's date of birth based on their email.
	 *
	 * @param array $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return string The contact's date of birth in 'Y-m-d' format.
	 * @since 1.5.6
	 */
	public function get_contact_dob( $automation_data ) {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';
		$contact    = ContactModel::get_contact_by_email( $user_email );
		$meta       = ContactModel::get_meta( $contact['id'] );
		return isset( $meta['meta_fields']['date_of_birth'] ) ? $meta['meta_fields']['date_of_birth'] : '';
	}


	/**
	 * Compares a date of birth with a specified date using a given comparison operator.
	 *
	 * @param string $dob              The date of birth in 'Y-m-d' format.
	 * @param string $value            The date to compare with (e.g., '2023-07-17').
	 * @param string $comparison_operator The comparison operator ('<', '>', '=').
	 *
	 * @access public
	 *
	 * @return bool True if the comparison is successful, false otherwise.
	 * @since 1.5.6
	 */
	public function compare_dob( $dob, $value, $comparison_operator ) {
		$formatted_date = ( new DateTime( $value ) )->format( 'Y-m-d' );
		$date_time      = new DateTime( $dob );
		$dob_date       = $date_time->format( 'Y-m-d' );
		return !empty( $dob_date ) && version_compare( $dob_date, $formatted_date, $comparison_operator );
	}
}
