<?php

/**
 * FirstName class for checking contact first name
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use DateTime;
use Mint\MRM\DataBase\Models\ContactModel;

class CustomField{

    /**
     * Checks if a given value is included in the first name.
     *
     * @param string $value The value to search for.
     * @param array  $automation_data The automation data containing the first name.
     *
     * @return bool True if the value is included in the first name, false otherwise.
     * @since 1.2.7
     */
    public function includes( $param, $value, $automation_data ): bool
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email( $user_email );
        // Check if the param exists in the contact array.
        if (isset($contact[$param])) {
            // If value is an array, check if any of its values is included in the contact param.
            if (is_array($value)) {
                // Convert $param to an array.
                $param_array = is_array($contact[$param]) ? $contact[$param] : array($contact[$param]);
                $values      = array_values(array_column($value, 'value'));
                $intersect   = array_intersect($values, $param_array);
                // If there is at least one common ID, return true.
                if (!empty($intersect)) {
                    return true;
                }
            } 
            // If value is a string, check if it's included in the contact param.
            else if (strpos($contact[$param], $value) !== false) {
                return true;
            }
        }

        return false;
    }

    /**
     * Checks if the specified value does not exist within the given automation data's first name.
     *
     * @param string $value           The value to check if it does not exist within the first name.
     * @param array  $automation_data The automation data containing the first name.
     *
     * @return bool                   True if the value does not exist within the first name, false otherwise.
     * @since 1.2.7
     */
    public function does_not_includes( $param, $value, $automation_data ): bool
    {
        return !$this->includes( $param, $value, $automation_data );
    }

    /**
     * Checks if the first name matches the specified value.
     *
     * @param string $value           The value to compare with the first name.
     * @param array  $automation_data The automation data containing the first name.
     *
     * @return bool                   True if the first name matches the value, false otherwise.
     * @since 1.2.7
     */
    public function equal( $param, $value, $automation_data ): bool
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email( $user_email );

        // Check if the param exists in the contact array and if its value equals to the provided value.
        if ( isset( $contact[ $param ] ) && $contact[ $param ] == $value ) {
            return true;
        }

        return false;
    }

    /**
     * Checks if the first name is not equal to the specified value (case-insensitive).
     *
     * @param string $value      The value to compare with the first name.
     * @param array  $automation_data The automation data containing the first name.
     *
     * @return bool              True if the first name is not equal to the value (case-insensitive), false otherwise.
     * @since 1.2.7
     */
    public function does_not_equal( $param, $value, $automation_data ): bool
    {
        // Use the equal function and negate its result
        return !$this->equal($param, $value, $automation_data);
    }

    /**
     * Checks if the first name is greater than the specified value.
     *
     * @param string $value      The value to compare with the first name.
     * @param array  $automation_data The automation data containing the first name.
     *
     * @return bool              True if the first name is not equal to the value (case-insensitive), false otherwise.
     * @since 1.2.7
     */
    public function greater_than( $param, $value, $automation_data ): bool
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email( $user_email );

        // Check if the param exists in the contact array and if its value is greater than the provided value.
        if ( isset( $contact[ $param ] ) && $contact[ $param ] > $value ) {
            return true;
        }

        return false;
    }

    /**
     * Checks if the first name is less than the specified value.
     *
     * @param string $value      The value to compare with the first name.
     * @param array  $automation_data The automation data containing the first name.
     *
     * @return bool              True if the first name is not equal to the value (case-insensitive), false otherwise.
     * @since 1.2.7
     */
    public function less_than( $param, $value, $automation_data ): bool
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email( $user_email );

        // Check if the param exists in the contact array and if its value is less than the provided value.
        if ( isset( $contact[ $param ] ) && $contact[ $param ] < $value ) {
            return true;
        }

        return false;
    }

    /**
     * Checks if all segment IDs from the given segments array are present in the group IDs of the automation data.
     *
     * @param array $segments           The value to check if it does not exist within the first name.
     * @param array $automation_data The automation data containing the first name.
     *
     * @return bool                   True if the first name matches the value, false otherwise.
     * @since 1.2.7
     */
    public function includes_all_of( $param, $value, $automation_data ): bool
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email($user_email);
        $values     = array_values(array_column($value, 'value'));

        // Convert $param to an array.
        $param_array = is_array($contact[$param]) ? $contact[$param] : array($contact[$param]);

        // Find the segment IDs that are not present in the group IDs.
        $diff = array_diff($values, $param_array);

        if (empty($diff)) {
            return true;
        }

        return false;
    }

    /**
     * Checks if none of the segment IDs from the given segments array are present in the group IDs of the automation data.
     *
     * @param array $segments           The value to check if it does not exist within the first name.
     * @param array $automation_data The automation data containing the first name.
     *
     * @return bool              True if the first name is not equal to the value (case-insensitive), false otherwise.
     * @since 1.2.7
     */
    public function includes_none_of( $param, $value, $automation_data ): bool
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email($user_email);
        $values     = array_values(array_column($value, 'value'));

        // Convert $param to an array.
        $param_array = is_array($contact[$param]) ? $contact[$param] : array($contact[$param]);

        // Find the segment IDs that are not present in the group IDs.
        $diff = array_diff($values, $param_array);

        if (count($diff) === count($param_array)) {
            return true;
        }
        return false;
    }

    /**
     * Checks if the contact's date of birth is after a specified date.
     *
     * @param string $value           The date to compare with (e.g., '2023-07-17').
     * @param array  $automation_data The automation data, including the user's email.
     *
     * @access public
     *
     * @return bool True if the contact's date of birth is after the specified date, false otherwise.
     * @since 1.5.6
     */
    public function after($param, $value, $automation_data)
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email($user_email);
        // Ensure $contact[$param] exists.
        if (!isset($contact[$param])) {
            return false;
        }

        // Convert $value and $contact[$param] to DateTime objects.
        $value_date_time   = new DateTime($value);
        $contact_date_time = new DateTime($contact[$param]);
        // Compare the two DateTime objects.
        return $contact_date_time > $value_date_time;
    }

    /**
     * Checks if the contact's date of birth is before a specified date.
     *
     * @param string $value           The date to compare with (e.g., '2023-07-17').
     * @param array  $automation_data The automation data, including the user's email.
     *
     * @access public
     *
     * @return bool True if the contact's date of birth is before the specified date, false otherwise.
     * @since 1.5.6
     */
    public function before($param, $value, $automation_data)
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email($user_email);
        // Ensure $contact[$param] exists.
        if (!isset($contact[$param])) {
            return false;
        }

        // Convert $value and $contact[$param] to DateTime objects.
        $value_date_time   = new DateTime($value);
        $contact_date_time = new DateTime($contact[$param]);
        // Compare the two DateTime objects.
        return $contact_date_time < $value_date_time;
    }

    /**
     * Checks if the contact's date of birth matches a specified date.
     *
     * @param string $value           The date to compare with (e.g., '2023-07-17').
     * @param array  $automation_data The automation data, including the user's email.
     *
     * @access public
     *
     * @return bool True if the contact's date of birth matches the specified date, false otherwise.
     * @since 1.5.6
     */
    public function in_the_date($param, $value, $automation_data)
    {
        $user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';
        $contact    = ContactModel::get_contact_data_by_email($user_email);
        // Ensure $contact[$param] exists.
        if (!isset($contact[$param])) {
            return false;
        }

        // Convert $value and $contact[$param] to DateTime objects.
        $value_date_time   = new DateTime($value);
        $contact_date_time = new DateTime($contact[$param]);
        // Compare the two DateTime objects.
        return $contact_date_time == $value_date_time;
    }
}
