<?php
/**
 * CreatedAt Class for Checking contacts' CreatedAt
 *
 * @package MintMail\App\Internal\Automation\Condition
 * @author MRM Team
 * @email support@getwpfunnels.com
 * @create date 2024-12-04 11:03:17
 * @modify date 2023-12-04 11:03:17
 */

namespace MintMail\App\Internal\Automation\Condition;

use DateTime;
use Mint\MRM\DataBase\Models\ContactModel;

/**
 * CreatedAt  Class for Checking contacts CreatedAt
 */
class CreatedAt {

	/**
	 * Checks if the contact's CreatedAt is after a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's CreatedAt is after the specified date, false otherwise.
	 * @since 1.17.0
	 */
	public function after( $value, $automation_data ) {
		$created_at = $this->get_contact_created( $automation_data );
		return $this->compare_date( $created_at, $value, '>' );
	}

	/**
	 * Checks if the contact's CreatedAt is before a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's CreatedAt is before the specified date, false otherwise.
	 * @since 1.17.0
	 */
	public function before( $value, $automation_data ) {
		$created_at = $this->get_contact_created( $automation_data );
		return $this->compare_date( $created_at, $value, '<' );
	}

	/**
	 * Checks if the contact's CreatedAt matches a specified date.
	 *
	 * @param string $value           The date to compare with (e.g., '2023-07-17').
	 * @param array  $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return bool True if the contact's CreatedAt matches the specified date, false otherwise.
	 * @since 1.17.0
	 */
	public function in_the_date( $value, $automation_data ) {

		$created_at = $this->get_contact_created( $automation_data );
		return $this->compare_date( $created_at, $value, '=' );
	}

	/**
	 * Gets the contact's CreatedAt based on their email.
	 *
	 * @param array $automation_data The automation data, including the user's email.
	 *
	 * @access public
	 *
	 * @return string The contact's CreatedAt in 'Y-m-d' format.
	 * @since 1.17.0
	 */

	 #get_contact_created
	public function get_contact_created( $automation_data ) {
		$user_email = isset( $automation_data['data']['user_email'] ) ? $automation_data['data']['user_email'] : '';
		$contact    = ContactModel::get_contact_by_email( $user_email );
		return isset( $contact['created_at'] ) ? $contact['created_at'] : '';
	}


	/**
	 * Compares a CreatedAt with a specified date using a given comparison operator.
	 *
	 * @param string $created_at       The CreatedAt in 'Y-m-d' format.
	 * @param string $value            The date to compare with (e.g., '2023-07-17').
	 * @param string $comparison_operator The comparison operator ('<', '>', '=').
	 *
	 * @access public
	 *
	 * @return bool True if the comparison is successful, false otherwise.
	 * @since 1.17.0
	 */
	public function compare_date( $created_at, $value, $comparison_operator ) {
		$formatted_created_at = (new DateTime($created_at))->format('Y-m-d');
		$formatted_value      = (new DateTime($value))->format('Y-m-d');
		return !empty($formatted_created_at) && version_compare($formatted_created_at, $formatted_value, $comparison_operator);
	}
}
