<?php
/**
 * City class for checking contact City
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2024-12-05 8:23:17
 * @modify date 2024-12-05 8:23:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use Mint\MRM\DataBase\Models\ContactModel;


/**
 * City class for checking contact City
 */
class City {

	/**
	 * Checks if a given value is included in the City
	 *
	 * @param string $value The value to search for.
	 * @param array  $automation_data The automation data containing the City
	 *
	 * @return bool True if the value is included in the City, false otherwise.
	 * @since 1.2.7
	 */
	public function includes( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$city = ContactModel::get_contact_meta_value_by_key($user_email, 'city');

		if (false !== stripos($city, $value)) {
			return true;
		}
		return false;
	}

	/**
	 * Checks if the specified value does not exist within the given automation data's first name.
	 *
	 * @param string $value           The value to check if it does not exist within the first name.
	 * @param array  $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the City, false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_includes( string $value, array $automation_data ): bool {
		return !$this->includes( $value, $automation_data );
	}

	/**
	 * Checks if the City matches the specified value.
	 *
	 * @param string $value           The value to compare with the City.
	 * @param array  $automation_data The automation data containing the City.
	 *
	 * @return bool                   True if the City matches the value, false otherwise.
	 * @since 1.2.7
	 */
	public function equal( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$city = ContactModel::get_contact_meta_value_by_key($user_email, 'city');
		return strcasecmp($value, $city) === 0;
	}

	/**
	 * Checks if the Cityis not equal to the specified value (case-insensitive).
	 *
	 * @param string $value      The value to compare with the City.
	 * @param array  $automation_data The automation data containing the City.
	 *
	 * @return bool              True if the City is not equal to the value (case-insensitive), false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_equal( string $value, array $automation_data ): bool {
		return !$this->equal( $value, $automation_data );
	}

	/**
	 * Checks if the city is empty.
	 *
	 * @param array $automation_data The automation data containing the city.
	 *
	 * @return bool True if the city is empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_empty(string $value, array $automation_data): bool
	{
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return true;
		}

		$city = ContactModel::get_contact_meta_value_by_key($user_email, 'city');
		return empty($city);
	}

	/**
	 * Checks if the city is not empty.
	 *
	 * @param array $automation_data The automation data containing the city.
	 *
	 * @return bool True if the city is not empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_not_empty(string $value, array $automation_data): bool
	{
		return !$this->is_empty($value, $automation_data);
	}
}
