<?php
/**
 * AddressLine1 class for checking contact AddressLine1
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2023-05-15 11:03:17
 * @modify date 2023-05-15 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use Mint\MRM\DataBase\Tables\ContactSchema;
use MintMailPro\Mint_Pro_Helper;
use Mint\MRM\DataBase\Models\ContactModel;


/**
 * AddressLine1 class for checking contact AddressLine1
 */
class AddressLine2 {

	/**
	 * Checks if a given value is included in the AddressLine1
	 *
	 * @param string $value The value to search for.
	 * @param array  $automation_data The automation data containing the AddressLine1
	 *
	 * @return bool True if the value is included in the AddressLine1, false otherwise.
	 * @since 1.2.7
	 */
	public function includes( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$address_line = ContactModel::get_contact_meta_value_by_key($user_email, 'address_line_2');

		if (false !== stripos($address_line, $value)) {
			return true;
		}
		return false;
	}

	/**
	 * Checks if the specified value does not exist within the given automation data's first name.
	 *
	 * @param string $value           The value to check if it does not exist within the first name.
	 * @param array  $automation_data The automation data containing the first name.
	 *
	 * @return bool                   True if the value does not exist within the AddressLine1, false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_includes( string $value, array $automation_data ): bool {
		return !$this->includes( $value, $automation_data );
	}

	/**
	 * Checks if the AddressLine1 matches the specified value.
	 *
	 * @param string $value           The value to compare with the AddressLine1.
	 * @param array  $automation_data The automation data containing the AddressLine1.
	 *
	 * @return bool                   True if the AddressLine1 matches the value, false otherwise.
	 * @since 1.2.7
	 */
	public function equal( string $value, array $automation_data ): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return false;
		}

		$address_line = ContactModel::get_contact_meta_value_by_key($user_email, 'address_line_2');
		return strcasecmp($value, $address_line) === 0;
	}

	/**
	 * Checks if the AddressLine1is not equal to the specified value (case-insensitive).
	 *
	 * @param string $value      The value to compare with the AddressLine1.
	 * @param array  $automation_data The automation data containing the AddressLine1.
	 *
	 * @return bool              True if the AddressLine1 is not equal to the value (case-insensitive), false otherwise.
	 * @since 1.2.7
	 */
	public function does_not_equal( string $value, array $automation_data ): bool {
		return !$this->equal( $value, $automation_data );
	}

	/**
	 * Checks if the address line 2 is empty.
	 *
	 * @param array $automation_data The automation data containing the address line 2.
	 *
	 * @return bool True if the address line 1 is empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_empty(string $value, array $automation_data): bool {
		$user_email = isset($automation_data['data']['user_email']) ? $automation_data['data']['user_email'] : '';

		if (empty($user_email) || empty($value)) {
			return true;
		}

		$address_line = ContactModel::get_contact_meta_value_by_key($user_email, 'address_line_2');
		return empty($address_line);
	}

	/**
	 * Checks if the address line 2 is not empty.
	 *
	 * @param array $automation_data The automation data containing the address line 2.
	 *
	 * @return bool True if the address line 1 is not empty, false otherwise.
	 * @since 1.17.0
	 */
	public function is_not_empty(string $value, array $automation_data): bool {
		return !$this->is_empty($value, $automation_data);
	}
}
