<?php
/**
 * Condition  Class for Checking Automation condition.
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2022-08-09 11:03:17
 * @modify date 2022-08-09 11:03:17
 * @package /app
 */

namespace MintMail\App\Internal\Automation\Condition;

use Mint\Mrm\Internal\Traits\Singleton;

use MintMail\App\Internal\Automation\ActionScheduler;
use MRM\Common\MrmCommon;
/**
 * Condition  Class for Checking Automation condition.
 */
class ConditionMap {

	use Singleton;

	/**
	 * Action scheduler.
	 *
	 * @var $action_scheduler
	 */
	private $action_scheduler;

	/**
	 * Initialization
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Map the condition settings and perform corresponding queries based on the conditions.
	 *
	 * This method iterates through the given condition settings and performs MySQL queries based on the specified conditions and values.
	 * The result of each query is stored in the corresponding element of the condition settings array.
	 *
	 * @param array $condition_settings The condition settings to be mapped and processed.
	 * @param array $automation_data The automation data to be used in the queries.
	 * @return array The mapped condition settings with query results.
	 *
	 * @since 1.2.5
	 */
	public function map( $condition_settings, $automation_data ) {
		foreach ( $condition_settings as $outer_key => $outer ) {
			foreach ( $outer as $inner_key => $inner ) {
				$condition = isset( $inner['condition_value'] ) ? $inner['condition_value'] : '';
				$value     = isset( $inner['value'] ) ? $inner['value'] : '';
				$segment   = isset( $inner['segmentValue'] ) ? $inner['segmentValue'] : '';
				$param     = isset( $inner['param'] ) ? $inner['param'] : '';

				// Perform a MySQL query based on the condition and value.
				$result = false;
				switch ( $condition ) {
					case 'after':
						// perform query for after condition.
						$result = $this->perform_after_query( $param, $value, $condition, $automation_data );
						break;
					case 'before':
						// perform query for before condition.
						$result = $this->perform_before_query( $param, $value, $condition, $automation_data );
						break;
					case 'in_the_date':
						// perform query for certain time condition.
						$result = $this->perform_in_the_date_query( $param, $value, $condition, $automation_data );
						break;
					case 'includes':
						// perform query for includes condition.
						$result = $this->perform_includes_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'does_not_includes':
						// perform query for includes condition.
						$result = $this->perform_does_not_includes_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'equal':
						// perform query for includes condition.
						$result = $this->perform_equal_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'does_not_equal':
						// perform query for includes condition.
						$result = $this->perform_does_not_equal_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'includes_all_of':
						// perform query for includes condition.
						$result = $this->perform_includes_all_of_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'includes_none_of':
						// perform query for includes condition.
						$result = $this->perform_includes_none_of_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'starts_with':
						// perform query for starts with condition.
						$result = $this->perform_starts_with_query( $param, $value, $condition, $automation_data );
						break;
					case 'is_empty':
						// perform query for is empty condition.
						$result = $this->perform_is_empty_query( $param, $value, $condition, $automation_data );
						break;
					case 'is_not_empty':
						// perform query for is not empty condition.
						$result = $this->perform_is_not_empty_query( $param, $value, $condition, $automation_data );
						break;
					case 'included_in':
						// perform query for included in condition.
						$result = $this->perform_includes_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'not_included_in':
						// perform query for not included in condition.
						$result = $this->perform_does_not_includes_query( $param, $value, $condition, $automation_data, $segment );
						break;
					case 'greater_than':
						// perform query for greater than condition.
						$result = $this->perform_greater_than_query( $param, $value, $condition, $automation_data );
						break;
					case 'less_than':
						// perform query for less than condition.
						$result = $this->perform_less_than_query( $param, $value, $condition, $automation_data );
						break;
					case 'yes':
					case 'no':
						// perform query for EDD customer condition.
						$result = $this->perform_boolean_query( $param, $value, $condition, $automation_data, array() );
						break;
					default:
						// Handle default condition.
						break;
				}


				/**
				 * Apply a filter to modify the result of the automation condition.
				 *
				 * @param mixed $result The current result of the automation condition.
				 * @param mixed $inner The inner value used in the automation condition.
				 * @param array $automation_data Additional data related to the automation.
				 * 
				 * @return bool The modified result of the automation condition.
				 * @since  1.15.4
				 */
				$result = apply_filters('mint_automation_condition_result', $result, $inner, $automation_data );
				// Replace the current element with the boolean result of the query.
				$data[ $outer_key ][ $inner_key ] = $result;
			}
		}
		return $data;
	}

	/**
	 * Determine if any of the inner arrays meet a certain condition.
	 *
	 * @param mixed $arr An array of arrays, where each inner array contains boolean values.
	 *
	 * @return bool Returns true if any of the inner arrays contain only true values, and false otherwise.
	 * @since 1.1.2
	 */
	public function get_conditional_node_result( $arr ) {
		foreach ( $arr as $inner_arr ) {
			$inner_result = true;
			foreach ( $inner_arr as $value ) {
				if ( !$value ) {
					$inner_result = false;
					break;
				}
			}
			if ( $inner_result ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Perform an action after querying the specified condition.
	 *
	 * @param mixed $param The name of the parameter to be queried.
	 * @param mixed $value The value of the parameter to be queried.
	 * @param mixed $condition The name of the condition to be queried.
	 * @param mixed $automation_data The automation data to be passed to the condition method.
	 *
	 * @return mixed The result of the condition method, or null if the class does not exist.
	 * @since 1.1.2
	 */
	private function perform_after_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Perform an action before querying the specified condition.
	 *
	 * @param mixed $param The name of the parameter to be queried.
	 * @param mixed $value The value of the parameter to be queried.
	 * @param mixed $condition The name of the condition to be queried.
	 * @param mixed $automation_data The automation data to be passed to the condition method.
	 *
	 * @return mixed The result of the condition method, or null if the class does not exist.
	 * @since 1.1.2
	 */
	private function perform_before_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Perform an action in the date querying the specified condition.
	 *
	 * @param mixed $param The name of the parameter to be queried.
	 * @param mixed $value The value of the parameter to be queried.
	 * @param mixed $condition The name of the condition to be queried.
	 * @param mixed $automation_data The automation data to be passed to the condition method.
	 *
	 * @return mixed The result of the condition method, or null if the class does not exist.
	 * @since 1.1.2
	 */
	private function perform_in_the_date_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Performs an includes query using the specified condition class.
	 *
	 * @param string $param           The parameter to query.
	 * @param mixed  $value           The value to match.
	 * @param string $condition       The condition to apply.
	 * @param array  $automation_data The automation data.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return mixed The result of the includes query.
	 * @since 1.2.7
	 */
	private function perform_includes_query( $param, $value, $condition, $automation_data, $segment ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, $segment );
	}

	/**
	 * Perform the "does not include" query based on the specified condition, value, and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return bool The result of the "does not include" query.
	 * @since 1.2.7
	 */
	private function perform_does_not_includes_query( $param, $value, $condition, $automation_data, $segment ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, $segment );
	}

	/**
	 * Performs an equal query using the specified condition and value.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return bool The result of the equal query.
	 * @since 1.2.7
	 */
	private function perform_equal_query( $param, $value, $condition, $automation_data, $segment ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, $segment );
	}

	/**
	 * Performs an does not equal query using the specified condition and value.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return bool The result of the equal query.
	 * @since 1.2.7
	 */
	private function perform_does_not_equal_query( $param, $value, $condition, $automation_data, $segment ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, $segment );
	}

	/**
	 * Performs the "includes_all_of" query for the given parameters and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return bool The result of the equal query.
	 * @since 1.2.7
	 */
	private function perform_includes_all_of_query( $param, $value, $condition, $automation_data, $segment ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, $segment );
	}

	/**
	 * Performs the "includes_none_of" query for the given parameters and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return bool The result of the equal query.
	 * @since 1.2.7
	 */
	private function perform_includes_none_of_query( $param, $value, $condition, $automation_data, $segment ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, $segment );
	}

	/**
	 * Perform a "starts with" query based on the specified condition, value, and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 *
	 * @return mixed The result of the "starts with" query.
	 * @since 1.7.1
	 */
	private function perform_starts_with_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Perform an "is empty" query based on the specified condition, value, and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 *
	 * @return mixed The result of the "is empty" query.
	 * @since 1.7.1
	 */
	private function perform_is_empty_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Perform an "is not empty" query based on the specified condition, value, and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 *
	 * @return mixed The result of the "is not empty" query.
	 * @since 1.7.1
	 */
	private function perform_is_not_empty_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Perform a "greater than" query based on the specified condition, value, and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 *
	 * @return mixed The result of the "greater than" query.
	 * @since 1.7.1
	 */
	private function perform_greater_than_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Perform a "less than" query based on the specified condition, value, and automation data.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 *
	 * @return mixed The result of the "less than" query.
	 * @since 1.7.1
	 */
	private function perform_less_than_query( $param, $value, $condition, $automation_data ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, array() );
	}

	/**
	 * Perform a boolean query based on the specified condition and value.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return mixed The result of the query, or null if the class does not exist.
	 * @since 1.2.7
	 */
	private function perform_boolean_query( $param, $value, $condition, $automation_data, $segment ) {
		return $this->perform_query( $param, $value, $condition, $automation_data, $segment );
	}

	/**
	 * Perform a query based on the specified condition and value.
	 *
	 * @param string $param          The parameter for the condition.
	 * @param mixed  $value           The value to check against.
	 * @param string $condition      The condition to be performed.
	 * @param array  $automation_data The automation data to be used in the query.
	 * @param array  $segment The segment value (applicable only for 'list' or 'tag' conditions).
	 *
	 * @return mixed The result of the query, or null if the class does not exist.
	 * @since 1.2.7
	 */
	private function perform_query( $param, $value, $condition, $automation_data, $segment ) {
		$class_name = MrmCommon::convert_to_camelcase( $param );
		if ( 'list' === $param || 'tag' === $param || 'country' === $param || 'status' === $param || 'wp_user_role' === $param ) {
			$class_name .= 'Segment';
			$class_name  = 'MintMail\\App\\Internal\\Automation\\Condition\\' . $class_name;

			if ( class_exists( $class_name ) ) {
				$class_object = new $class_name();
				return $class_object->$condition( $segment, $automation_data );
			}
		} else {

			$class_name = 'MintMail\\App\\Internal\\Automation\\Condition\\' . $class_name;
			if ( class_exists( $class_name ) ) {
				$class_object = new $class_name();
				return $class_object->$condition( $value, $automation_data );
			} else {
				$object = new CustomField();
				return $object->$condition( $param, $value, $automation_data );
			}
		}

		return false;
	}
}
