<?php
/**
 * Automation action class for MRM Autoamtion
 *
 * Class Delay
 *
 * @package MintMail\App\Internal\Automation\Action
 */

namespace MintMail\App\Internal\Automation\Action;

use DateTime;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;
/**
 * Delay
 *
 * Class Delay
 *
 * @package MintMail\App\Internal\Automation\Action
 */
class Condition extends AbstractAutomationAction {

	use Singleton;

	/**
	 * Action scheduler.
	 *
	 * @object $action_scheduler
	 */
	private $action_scheduler;

	/**
	 * Initialization
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Get action name
	 *
	 * @return String
	 * @since  1.0.0
	 */
	public function get_name() {
		return __( 'Condition', 'mailmint-pro' );
	}

	/**
	 * Process data for an automation system.
	 *
	 * @param array $data An array of data to be processed.
	 * @return void
	 * @since 1.1.2
	 */
	public function process( $data ) {
		if ( $data ) {
			// Create a payload for logging purposes.
			$log_payload = array(
				'automation_id' => $data['automation_id'],
				'step_id'       => $data['step_id'],
				'status'        => 'hold',
				'identifier'    => $data['identifier'],
				'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
			);
			// Log the payload.
			HelperFunctions::update_log( $log_payload );
			$step_id = $data['step_id'];
			// Get data for current step of automation.
			$step_data = HelperFunctions::get_step_data( $data['automation_id'], $data['step_id'] );
			// Extract condition settings from step data.
			$condition_settings = isset( $step_data['settings']['rules']['condition'] ) ? $step_data['settings']['rules']['condition'] : array();
			// The string representation of the date and time to be converted to a DateTime object.
			$automation_meta = HelperFunctions::get_automation_meta( $data['automation_id'], '_at_most_date' );
			$date_string     = HelperFunctions::get_date_from_conditional_step_array( $automation_meta, $data['step_id'] );
			$date_string     = !empty( $date_string ) ? $date_string : '';
			$date_time       = DateTime::createFromFormat( 'n/j/Y, g:i:s A', $date_string );

			// Get the difference in seconds between the two date and time values.
			$date_time_string = $date_time instanceof DateTime ? $date_time->format( 'Y-m-d H:i:s' ) : '';
			$current_datetime = current_datetime()->format('Y-m-d H:i:s'); //phpcs:ignore
			$time             = HelperFunctions::get_time_diff_in_seconds( $current_datetime, $date_time_string );

			if ( !$this->validate_delay( $time ) ) {
				$time = 0;
			}

			$email_opened  = isset( $data['email_opened'] ) ? $data['email_opened'] : '';
			$email_clicked = isset( $data['email_clicked'] ) ? $data['email_clicked'] : '';

			// Check if the conditions for this step are met.
			$maybe_matched = true;
			if ( !empty( $condition_settings ) && is_array( $condition_settings ) ) {
				$class_name = 'MintMail\\App\\Internal\\Automation\\Condition\\ConditionMap';
				if ( class_exists( $class_name ) ) {
					$condition_map_arr = $class_name::get_instance()->map( $condition_settings, $data );
					$is_matched        = $class_name::get_instance()->get_conditional_node_result( $condition_map_arr );
					if ( !$is_matched ) {
						$maybe_matched = false;
					}
				}
			}
			if ( $date_time_string < $current_datetime ) {
				$this->run_next_step_based_on_condition( $data, $maybe_matched, $step_id );
			} elseif ( ( 'yes' === $email_opened || 'yes' === $email_clicked ) && $maybe_matched ) {
				$this->run_next_step_based_on_condition( $data, $maybe_matched, $step_id );
			} else {
				$this->schedule_step_execution( $data, $time );
			}
		}
	}

	/**
	 * Validates if the delay time in seconds is valid.
	 *
	 * @param int $seconds The delay time in seconds.
	 *
	 * @return bool True if the delay time is valid, false otherwise.
	 * @since 1.2.7
	 */
	private function validate_delay( $seconds ) {
		if ( $seconds <= 0 ) {
			return false;
		}
		if ( $seconds > 2 * YEAR_IN_SECONDS ) {
			return false;
		}
		return true;
	}


	/**
	 * Runs the next step after the condition step and updates the logs and job status.
	 *
	 * @param array $data The step data array.
	 * @param bool  $maybe_matched The boolean value of whether the condition was matched or not.
	 * @param int   $step_id The ID of the current step.
	 * @return void
	 */
	public function run_next_step_based_on_condition( $data, $maybe_matched, $step_id ) {
		$next_step = HelperFunctions::get_next_step_after_logical_step( $data['automation_id'], $maybe_matched, $data['step_id'] );

		if ( empty( $next_step ) ) {
			$next_step = HelperFunctions::get_next_step( $data['automation_id'], $data['step_id'] );
		}
		$payload = array(
			'automation_id' => $data['automation_id'],
			'step_id'       => $step_id,
			'status'        => 'completed',
			'identifier'    => $data['identifier'],
			'email'         => !empty( $data['data']['user_email'] ) ? $data['data']['user_email'] : '',
		);
		HelperFunctions::update_log( $payload );

		HelperFunctions::update_job( $data['automation_id'], isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
		if ( $next_step ) {
			$next_step['data']       = $data['data'];
			$next_step['identifier'] = $data['identifier'];
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
	}

	/**
	 * Schedule hold condition step to process after specified time
	 *
	 * @param array $data The data associated with the hold condition step.
	 * @param int   $time The time after which the hold condition step should be processed.
	 * @return void
	 */
	public function schedule_step_execution( $data, $time ) {
		$step_data                = HelperFunctions::get_step_data( $data['automation_id'], $data['step_id'] );
		$payload                  = array(
			array(
				'automation_id' => $step_data['automation_id'],
				'step_id'       => $step_data['step_id'],
				'step_type'     => $step_data['step_type'],
				'key'           => $step_data['key'],
			),
		);
		$payload[0]['data']       = isset( $data['data'] ) ? $data['data'] : array();
		$payload[0]['identifier'] = $data['identifier'];
		$this->action_scheduler->schedule( time() + $time, MINT_PROCESS_AUTOMATION, $payload );
	}

}
