<?php
/**
 * WooCommerce order status change action class for Mail Mint Automation.
 *
 * Class ChangeOrderStatus
 *
 * @package MintMail\App\Internal\Automation\Action
 * @namespace MintMail\App\Internal\Automation\Action
 * @author [getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-05-21 11:03:17
 * @modify date 2024-05-21 11:03:17
 */

namespace MintMail\App\Internal\Automation\Action;

use Mint\MRM\DataBase\Models\ContactModel;
use Mint\Mrm\Internal\Traits\Singleton;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;

/**
 * ChangeOrderStatus action
 *
 * Class ChangeOrderStatus
 *
 * @package MintMail\App\Internal\Automation\Action
 * @since 1.12.0
 */
class ChangeOrderStatus extends AbstractAutomationAction {

	use Singleton;

	/**
	 * Holds an instance of the ActionScheduler class.
	 *
	 * @var ActionScheduler
	 *
	 * @access private
	 * @since 1.12.0
	 */
	private $action_scheduler;

	/**
	 * Class constructor.
	 *
	 * @access public
	 * @since 1.12.0
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Get the name of the action.
	 *
	 * Description: Retrieves the localized name for the "Change Order Status" action.
	 *
	 * @access public
	 * @return string The localized name of the action.
	 * @since 1.12.0
	 */
	public function get_name() {
		return 'Change Order Status';
	}

	/**
	 * Process automation step for changing the order status.
	 *
	 * Description: Processes the automation step for changing the order status.
	 *
	 * @access public
	 * @param array $data The data for the automation step.
	 * @since 1.8.0
	 */
	public function process( $data ) {
		// Check if essential data is provided.
		if ( ! isset( $data['automation_id'] ) || ! isset( $data['step_id'] ) ) {
			return;
		}

		// Extract necessary data from the input.
		$order_id = isset( $data['data']['order_id'] ) ? $data['data']['order_id'] : '';
		if ( empty( $order_id ) ) {
			return;
		}

		$user_email    = isset( $data['data']['user_email'] ) ? $data['data']['user_email'] : '';
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$identifier    = isset( $data['identifier'] ) ? $data['identifier'] : '';

		// Log that the step is in a 'hold' status.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'hold',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// Retrieve settings data for the current step.
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );

		// Check if coupon settings are available.
		if ( ! isset( $step_data['settings']['status_settings'] ) || empty( $step_data['settings']['status_settings'] ) ) {
			return;
		}

		$new_status = isset( $step_data['settings']['status_settings']['status'] ) ? $step_data['settings']['status_settings']['status'] : '';

		if( !$new_status ) {
			// Log that the step is fail.
			$log_payload = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'status'        => 'fail',
				'identifier'    => $identifier,
				'email'         => $user_email,
			);
			HelperFunctions::update_log( $log_payload );
		} else {
			$order = wc_get_order( $order_id );
			$order->update_status( $new_status, 'Order status has been changed to <b>'.$new_status.'</b> by Mail Mint Automation ID: '.$automation_id );
			// Log that the step is completed.
			$log_payload = array(
				'automation_id' => $automation_id,
				'step_id'       => $step_id,
				'status'        => 'completed',
				'identifier'    => $identifier,
				'email'         => $user_email,
			);
			HelperFunctions::update_log( $log_payload );
		}

		// If there's a next step, enqueue it for processing.
		$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
		HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
		if ( $next_step ) {
			$next_step['data']       = $data['data'];
			$next_step['identifier'] = $data['identifier'];
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
	}
}
