<?php
/**
 * Add note to WooCommerce order action class for Mail Mint Automation.
 *
 * Class AddOrderNote
 *
 * @package MintMail\App\Internal\Automation\Action
 * @namespace MintMail\App\Internal\Automation\Action
 * @author [getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-05-23 11:03:17
 * @modify date 2024-05-23 11:03:17
 */

namespace MintMail\App\Internal\Automation\Action;

use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\Internal\Parser\Parser;
use Mint\Mrm\Internal\Traits\Singleton;
use Mint\MRM\Utilites\Helper\Email;
use MintMail\App\Internal\Automation\Action\AbstractAutomationAction;
use MintMail\App\Internal\Automation\HelperFunctions;
use MintMail\App\Internal\Automation\ActionScheduler;
use MRM\Common\MrmCommon;

/**
 * AddOrderNote action
 *
 * Class AddOrderNote
 *
 * @package MintMail\App\Internal\Automation\Action
 * @since 1.12.0
 */
class AddOrderNote extends AbstractAutomationAction {

	use Singleton;

	/**
	 * Holds an instance of the ActionScheduler class.
	 *
	 * @var ActionScheduler
	 *
	 * @access private
	 * @since 1.12.0
	 */
	private $action_scheduler;

	/**
	 * Class constructor.
	 *
	 * @access public
	 * @since 1.12.0
	 */
	public function __construct() {
		$this->action_scheduler = new ActionScheduler();
	}

	/**
	 * Get the name of the action.
	 *
	 * Description: Retrieves the localized name for the "Add Order Note" action.
	 *
	 * @access public
	 * @return string The localized name of the action.
	 * @since 1.12.0
	 */
	public function get_name() {
		return 'Add Order Note';
	}

	/**
	 * Process automation step for changing the order status.
	 *
	 * Description: Processes the automation step for changing the order status.
	 *
	 * @access public
	 * @param array $data The data for the automation step.
	 * @since 1.8.0
	 */
	public function process( $data ) {
		// Check if essential data is provided.
		if ( ! isset( $data['automation_id'] ) || ! isset( $data['step_id'] ) ) {
			return;
		}
		$user_email   = !empty($data['data']['user_email']) ? $data['data']['user_email'] : '';
		$post_id      = isset($data['data']['post_id']) ? $data['data']['post_id'] : '';
		$abandoned_id = isset($data['data']['abandoned_id']) ? $data['data']['abandoned_id'] : '';
		$payment_id   = isset($data['data']['payment_id']) ? $data['data']['payment_id'] : '';
		$wishlist_id  = isset($data['data']['wishlist_id']) ? $data['data']['wishlist_id'] : '';
		$product_id   = isset($data['data']['product_id']) ? $data['data']['product_id'] : '';

		// Extract necessary data from the input.
		$order_id = isset( $data['data']['order_id'] ) ? $data['data']['order_id'] : '';
		if ( empty( $order_id ) ) {
			return;
		}

		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$step_id       = isset( $data['step_id'] ) ? $data['step_id'] : '';
		$identifier    = isset( $data['identifier'] ) ? $data['identifier'] : '';
		$user_membership_id = isset($data['data']['user_membership_id']) ? $data['data']['user_membership_id'] : '';

		// Log that the step is in a 'hold' status.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'hold',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// Retrieve settings data for the current step.
		$step_data = HelperFunctions::get_step_data( $automation_id, $step_id );

		// Check if coupon settings are available.
		if ( ! isset( $step_data['settings']['order_note'] ) || empty( $step_data['settings']['order_note'] ) ) {
			return;
		}

		$type = isset( $step_data['settings']['order_note']['type'] ) ? $step_data['settings']['order_note']['type'] : '';
		$body = isset( $step_data['settings']['order_note']['body'] ) ? $step_data['settings']['order_note']['body'] : '';
		if( empty( $body ) ) {
			return;
		}

		// Get contact and merge meta fields with contact fields.
		$contact    = ContactModel::get_contact_by_email( $user_email );
		$contact_id = isset( $contact['id'] ) ? $contact['id'] : '';
		$contact    = ContactModel::get( $contact_id );
		
		if (isset($contact['meta_fields']) && is_array($contact['meta_fields'])) {
			$contact = array_merge($contact, $contact['meta_fields']);
			unset($contact['meta_fields']);
		}

		$body        = Parser::parse($body, $contact, $post_id, $order_id, array('abandoned_id' => $abandoned_id, 'edd_payment_id' => $payment_id, 'product_id' => $product_id, 'user_membership_id' => $user_membership_id, 'wishlist_id' => $wishlist_id));
		$is_customer = $type === 'customer' ? true : false;

		$order = wc_get_order( $order_id );
		$order->add_order_note( $body, $is_customer );
		// Log that the step is completed.
		$log_payload = array(
			'automation_id' => $automation_id,
			'step_id'       => $step_id,
			'status'        => 'completed',
			'identifier'    => $identifier,
			'email'         => $user_email,
		);
		HelperFunctions::update_log( $log_payload );

		// If there's a next step, enqueue it for processing.
		$next_step = HelperFunctions::get_next_step( $automation_id, $step_id );
		HelperFunctions::update_job( $automation_id, isset( $next_step['step_id'] ) ? $next_step['step_id'] : null, isset( $next_step['step_id'] ) ? 'processing' : 'completed' );
		if ( $next_step ) {
			$next_step['data']       = $data['data'];
			$next_step['identifier'] = $data['identifier'];
			do_action(MINT_PROCESS_AUTOMATION, $next_step);
		}
	}
}
