<?php
/**
 * Mail Mint
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2022-08-09 11:03:17
 * @modify date 2022-08-09 11:03:17
 * @package /app/Internal/Admin/Segmentation/
 */

namespace MailMintPro\Mint\Internal\Admin\Segmentation;

use Mint\MRM\DataBase\Models\ContactGroupModel;
use Mint\MRM\DataBase\Tables\ContactGroupPivotSchema;
use Mint\MRM\DataBase\Tables\ContactGroupSchema;
use Mint\MRM\DataBase\Tables\ContactMetaSchema;
use Mint\MRM\DataBase\Tables\ContactSchema;
use MRM\Common\MrmCommon;

/**
 * [Helper class to filter segment contacts]
 *
 * @package /app/Internal/Admin/Segmentation/
 * @since 1.0.0
 */
class FilterSegmentContacts {

	/**
	 * Contact primary field
	 *
	 * @var array
	 *
	 * @since 1.0.0
	 */
	private static $contact_primary_field = array(
		'email',
		'first_name',
		'last_name',
		'status',
		'list',
		'tag',
	);

	/**
	 * Get filtered contacts by segment id
	 *
	 * @param string|int $segment_id Segment ID.
	 * @param string     $columns Column name separated with comma.
	 *
	 * @return array|object|\stdClass[]|null
	 * @since 1.0.0
	 */
	public static function get_contacts( $segment_id, $columns = '', $offset = 0, $per_batch = 0 ) {
		$filter_options = self::get_filters( $segment_id );
		$filter_options = self::get_prepared_data( $filter_options );
		$where_query    = self::get_filter_where_query( $filter_options );

		return self::filter_contacts( $where_query, $offset, $per_batch, 'email', 'ASC', false, $columns );
	}

	/**
	 * Perform query to fetch filtered contacts
	 *
	 * @param string $where_query Where mysql query.
	 *
	 * @return array|object|\stdClass[]|null|int
	 * @since 1.0.0
	 */
	public static function filter_contacts( string $where_query, $offset = 0, $per_page = 0, $order_by = 'email', $order_type = 'ASC', $count_only = false, $columns = '' ) {
		global $wpdb;
		$contacts_table_name = $wpdb->prefix . ContactSchema::$table_name;
		$select_fields       = $count_only ? 'COUNT(id)' : '*';
		$select_fields       = $columns !== '' ? $columns : $select_fields;
		$wpdb_get_function   = $count_only ? 'get_var' : 'get_results';
		$wpdb_return_type    = $count_only ? 0 : ARRAY_A;

		$query  = $wpdb->prepare( 'SELECT %1s FROM %1s AS c1 ', $select_fields, $contacts_table_name );
		$query .= 'WHERE ' . $where_query;
		$query .= $wpdb->prepare( ' ORDER BY %1s %1s', $order_by, $order_type );
		$query .= $per_page ? $wpdb->prepare( ' LIMIT %d, %d', (int) $offset, (int) $per_page ) : '';
		$results = array(
			'data'        => array(),
			'total_pages' => 0,
			'total_count' => 0,
		);

		$count_query  = $wpdb->prepare( 'SELECT COUNT(id) FROM %1s AS c1 ', $contacts_table_name );
		$count_query .= 'WHERE ' . $where_query;

		try {
			if ( $count_only ) {
				return $wpdb->$wpdb_get_function( $query, $wpdb_return_type );
			}
			$results[ 'data' ]        = $wpdb->$wpdb_get_function( $query, $wpdb_return_type );
			$total_count              = $wpdb->get_var( $count_query );
			$results[ 'total_pages' ] = $per_page > 0 ? ceil( $total_count / $per_page ) : 0;
			$results[ 'total_count' ] = $total_count;

			return $results;
		} catch ( \Exception $e ) {
			return $count_only ? 0 : $results;
		}
	}

	/**
	 * Get segment filter options by segment id
	 *
	 * @param int|string $segment_id Segment ID.
	 *
	 * @return array|mixed|string
	 * @since 1.0.0
	 */
	private static function get_filters( $segment_id ) {
		global $wpdb;
		$contact_group_table = $wpdb->prefix . ContactGroupSchema::$table_name;

		$sql_query  = "SELECT `data` FROM {$contact_group_table} ";
		$sql_query .= 'WHERE `id` = %d ';
		$sql_query .= 'AND `type` = %s ';

		try {
			$sql_query       = $wpdb->prepare( $sql_query, $segment_id, 'segments' );
			$segment_data    = $wpdb->get_row( $sql_query, ARRAY_A );
			$segment_data    = isset( $segment_data[ 'data' ] ) ? maybe_unserialize( $segment_data[ 'data' ] ) : array();
			$segment_filters = isset( $segment_data[ 'filters' ] ) ? maybe_unserialize( $segment_data[ 'filters' ] ) : array();
		} catch ( \Exception $e ) {
			$segment_filters = array();
		}

		return $segment_filters;
	}

	/**
	 * Prepare and get where query for segment filters
	 *
	 * @param array $filter_options Segment filters.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public static function get_filter_where_query( $filter_options ) {
		$where         = '';
		$self_instance = new self();

		if ( ! is_wp_error( $filter_options ) && is_array( $filter_options ) && ! empty( $filter_options ) ) {
			$filter_option_size   = sizeof( $filter_options );
			$filters_option_count = 0;
			foreach ( $filter_options as $filters ) {
				$filter_size   = sizeof( $filters );
				$filters_count = 0;
				foreach ( $filters as $filter ) {
					if ( isset( $filter[ 'name' ], $filter[ 'condition' ], $filter[ 'value' ] ) ) {
						$filter_name      = $filter[ 'name' ];
						$filter_condition = $filter[ 'condition' ];
						$filter_value     = $filter[ 'value' ];
						$filters_count ++;
						if ( method_exists( $self_instance, "{$filter_condition}" ) ) {
							if ( 'in_the_between_dates' === $filter_condition ) {
								$end_date = isset( $filter['endDate'] ) ? $filter['endDate'] : '';
								$where .= $self_instance->$filter_condition($filter_name, $filter_value, $end_date);
							}else{
								$where .= $self_instance->$filter_condition( $filter_name, $filter_value );
							}
							$where .= $filter_size > $filters_count ? 'AND ' : '';
						}
					}
				}
				$where .= $filter_option_size > ++ $filters_option_count ? 'OR ' : '';
			}
		}

		return $where;
	}

	/**
	 * Prepare segment filter data as per API data format
	 *
	 * @param array $raw_filters Raw filters from segment.
	 *
	 * @return array
	 */
	public static function get_prepared_data( $raw_filters ) {
		$formatted_filters = array();

		foreach ( $raw_filters as $filter ) {
			if ( ! empty( $filter[ 'field' ][ 'value' ] ) && ! empty( $filter[ 'action_condition' ][ 'value' ] ) ) {
				if ( is_array( $filter[ 'action_values' ][ 'value' ] ) ) {
					$values = array_map(function ($item) {
						return isset($item['id']) ? $item['id'] : $item['value'];
					}, $filter['action_values']['value']);
				} else {
					$values = array( $filter[ 'action_values' ][ 'value' ] );
				}

				$temp_data = array(
					'name'      => $filter[ 'field' ][ 'value' ],
					'condition' => $filter[ 'action_condition' ][ 'value' ],
					'value'     => $values,
				);

				if( isset( $filter['action_values']['endDate'] ) ) {
					$temp_data['endDate'] = $filter['action_values']['endDate'];

				}

				if ( ! empty( $filter[ 'condition_logic' ] ) ) {
					if ( 'AND' === $filter[ 'condition_logic' ] ) {
						$size = sizeof( $formatted_filters );
						if ( 0 === (int) $size ) {
							$formatted_filters[ $size ][] = $temp_data;
						} else {
							$formatted_filters[ $size - 1 ][] = $temp_data;
						}
					} elseif ( 'OR' === $filter[ 'condition_logic' ] ) {
						$formatted_filters[][] = $temp_data;
					}
				}
			}
		}

		return $formatted_filters;
	}

	/**
	 * Get segment id
	 *
	 * @param int|string $segment_id Segment ID.
	 * @param string     $columns Column name separated with comma.
	 *
	 * @return array
	 *
	 * @since 1.0.0
	 */
	public static function get_segment( $segment_id, $columns = '', $offset = 0, $per_batch = 0 ) {
		$segment_data[ 'segment_data' ]              = $segment_id ? ContactGroupModel::get( $segment_id ) : array();
		$segment_data[ 'segment_data' ][ 'filters' ] = array();

		if ( ! empty( $segment_data[ 'segment_data' ] ) && isset( $segment_data[ 'segment_data' ][ 'data' ] ) ) {
			$segment_data[ 'segment_data' ][ 'filters' ] = unserialize( $segment_data[ 'segment_data' ][ 'data' ] );
			unset( $segment_data[ 'segment_data' ][ 'data' ] );
		}
		$segment_data[ 'contacts' ] = $segment_id ? self::get_contacts( $segment_id, $columns, $offset, $per_batch ) : array();

		return $segment_data;
	}

	/**
	 * Get where query that are exactly
	 * withing a list of given values
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function are_exactly_in( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return $wpdb->prepare( '%1s IN (%s) ', 'c1.' . $filter_name, $filter_value );
		} else {
			return self::get_meta_join_query( $filter_name, "('$filter_value')", 'IN' );
		}
	}

	/**
	 * Get where query that are exactly
	 * withing a list of given values
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function are_not_in( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return $wpdb->prepare( '%1s NOT IN (%s) ', 'c1.' . $filter_name, $filter_value );
		} else {
			return self::get_meta_join_query( $filter_name, "('$filter_value')", 'NOT IN' );
		}
	}

	/**
	 * Get where query that contains a substring
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function contains( string $filter_name, array $filter_value ) {
		global $wpdb;
		$query       = '';
		$filter_size = sizeof( $filter_value );

		foreach ( $filter_value as $value ) {
			if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
				$query .= "c1.{$filter_name} LIKE '%%{$wpdb->esc_like( $value )}%%' ";
				$query .= $filter_size > 1 ? 'OR ' : '';
			} else {
				$query .= self::get_meta_join_query( $filter_name, "'%%{$wpdb->esc_like( $value )}%%'", 'LIKE' );
				$query .= $filter_size > 1 ? 'OR ' : '';
			}
		}

		return $query;
	}

	/**
	 * Get where query that does not contain a substring
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function does_not_contain( string $filter_name, array $filter_value ) {
		global $wpdb;
		$query       = '';
		$filter_size = sizeof( $filter_value );

		foreach ( $filter_value as $value ) {
			if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
				$query .= "c1.{$filter_name} NOT LIKE '%%{$wpdb->esc_like( $value )}%%' ";
				$query .= $filter_size > 1 ? 'OR ' : '';
			} else {
				$query .= self::get_meta_join_query( $filter_name, "'%%{$wpdb->esc_like( $value )}%%'", 'NOT LIKE' );
				$query .= $filter_size > 1 ? 'OR ' : '';
			}
		}

		return $query;
	}

	/**
	 * Get where query that starts with a specific string
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function starts_with( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return "c1.{$filter_name} LIKE '{$wpdb->esc_like( $filter_value )}%' ";
		} else {
			return self::get_meta_join_query( $filter_name, "'{$wpdb->esc_like( $filter_value )}%'", 'LIKE' );
		}
	}

	/**
	 * Get where query that does not start with a specific string
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function does_not_start_with( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return "c1.{$filter_name} NOT LIKE '{$wpdb->esc_like( $filter_value )}%' ";
		} else {
			return self::get_meta_join_query( $filter_name, "'{$wpdb->esc_like( $filter_value )}%'", 'NOT LIKE' );
		}
	}

	/**
	 * Get where query that ends with a specific string
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function ends_with( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return "c1.{$filter_name} LIKE '%{$wpdb->esc_like( $filter_value )}' ";
		} else {
			return self::get_meta_join_query( $filter_name, "'%{$wpdb->esc_like( $filter_value )}'", 'LIKE' );
		}
	}

	/**
	 * Get where query that does not end with a specific string
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function does_not_end_with( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return "c1.{$filter_name} NOT LIKE '%{$wpdb->esc_like( $filter_value )}' ";
		} else {
			return self::get_meta_join_query( $filter_name, "'%{$wpdb->esc_like( $filter_value )}'", 'NOT LIKE' );
		}
	}

	/**
	 * Get where query that is not empty
	 *
	 * @param string $filter_name Filter name.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function have_been_entered( string $filter_name ) {
		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return "c1.{$filter_name} <> '' ";
		} else {
			return self::get_meta_join_query( $filter_name, "''", '<>' );
		}
	}

	/**
	 * @desc Get where query that not empty
	 *
	 * @param string $filter_name Filter name.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function have_not_been_entered( string $filter_name ) {
		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			return "c1.{$filter_name} = '' ";
		} else {
			return self::get_meta_join_query( $filter_name, "''", '=' );
		}
	}

	/**
	 * Get where query that filters with email subscription status
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function email_subscription_status_is_equal( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';

		return $wpdb->prepare( "c1.{$filter_name} = %s ", $filter_value );
	}

	/**
	 * Get where query that filters with email subscription status not equal
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function email_subscription_status_is_not_equal( string $filter_name, array $filter_value ) {
		global $wpdb;
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';

		return $wpdb->prepare( "c1.{$filter_name} <> %s ", $filter_value );
	}

	/**
	 * Get where query that filters with greater than filter value
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function greater_than( string $filter_name, array $filter_value ) {
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';
		$filter_names = array( 'total_order_count', 'total_order_value', 'average_order_value' );

		if ( in_array( $filter_name, $filter_names, true ) ) {
			$self_instance = new self();
			return $self_instance->$filter_name($filter_value, '>');
		}

		return self::get_meta_join_query( $filter_name, $filter_value, '>' );
	}

	/**
	 * Get where query that filters before a specific date
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function before( string $filter_name, array $filter_value ) {
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';

		if ( 'first_order_date' === $filter_name || 'last_order_date' === $filter_name ) {
			$self_instance = new self();
			return $self_instance->$filter_name($filter_value, '<');
		}

		return self::get_meta_join_query( $filter_name, "'{$filter_value}'", '<' );
	}

	/**
	 * Get where query that filters after a specific date
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function after( string $filter_name, array $filter_value ) {
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';

		if ('first_order_date' === $filter_name || 'last_order_date' === $filter_name) {
			$self_instance = new self();
			return $self_instance->$filter_name($filter_value, '>');
		}

		return self::get_meta_join_query( $filter_name, "'{$filter_value}'", '>' );
	}

	/**
	 * Get where query that filters after a specific date
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function in_the_date( string $filter_name, array $filter_value ) {
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';

		if ('first_order_date' ===$filter_name || 'last_order_date' === $filter_name) {
			$self_instance = new self();
			return $self_instance->$filter_name($filter_value, '=');
		}

		return self::get_meta_join_query( $filter_name, "'{$filter_value}'", '=' );
	}

	/**
	 * Get where query that filters between two dates.
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function in_the_between_dates( $filter_name, $start_date, $end_date ) {
		$start_date = isset($start_date[0]) ? $start_date[0] : '';

		if ('first_order_date' === $filter_name || 'last_order_date' === $filter_name) {
			$self_instance = new self();
			return $self_instance->{$filter_name . '_between'}($start_date, $end_date);
		}
	}

	/**
	 * Get where query that filters with less than filter value
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function less_than( string $filter_name, array $filter_value ) {
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';
		$filter_names = array('total_order_count', 'total_order_value', 'average_order_value');

		if (in_array($filter_name, $filter_names, true)) {
			$self_instance = new self();
			return $self_instance->$filter_name($filter_value, '<');
		}

		return self::get_meta_join_query( $filter_name, $filter_value, '<' );
	}

	/**
	 * Get where query that filters with less than filter value
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function equal( string $filter_name, array $filter_value ) {
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';
		$filter_names = array('total_order_count', 'total_order_value', 'average_order_value');

		if (in_array($filter_name, $filter_names, true)) {
			$self_instance = new self();
			return $self_instance->$filter_name($filter_value, '=');
		}

		return self::get_meta_join_query( $filter_name, $filter_value, '=' );
	}

	/**
	 * Get where query that filters with less than filter value
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function does_not_equal( string $filter_name, array $filter_value ) {
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';
		$filter_names = array('total_order_count', 'total_order_value', 'average_order_value');

		if (in_array($filter_name, $filter_names, true)) {
			$self_instance = new self();
			return $self_instance->$filter_name($filter_value, '<>');
		}

		return self::get_meta_join_query( $filter_name, $filter_value, '<>' );
	}

	/**
	 * Get where query that filters with lists/tags includes in
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function groups_includes_in( string $filter_name, array $filter_value ) {
		global $wpdb;
		$contact_table             = $wpdb->prefix . ContactSchema::$table_name;
		$contact_group_pivot_table = $wpdb->prefix . ContactGroupPivotSchema::$table_name;
		$filter_value              = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			$query  = $wpdb->prepare( 'c1.id IN (SELECT c2.id FROM %1s AS c2 ', $contact_table );
			$query .= $wpdb->prepare( 'JOIN %1s AS cgp1 ', $contact_group_pivot_table );
			$query .= 'ON c2.id = cgp1.contact_id ';
			$query .= $wpdb->prepare( 'WHERE cgp1.group_id IN (%1s)) ', $filter_value );

			return $query;
		} else {
			return self::get_meta_join_query( $filter_name, "IN ('$filter_value')", '' );
		}
	}

	/**
	 * Get where query that filters with lists/tags includes in
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function custom_field_includes_in( string $filter_name, array $filter_value ) {
		global $wpdb;

		$query        = '';
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';
		$query       .= self::get_meta_join_query( $filter_name, "'%%{$wpdb->esc_like( $filter_value )}%%'", 'LIKE' );

		return $query;
	}

	/**
	 * Get where query that filters with lists/tags includes in
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function custom_field_does_not_include_in( string $filter_name, array $filter_value ) {
		global $wpdb;

		$query        = '';
		$filter_value = isset( $filter_value[ 0 ] ) ? $filter_value[ 0 ] : '';
		$query       .= self::get_meta_join_query( $filter_name, "'%%{$wpdb->esc_like( $filter_value )}%%'", 'NOT LIKE' );

		return $query;
	}

	/**
	 * Get where query that filters with lists/tags does not include in
	 *
	 * @param string $filter_name Filter name.
	 * @param array  $filter_value Filter values.
	 *
	 * @return string
	 * @since 1.0.0
	 */
	private function groups_does_not_include_in( string $filter_name, array $filter_value ) {
		global $wpdb;
		$contact_table             = $wpdb->prefix . ContactSchema::$table_name;
		$contact_group_pivot_table = $wpdb->prefix . ContactGroupPivotSchema::$table_name;
		$filter_value              = implode( ', ', $filter_value );

		if ( in_array( $filter_name, self::$contact_primary_field, true ) ) {
			$query  = $wpdb->prepare( 'c1.id NOT IN (SELECT c3.id FROM %1s AS c3 ', $contact_table );
			$query .= $wpdb->prepare( 'JOIN %1s AS cgp2 ', $contact_group_pivot_table );
			$query .= 'ON c3.id = cgp2.contact_id ';
			$query .= $wpdb->prepare( 'WHERE cgp2.group_id IN (%1s)) ', $filter_value );
			return $query;
		} else {
			return self::get_meta_join_query( $filter_name, "NOT IN ('$filter_value')", '' );
		}
	}

	/**
	 * Get join query for contact meta keys
	 *
	 * @param string $filter_name Filter condition name.
	 * @param string $filter_value Filter condition value.
	 * @param string $operator Operator.
	 *
	 * @return string
	 *
	 * @since 1.0.0
	 */
	private static function get_meta_join_query( string $filter_name, string $filter_value, string $operator ) {
		global $wpdb;
		$contact_table      = $wpdb->prefix . ContactSchema::$table_name;
		$contact_meta_table = $wpdb->prefix . ContactMetaSchema::$table_name;

		$query  = $wpdb->prepare( 'c1.id IN (SELECT c4.id FROM %1s AS c4 ', $contact_table );
		$query .= $wpdb->prepare( 'JOIN %1s AS cm1 ', $contact_meta_table );
		$query .= 'ON c4.id = cm1.contact_id ';
		$query .= $wpdb->prepare( 'WHERE cm1.meta_key = %s ', $filter_name );
		$query .= "AND cm1.meta_value {$operator} {$filter_value})";
		return $query;
	}

	/**
	 * Constructs an SQL query snippet to filter customers based on their total order count.
	 *
	 * This function uses the provided filter value and operator to create a query snippet
	 * that checks if a customer's email is in a subquery result set.
	 *
	 * @param int $filter_value The value to compare the total order count against.
	 * @param string $operator The SQL operator to use for comparison (e.g., '=', '>', '<').
	 * 
	 * @return string The constructed SQL query snippet.
	 * 
	 * @since 1.6.1
	 */
	private function total_order_count( $filter_value, $operator ) {
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE t2.total_order_count {$operator} %d )", $filter_value );
		return $query_snippet;
	}

	/**
	 * Constructs an SQL query snippet to filter customers based on their total order value.
	 *
	 * This function uses the provided filter value and operator to create a query snippet
	 * that checks if a customer's email is in a subquery result set.
	 *
	 * @param int $filter_value The value to compare the total order value against.
	 * @param string $operator The SQL operator to use for comparison (e.g., '=', '>', '<').
	 * 
	 * @return string The constructed SQL query snippet.
	 * 
	 * @since 1.6.1
	 */
	private function total_order_value( $filter_value, $operator ) {
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE t2.total_order_value {$operator} %s )", $filter_value );
		return $query_snippet;
	}

	private function average_order_value( $filter_value, $operator ) {
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';
		$decimal_precision = get_option( 'woocommerce_price_num_decimals', 2 );

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE ROUND(t2.aov, $decimal_precision) {$operator} %s )", ROUND($filter_value, $decimal_precision) );
		return $query_snippet;
	}

	/**
	 * Generate a SQL query to check the existence of contacts with a specific filter.
	 *
	 * This function generates a SQL query to check if contacts with a specific filter
	 * (e.g., billing email) exist based on the provided filter name and filter value.
	 *
	 * @access private
	 *
	 * @param string $filter_name  The name of the filter (e.g., 'billing_email').
	 * @param mixed  $filter_value The value to filter by.
	 *
	 * @return string The generated SQL query to check the existence of contacts.
	 * 
	 * @since 1.5.5
	 */
	private function is_exist( $filter_name, $filter_value ) {
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';
		
		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT DISTINCT t2.email_address FROM {$wc_customers_table} AS t2)" );
		return $query_snippet;
	}

	/**
	 * Generate a SQL query to check the non-existence of contacts with a specific filter.
	 *
	 * This function generates a SQL query to check if contacts with a specific filter
	 * (e.g., billing email) do not exist based on the provided filter name and filter value.
	 *
	 * @access private
	 *
	 * @param string $filter_name  The name of the filter (e.g., 'billing_email').
	 * @param mixed  $filter_value The value to filter by.
	 *
	 * @return string The generated SQL query to check the non-existence of contacts.
	 * @since 1.5.5
	 */
	private function is_not_exist( $filter_name, $filter_value ) {
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare("c1.email NOT IN ( SELECT DISTINCT t2.email_address FROM {$wc_customers_table} AS t2)");
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for first_order_date condition.
	 *
	 * @param string $date The date to compare against.
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function first_order_date( $filter_value, $operator ){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE DATE(t2.f_order_date) {$operator} %s )", $filter_value );

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for last_order_date condition.
	 *
	 * @param string $date The date to compare against.
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function last_order_date( $filter_value, $operator ){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE DATE(t2.l_order_date) {$operator} %s )", $filter_value );

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for first_order_date condition.
	 *
	 * @param string $start_date The start date to compare against.
	 * @param string $end_date The end date to compare against.
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function first_order_date_between( $start_date, $end_date ){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE DATE(t2.f_order_date) BETWEEN %s AND %s )", $start_date, $end_date );

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for last_order_date condition.
	 *
	 * @param string $start_date The start date to compare against.
	 * @param string $end_date The end date to compare against.
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function last_order_date_between( $start_date, $end_date ){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the SQL query snippet.
		$query_snippet = $wpdb->prepare( "c1.email IN ( SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE DATE(t2.l_order_date) BETWEEN %s AND %s )", $start_date, $end_date );

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for purchased_products condition.
	 *
	 * @param array $filter_value The filter value for the purchased products.
	 * @param string $operator The comparison operator (e.g., 'JSON_CONTAINS', 'NOT JSON_CONTAINS').
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function wc_select_field_includes_in( $filter_name, $filter_value ) {
		if( empty( $filter_value ) ) {
			return '';
		}

		$self_instance = new self();
		return $self_instance->$filter_name($filter_value, 'JSON_CONTAINS');
	}

	/**
	 * Generate SQL query snippet for purchased_products condition.
	 *
	 * @param array $filter_value The filter value for the purchased products.
	 * @param string $operator The comparison operator (e.g., 'JSON_CONTAINS', 'NOT JSON_CONTAINS').
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function wc_select_field_does_not_include_in($filter_name, $filter_value){
		if (empty($filter_value)) {
			return '';
		}

		$self_instance = new self();
		return $self_instance->$filter_name($filter_value, 'NOT JSON_CONTAINS');
	}

	/**
	 * Generate SQL query snippet for purchased_products condition.
	 *
	 * @param array $filter_value The filter value for the purchased products.
	 * @param string $operator The comparison operator (e.g., 'JSON_CONTAINS', 'NOT JSON_CONTAINS').
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function purchased_products($values, $operator){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the JSON_CONTAINS query for each value.
		$json_clauses = array();
		foreach ($values as $value) {
			$json_clauses[] = $wpdb->prepare("{$operator}(t2.purchased_products, %s)", json_encode($value));
		}

		// Join the JSON_CONTAINS conditions with OR.
		$where_clause = implode(' OR ', $json_clauses);

		// Construct the subquery for the IN clause.
		$query_snippet = $wpdb->prepare("c1.email IN (SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE ({$where_clause}))");

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for purchased_products_tags condition.
	 *
	 * @param array $filter_value The filter value for the purchased products tags.
	 * @param string $operator The comparison operator (e.g., 'JSON_CONTAINS', 'NOT JSON_CONTAINS').
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function purchased_tags($values, $operator){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the JSON_CONTAINS query for each value.
		$json_clauses = array();
		foreach ($values as $value) {
			$json_clauses[] = $wpdb->prepare("{$operator}(t2.purchased_products_tags, %s)", json_encode($value));
		}

		// Join the JSON_CONTAINS conditions with OR.
		$where_clause = implode(' OR ', $json_clauses);

		// Construct the subquery for the IN clause.
		$query_snippet = $wpdb->prepare("c1.email IN (SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE ({$where_clause}))");

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for purchased_products_cats condition.
	 *
	 * @param array $filter_value The filter value for the purchased products categories.
	 * @param string $operator The comparison operator (e.g., 'JSON_CONTAINS', 'NOT JSON_CONTAINS').
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.15.5
	 */
	private function purchased_categories($values, $operator){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		// Construct the JSON_CONTAINS query for each value.
		$json_clauses = array();
		foreach ($values as $value) {
			$json_clauses[] = $wpdb->prepare("{$operator}(t2.purchased_products_cats, %s)", json_encode($value));
		}

		// Join the JSON_CONTAINS conditions with OR.
		$where_clause = implode(' OR ', $json_clauses);

		// Construct the subquery for the IN clause.
		$query_snippet = $wpdb->prepare("c1.email IN (SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE ({$where_clause}))");

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}

	/**
	 * Generate SQL query snippet for used_coupons condition.
	 *
	 * @param array $filter_value The filter value for the used coupons.
	 * @param string $operator The comparison operator (e.g., 'JSON_CONTAINS', 'NOT JSON_CONTAINS').
	 * @return string The SQL query snippet.
	 * 
	 * @since 1.16.1
	 */
	private function used_coupons($values, $operator){
		global $wpdb;
		$wc_customers_table = $wpdb->prefix . 'mint_wc_customers';

		if(!is_array($values)){
			return '';
		}
	
		// Construct the JSON_CONTAINS query for each value.
		$json_clauses = array();
		foreach ($values as $value) {
			if (!empty($value)) {
				$coupon = new \WC_Coupon($value);
				if ($coupon && $coupon->get_id()) {
					$json_clauses[] = $wpdb->prepare("{$operator}(t2.used_coupons, %s)", json_encode($coupon->get_code()));
				}
			}
		}

		// Join the JSON_CONTAINS conditions with OR.
		$where_clause = implode(' OR ', $json_clauses);

		// Construct the subquery for the IN clause.
		$query_snippet = $wpdb->prepare("c1.email IN (SELECT t2.email_address FROM {$wc_customers_table} AS t2 WHERE ({$where_clause}))");

		// Return the constructed SQL query snippet.
		return $query_snippet;
	}
}
