<?php
/**
 * Mail Mint Webhooks functions
 *
 * This class handles Webhook Incomming functions for each WP Actions.
 *
 * @author   Mail Mint Team
 * @category Action
 * @package  MRM
 * @since    1.0.0
 */

namespace MintMailPro\App\Actions;

use MailMintPro\Mint\DataBase\Models\WebHookModel;
use Mint\MRM\Admin\API\Controllers\MessageController;
use Mint\MRM\DataBase\Models\ContactGroupModel;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataStores\ContactData;
use MRM\Common\MrmCommon;

/**
 * WebHooks class.
 */
class WebHooks {

	/**
	 * WebHook into WordPress ready to init.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'mailmint_incoming_webhook' ), 10 );
	}

	/**
	 * Get WenHook data form another action.
	 *
	 * @return void
	 */
	public function mailmint_incoming_webhook() {
		$request = MrmCommon::get_sanitized_get_post();
		$request = $request['get'];
		if ( isset( $request['mailmint'], $request['topic'], $request['route'], $request['hash'] ) && $request['mailmint'] && 'contact' === $request['topic'] && 'webhook' === $request['route'] && !empty( $request['hash'] ) ) {
			if ( $this->method() !== 'POST' ) {
				wp_send_json_error(
					array(
						'message' => __( 'Webhook must need to be as POST Method', 'mailmint-pro' ),
						'type'    => 'invalid_request_method',
					),
					200
				);
			}

			$maybe_allowed = apply_filters( 'mint_webhook_allowed', true );
			if ( !$maybe_allowed ) {
				wp_send_json_error(
					array(
						'message' => __( 'Webhook is not allowed.', 'mailmint-pro' ),
						'type'    => 'invalid_request',
					),
					200
				);
			}

			$hash        = $request['hash'];
			$webhook = WebHookModel::get_hook_by_hash( $hash );
			if ( !empty( $webhook['hash'] ) && $hash === $webhook['hash'] ) {
				$hook_setting = !empty( $webhook['settings'] ) ? $webhook['settings'] : array();
				$post_data    = $this->get_json();

				if ( empty( $post_data ) ) {
					$post_row_data = $this->get_row();
					parse_str( $post_row_data, $post_data );
				}
				if ( !empty( $post_data ) ) {
					if ( !empty( $post_data['fields'] ) ) {
						$fields         = $post_data['fields'];
						$elementor_data = array_map(
							function( $element ) {
								return $element['value'];
							},
							$fields
						);
						$exist_email    = !empty( $elementor_data['email'] ) ? $elementor_data['email'] : '';
						if ( $exist_email ) {
							$maybe_email_allowed = apply_filters( 'mint_webhook_allowed_email', true, $exist_email );
							if ( !$maybe_email_allowed ) {
								wp_send_json_error(
									array(
										'message' => __( 'Email is not allowed.', 'mailmint-pro' ),
										'type'    => 'invalid_request',
									),
									200
								);
							}
							$contact = $this->create_contact( $elementor_data, $hook_setting );

							/**
							 * Fires after a webhook action is completed in Mail Mint.
							 *
							 * This action hook is triggered after the incoming webhook is processed and a contact is created or updated.
							 *
							 * @param array $webhook The webhook data, including settings and metadata.
							 * @param array $contact The contact data created or updated from the webhook.
							 * @since 1.14.0
							 */
							do_action( 'mint_after_incoming_webhook', $webhook, $contact );
							wp_send_json_success(
								array(
									'message' => $contact['message'],
									'id'      => $contact['contact_id'],
									'type'    => 'success',
								),
								200
							);
						} else {
							wp_send_json_error(
								array(
									'message' => __( 'The email has not been found.', 'mailmint-pro' ),
									'type'    => 'invalid_validation',
								),
								200
							);
						}
					}
					$email = !empty( $post_data['email'] ) ? $post_data['email'] : '';
					if ( $email ) {
						$contact = $this->create_contact( $post_data, $hook_setting );

						/**
						 * Fires after a webhook action is completed in Mail Mint.
						 *
						 * This action hook is triggered after the incoming webhook is processed and a contact is created or updated.
						 *
						 * @param array $webhook The webhook data, including settings and metadata.
						 * @param array $contact The contact data created or updated from the webhook.
						 * @since 1.14.0
						 */
						do_action( 'mint_after_incoming_webhook', $webhook, $contact );
						wp_send_json_success(
							array(
								'message' => $contact['message'],
								'id'      => $contact['contact_id'],
								'type'    => 'success',
							),
							200
						);
					} else {
						wp_send_json_error(
							array(
								'message' => __( 'The email has not been found.', 'mailmint-pro' ),
								'type'    => 'invalid_validation',
							),
							200
						);
					}
				} else {
					wp_send_json_error(
						array(
							'message' => __( 'The request data must be provided in JSON format.', 'mailmint-pro' ),
							'type'    => 'invalid_validation',
						),
						200
					);
				}
			} else {
				wp_send_json_error(
					array(
						'message' => __( 'The Hash does not match.', 'mailmint-pro' ),
						'type'    => 'invalid_validation',
					),
					200
				);
			}
		}
	}

	/**
	 * Create Or update Contact
	 *
	 * @param array $post_data Post contact Data.
	 * @param array $hook_setting get hook setting data form setting panel.
	 * @return array
	 */
	public function create_contact( $post_data, $hook_setting ) {
		$contact_data                = array();
		$contact_data['meta_fields'] = array();
		$hook_list                   = !empty( $hook_setting['lists'] ) ? $hook_setting['lists'] : array();
		$hook_tag                    = !empty( $hook_setting['tags'] ) ? $hook_setting['tags'] : array();
		$hook_status                 = !empty( $hook_setting['status'] ) ? $hook_setting['status'] : 'pending';

		if ( $post_data ) {
			foreach ( $post_data as $key => $value ) {
				if ( 'email' === $key ) {
					$contact_data['email'] = sanitize_email( $value );
				} elseif ( 'last_name' === $key ) {
					$contact_data['last_name'] = sanitize_text_field( $value );
				} elseif ( 'id' === $key ) {
					$contact_data['id'] = sanitize_text_field( $value );
				} elseif ( 'first_name' === $key ) {
					$contact_data['first_name'] = sanitize_text_field( $value );
				} elseif ( 'scores' === $key ) {
					$contact_data['scores'] = sanitize_text_field( $value );
				} elseif ( 'source' === $key ) {
					$contact_data['source'] = sanitize_text_field( $value );
				} elseif ( 'stage' === $key ) {
					$contact_data['stage'] = sanitize_text_field( $value );
				} elseif ( 'hash' === $key ) {
					$contact_data['hash'] = sanitize_text_field( $value );
				} elseif ( 'wp_user_id' === $key ) {
					$contact_data['wp_user_id'] = sanitize_text_field( $value );
				} elseif ( 'status' === $key ) {
					$contact_data['status'] = sanitize_text_field( $value );
				} elseif ( 'meta_fields' !== $key ) {
					$contact_data['meta_fields'][ $key ] = $value;
				}
			}
			$meta_fields = !empty( $post_data['meta_fields'] ) ? $post_data['meta_fields'] : array();
			foreach ( $meta_fields as $key => $value ) {
				$contact_data['meta_fields'][ $key ] = sanitize_text_field( $value );
			}
		}
		unset( $contact_data['id'] );
		$email                      = !empty( $contact_data['email'] ) ? $contact_data['email'] : '';
		$contact_data['source']     = 'WebHook';
		$contact_data['status']     = $hook_status;
		$meta_fields['meta_fields'] = isset( $contact_data['meta_fields'] ) ? $contact_data['meta_fields'] : array();
		$contact                    = new ContactData( $email, $contact_data );
		$exist_email                = ContactModel::is_contact_exist( $email );
		if ( $exist_email ) {
			$contact_info   = ContactModel::get_contact_by_email( $email );
			$contact_id     = !empty( $contact_info['id'] ) ? $contact_info['id'] : 0;
			$contact_status = !empty( $contact_info['status'] ) ? $contact_info['status'] : $hook_status;

			if ( !empty( $contact_info['status'] ) && 'unsubscribed' === $contact_info['status'] ) {
				$contact_status = $hook_status;
			}
			/**
			 * Send Double Optin Email
			 */
			if ( 'pending' === $contact_status ) {
				MessageController::get_instance()->send_double_opt_in( $contact_id );
			}

			$contact_data['status'] = $contact_status;
			unset( $contact_data['source'] );
			ContactModel::update( $contact_data, $contact_id );
			ContactModel::update_meta_fields( $contact_id, $meta_fields );
			$message = 'Contact Updated';
		} else {
			$contact_id = ContactModel::insert( $contact );
			/**
			 * Send Double Optin Email
			 */
			if ( 'pending' === $contact_data['status'] ) {
				MessageController::get_instance()->send_double_opt_in( $contact_id );
			}
			ContactModel::update_meta_fields( $contact_id, $meta_fields );
			$message = 'Contact Created';
		}

		ContactGroupModel::set_tags_to_contact( $hook_tag, $contact_id );
		ContactGroupModel::set_lists_to_contact( $hook_list, $contact_id );

		return array(
			'contact_id' => $contact_id,
			'message'    => $message,
		);
	}


	/**
	 * Request Method
	 *
	 * @return mixed
	 */
	public function method() {
		return $_SERVER['REQUEST_METHOD']; //phpcs:ignore
	}

	/**
	 * Json data recived
	 *
	 * @return mixed
	 */
	public function get_json() {
		$content = file_get_contents( 'php://input' );

		$json = json_decode( $content, true );
		return $json;
	}

	/**
	 * Get Raw Data from Request.
	 *
	 * @return false|string
	 */
	public function get_row() {
		return file_get_contents( 'php://input' );
	}


}
