<?php
/**
 * Mail Mint WP actions and callback functions
 *
 * This class handles callback functions for each WP Actions.
 *
 * @author   Mail Mint Team
 * @category Action
 * @package  MRM
 * @since    1.0.0
 */

namespace MintMailPro\App\Actions;

use DateTime;
use MailMint\App\Helper;
use MailMintPro\App\Utilities\Helper\MintWC;
use MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts;
use Mint\MRM\Admin\API\Controllers\CampaignController;
use Mint\MRM\DataBase\Models\CampaignModel;
use Mint\MRM\DataBase\Models\CustomFieldModel;
use Mint\MRM\DataBase\Models\EmailModel;
use Mint\MRM\DataBase\Tables\EmailSchema;
use Mint\MRM\Utilites\Helper\Email;
use MintMail\App\Internal\Automation\HelperFunctions;
use MRM\Common\MrmCommon;
use Mint\Utilities\Arr;

/**
 * Hooks class.
 */
class Hooks {

	/**
	 * Hook into WordPress ready to init.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_filter( 'mail_mint_contact_profile_stats', array( $this, 'contact_woocommerce_customer_history' ), 10, 1 );
		add_filter( 'mail_mint_remove_email_footer_watermark', array( $this, 'remove_email_footer_watermark' ), 10, 1 );
		add_filter( 'is_mail_mint_pro_active', array( $this, 'is_pro_plugin_active' ) );
		add_filter( 'is_mail_mint_pro_license_active', array( $this, 'is_license_active' ) );
		add_filter( 'admin_notices', array( $this, 'inactive_license_notice' ) );
		add_filter( 'mint_contacts_attrs', array( $this, 'add_custom_fields_to_contact_attrs' ), 10, 1 );
		add_filter( 'mint_contact_form_7_fields_map', array( $this, 'map_contact_form_7_fields' ), 10, 2 );
		add_action( 'mailmint_process_recurring_campaign', array( $this, 'mint_process_recurring_campaign' ), 10, 1 );
		add_action( MAILMINT_RECURRING_CAMPAIGN_SCHEDULE, array( $this, 'mint_recurring_campaign_schedule' ) );
		add_filter( 'mint_bricks_form_fields_map', array( $this, 'mint_process_bricks_form_fields_map' ), 10, 2 );
		add_filter( 'mint_fluent_form_fields_map', array( $this, 'mint_process_fluent_form_fields_map' ), 10, 2 );
		add_filter( 'mint_wpforms_form_fields_map', array( $this, 'mint_process_wpforms_form_fields_map' ), 10, 2 );
		add_filter( 'mint_gravity_form_fields_map', array( $this, 'mint_process_gravity_form_fields_map' ), 10, 2 );
		add_filter( 'mint_jet_form_builder_fields_map', array( $this, 'mint_process_mint_jet_form_builder_fields_map' ), 10, 2 );
		add_action( 'mailmint_after_clear_transient', array( $this, 'clear_pro_plugin_transients' ) );
	}

	/**
	 * Summary: Generates WooCommerce customer history data for a contact.
	 *
	 * Description: This method compiles various WooCommerce-related metrics for a given contact and enriches the 'customer_summery' array within the contact data.
	 *
	 * @access public
	 *
	 * @param array $contact An array containing contact information, typically retrieved from a database.
	 *
	 * @return array Returns an array containing WooCommerce customer history metrics, added to the 'customer_summery' key within the contact array.
	 *
	 * @since 1.7.0
	 */
	public function contact_woocommerce_customer_history( $contact ) {
		// Extract relevant contact information.
		$email         = isset( $contact['email'] ) ? $contact['email'] : '';
		$contact_id    = isset( $contact['id'] ) ? $contact['id'] : '';
		$customer_info = MrmCommon::get_customer_id_by_email( $email );

		// Get additional customer information from WooCommerce.
		$customer_id      = isset( $customer_info['customer_id'] ) ? $customer_info['customer_id'] : 0;
		$date_last_active = isset( $customer_info['date_last_active'] ) ? $customer_info['date_last_active'] : '';

		// Retrieve timestamps for the last opened, clicked, and sent emails.
		$last_email_open = EmailModel::last_opened_email_single_contact( $contact_id, 'is_open' );
		$last_email_open = $last_email_open ? MrmCommon::date_time_format_with_core( $last_email_open ) : '-';

		$last_email_click = EmailModel::last_clicked_email_single_contact( $contact_id, 'is_click' );
		$last_email_click = $last_email_click ? MrmCommon::date_time_format_with_core( $last_email_click ) : '-';

		$last_email_sent = EmailModel::last_email_sent_single_contact( $contact_id );
		$last_email_sent = $last_email_sent ? MrmCommon::date_time_format_with_core( $last_email_sent ) : '-';

		// Initialize variables for total spent and total orders.
		$total_spent  = 0;
		$total_orders = 0;

		// Retrieve orders associated with the customer.
		$orders = MrmCommon::get_orders_by_customer_email( $email, 'lifetime' );

		// If no orders are found, return default customer summary with zeros.
		if ( empty( $orders ) ) {
			return $contact['customer_summery'] = array(
				'total_orders'     => $total_orders,
				'total_spent'      => MrmCommon::price_format_with_wc_currency( $total_spent ),
				'last_order_date'  => $date_last_active,
				'aov'              => MrmCommon::price_format_with_wc_currency( 0 ),
				'last_email_open'  => $last_email_open,
				'last_email_click' => $last_email_click,
				'last_email_sent'  => $last_email_sent,
			);
		}

		// Iterate through orders to calculate total spent and total orders.
		foreach ( $orders as $order ) {
			if ( $order ) {
				$total_spent += $order['total_amount'];
			}
		}

		// Avoid division by zero; calculate average order value (AOV).
		$total_orders = count( $orders );
		$divide_by    = ( 0 === $total_orders ) ? 1 : $total_orders;

		// Return updated customer summary with calculated metrics.
		return $contact['customer_summery'] = array(
			'total_orders'     => $total_orders,
			'total_spent'      => MrmCommon::price_format_with_wc_currency( $total_spent ),
			'last_order_date'  => MrmCommon::date_time_format_with_core( $date_last_active ),
			'aov'              => MrmCommon::price_format_with_wc_currency( $total_spent / $divide_by ),
			'last_email_open'  => $last_email_open,
			'last_email_click' => $last_email_click,
			'last_email_sent'  => $last_email_sent,
		);
	}

	/**
	 * Remove footer watermark from email
	 *
	 * @param mixed $res Footer watermark result.
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function remove_email_footer_watermark( $res ) {
		$condition = get_option( '_mrm_general_footer_watermark', 'yes' );

		if ( 'yes' === $condition ) {
			return $res;
		}
		return false;
	}


	/**
	 * Define the state if Mail Mint pro plugin is activated or not
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function is_pro_plugin_active() {
		return true;
	}


	/**
	 * Define the state if Mail Mint pro license is activated or not
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function is_license_active() {
		return 'yes' === get_option( 'mail_mint_pro_is_premium', 'no' );
	}

	/**
	 * Show admin notice if Pro license is not activated
	 *
	 * @return void
	 */
	public function inactive_license_notice() {
		if ( !$this->is_license_active() ) {
			$message = sprintf( __( 'Your <strong>Email Marketing Automation - Mail Mint (Pro)</strong> license is not activated. Please <a href="%1$s">activate your license</a> to use all the pro features of <strong>Email Marketing Automation - Mail Mint (Pro)</strong>', 'mailmint-pro' ), esc_url( admin_url( 'admin.php?page=mail-mint-licence' ) ) ); //phpcs:ignore
			?>
		<div class="mailmint-notice notice notice-error is-dismissible">
			<p>
				<?php echo wp_kses_post( $message ); ?>
			</p>
		</div>
			<?php
		}
	}

	/**
	 * Return contact mapping attributes
	 *
	 * @param mixed $contacts_attrs contact mapping attributes.
	 *
	 * @return array
	 * @since 1.0.1
	 */
	public function add_custom_fields_to_contact_attrs( $contacts_attrs ) {
		$customer_fields = CustomFieldModel::get_custom_fields_to_map();

		if ( !is_array( $customer_fields ) || empty( $customer_fields ) ) {
			return $contacts_attrs;
		}

		$customer_fields = array_map(
			function( $customer_field ) {
				$meta                   = isset( $customer_field['meta'] ) ? maybe_unserialize( $customer_field['meta'] ) : array();
				$customer_field['name'] = isset( $meta['label'] ) ? $meta['label'] : '';
				unset( $customer_field['meta'] );
				unset( $customer_field['type'] );
				return $customer_field;
			},
			$customer_fields
		);
		return array_merge( $contacts_attrs, $customer_fields );
	}

	/**
	 * Maps and extracts relevant data from a Contact Form 7 submission based on automation settings.
	 *
	 * This method is used to map and extract specific data from a Contact Form 7 submission based on
	 * the settings configured in the associated automation step.
	 *
	 * @param array $data      The original data array containing the Contact Form 7 submission data.
	 * @param array $step_data Additional data related to the automation step.
	 *
	 * @return array The modified data array with mapped and extracted Contact Form 7 fields.
	 *
	 * @since 1.5.14
	 */
	public function map_contact_form_7_fields( $data, $step_data ) {
		if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
			$step_data     = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			$form_settings = isset( $step_data['settings']['contact_form_settings'] ) ? $step_data['settings']['contact_form_settings'] : array();
			$mapping_data  = isset( $form_settings['mapping'] ) ? $form_settings['mapping'] : array();
			$form_data     = isset( $data['data']['form_data'] ) ? $data['data']['form_data'] : array();
			$form_fields   = array();

			if( ! empty( $mapping_data ) ) {
				foreach ($mapping_data as $mapping) {
					$form_fields[ $mapping['target'] ] = $mapping['source'];
				}

				$map_fields = array();

				foreach ( $form_fields as $key => $value ) {
					if ( isset( $form_data[ $value ] ) ) {
						$map_fields[ $key ] = $form_data[ $value ];
					}
				}

				unset( $data['data']['form_data'] );

				// Dynamically map $map_fields to $data['data'].
				foreach ( $map_fields as $key => $value ) {
					// Ensure the key is formatted properly for $data['data'].
					if( 'email' === $key ) {
						$data['data']['user_email'] = $value;
					} else {
						$data['data'][ $key ] = $value;
					}
				}
				return $data;
			}

			foreach ( $form_settings as $key => $value ) {
				if ( is_array( $value ) && isset( $value['value'] ) ) {
					$form_fields[ $key ] = $value['value'];
				}
			}
			
			$map_fields = array();

			foreach ( $form_fields as $key => $value ) {
				if ( isset( $form_data[ $value ] ) ) {
					$map_fields[ $key ] = $form_data[ $value ];
				}
			}
			unset( $data['data']['form_data'] );
			$data['data']['user_email'] = isset( $map_fields['email'] ) ? $map_fields['email'] : '';
			$data['data']['first_name'] = isset( $map_fields['firstName'] ) ? $map_fields['firstName'] : '';
			$data['data']['last_name']  = isset( $map_fields['lastName'] ) ? $map_fields['lastName'] : '';
			return $data;
		}
		return $data;
	}

	/**
	 * Process a recurring campaign in MailMint.
	 *
	 * This method handles the processing of recurring campaigns based on the specified recurrence frequency
	 * (daily, weekly, monthly). It retrieves the necessary data from the campaign and schedules the recurrence
	 * accordingly.
	 *
	 * @param array $campaign An array containing information about the recurring campaign.
	 *                        - 'id' (int) The ID of the recurring campaign.
	 *
	 * @return void
	 * @since 1.7.0
	 */
	public function mint_process_recurring_campaign( $campaign ) {
		// Extract campaign details.
		$campaign_id          = isset( $campaign[ 'id' ] ) ? $campaign[ 'id' ] : 0;
		$recurring_properties = CampaignModel::get_campaign_meta_value( $campaign_id, 'recurring_properties' );
		$total_recipients     = CampaignModel::get_campaign_meta_value( $campaign_id, 'total_recipients' );

		// Unserialize recurring properties if available and recurrence frequency.
		$recurring_properties = !empty( $recurring_properties ) ? maybe_unserialize( $recurring_properties ) : array();
		$recurring_repeat     = isset( $recurring_properties['schedule'][ 'recurringRepeat' ] ) ? $recurring_properties['schedule'][ 'recurringRepeat' ] : '';

		// Process based on the recurrence frequency.
		switch ( $recurring_repeat ) {
			case 'daily':
				$this->schedule_daily_recurring( $recurring_properties, $campaign_id, $total_recipients );
				break;

			case 'weekly':
				$this->schedule_weekly_recurring( $recurring_properties, $campaign_id, $total_recipients );
				break;

			case 'monthly':
				$this->schedule_monthly_recurring( $recurring_properties, $campaign_id, $total_recipients );
				break;
		}
	}

	/**
	 * Schedule daily recurring campaign in MailMint.
	 *
	 * This method calculates the starting timestamp, interval in seconds, and schedules the asynchronous
	 * recurring action for a daily recurring campaign.
	 *
	 * @param array $recurring_properties An array containing properties of the recurring campaign.
	 * @param int   $campaign_id           The ID of the recurring campaign.
	 * @param int   $total_recipients      The total number of recipients for the campaign.
	 *
	 * @return void
	 * @since 1.7.0
	 */
	public function schedule_daily_recurring( $recurring_properties, $campaign_id, $total_recipients ) {
		// Extract properties.
		$start_date    = isset( $recurring_properties['schedule'][ 'startDate' ] ) ? $recurring_properties['schedule'][ 'startDate' ] : '';
		$recurring_at  = isset( $recurring_properties['schedule'][ 'recurringAt' ] ) ? $recurring_properties['schedule'][ 'recurringAt' ] : '';
		$schedule_date = $start_date . ' ' . $recurring_at;

		// Calculate the starting timestamp.
		$timestamp       = $this->get_recurring_starting_timestamp( $schedule_date );
		$recurring_every = isset( $recurring_properties['schedule'][ 'recurringEvery' ] ) ? $recurring_properties['schedule'][ 'recurringEvery' ] : '';
		// Calculate the interval in seconds (1 day = 24 hours * 60 minutes * 60 seconds).
		$interval_in_seconds = $recurring_every * DAY_IN_SECONDS; // 1 day = 24 hours * 60 minutes * 60 seconds
		$this->schedule_asynchronous_recurring_action( $campaign_id, $total_recipients, $recurring_properties, $timestamp, $interval_in_seconds );
	}

	/**
	 * Get the starting timestamp for a recurring campaign based on the scheduled date.
	 *
	 * This method calculates the difference between the scheduled date and the current date, then adds
	 * this difference to the current timestamp to get the starting timestamp for the recurring campaign.
	 *
	 * @param string $schedule_date The date and time when the recurring campaign is scheduled to start.
	 *
	 * @return int The starting timestamp for the recurring campaign.
	 * @since 1.7.0
	 */
	public function get_recurring_starting_timestamp( $schedule_date ) {
		$current_date  = new \DateTime( 'now', wp_timezone() );
		$schedule_date = date_create( gmdate( 'Y-m-d H:i:s', strtotime( $schedule_date ) ), wp_timezone() );
		$date_diff     = date_diff( $schedule_date, $current_date );
		$date_diff     = '+' . $date_diff->y . 'year' . $date_diff->m . 'month' . $date_diff->d . 'day' . $date_diff->h . 'hour' . $date_diff->i . 'minute' . ( $date_diff->s + 1 ) . 'second';
		return strtotime( $date_diff );
	}

	/**
	 * Schedule an asynchronous recurring action for a campaign.
	 *
	 * This method schedules a recurring action to send campaign emails asynchronously based on the provided
	 * timestamp and interval.
	 *
	 * @param int   $campaign_id          The ID of the campaign.
	 * @param int   $total_recipients     The total number of recipients for the campaign.
	 * @param array $recurring_properties The properties of the recurring campaign.
	 * @param int   $timestamp            The timestamp when the recurring action should start.
	 * @param int   $interval             The interval (in seconds) between each recurring action.
	 *
	 * @since 1.7.0
	 */
	public function schedule_asynchronous_recurring_action( $campaign_id, $total_recipients, $recurring_properties, $timestamp, $interval ) {
		// Get the first email of the campaign.
		$first_email = CampaignModel::get_first_campaign_email( $campaign_id );

		// Prepare arguments for the recurring action.
		$args  = array(
			array(
				'campaign_id'          => $campaign_id,
				'campaign_status'      => 'schedule',
				'email'                => $first_email,
				'total_recipients'     => $total_recipients,
				'recurring_properties' => $recurring_properties,
			),
		);
		$group = 'mailmint-recurring-campaign-schedule-' . $campaign_id;
		as_schedule_recurring_action( $timestamp, $interval, MAILMINT_RECURRING_CAMPAIGN_SCHEDULE, $args, $group );
	}

	/**
	 * Schedule a weekly recurring action for a campaign.
	 *
	 * This method calculates and schedules recurring actions for each specified day of the week.
	 *
	 * @param array $recurring_properties The properties of the recurring campaign.
	 * @param int   $campaign_id          The ID of the campaign.
	 * @param int   $total_recipients     The total number of recipients for the campaign.
	 * @since 1.7.0
	 */
	public function schedule_weekly_recurring( $recurring_properties, $campaign_id, $total_recipients ) {
		$start_date = isset( $recurring_properties['schedule'][ 'startDate' ] ) ? $recurring_properties['schedule'][ 'startDate' ] : '';
		// Get timestamp of the start date.
		$start_timestamp = strtotime( $start_date );

		// Get the numeric representation of the day of the week (0 = Sunday, 1 = Monday, ..., 6 = Saturday).
		$day_number     = gmdate( 'w', $start_timestamp );
		$recurring_at   = isset( $recurring_properties['schedule'][ 'recurringAt' ] ) ? $recurring_properties['schedule'][ 'recurringAt' ] : '';
		$recurring_days = isset( $recurring_properties['schedule'][ 'recurringOn' ] ) ? $recurring_properties['schedule'][ 'recurringOn' ] : array();

		if ( !empty( $recurring_days ) ) {
			foreach ( $recurring_days as $day ) {
				// Calculate the difference in days between the desired day and the day of the start date.
				$day_difference = ( 7 + ( gmdate( 'w', strtotime( $day ) ) - $day_number ) ) % 7;

				// Calculate the timestamp of the desired day.
				$desired_day_timestamp = $start_timestamp + ( $day_difference * 24 * 60 * 60 );

				// Format the timestamp to the desired date format.
				$desired_day_date = gmdate( 'Y-m-d', $desired_day_timestamp );

				$schedule_date = $desired_day_date . ' ' . $recurring_at;

				$timestamp       = $this->get_recurring_starting_timestamp( $schedule_date );
				$recurring_every = isset( $recurring_properties['schedule'][ 'recurringEvery' ] ) ? $recurring_properties['schedule'][ 'recurringEvery' ] : '';
				// Calculate the interval in seconds.
				$interval_in_seconds = $recurring_every * WEEK_IN_SECONDS;
				$this->schedule_asynchronous_recurring_action( $campaign_id, $total_recipients, $recurring_properties, $timestamp, $interval_in_seconds );
			}
		}
	}

	/**
	 * Schedule a monthly recurring action for a campaign.
	 *
	 * This method calculates and schedules recurring actions for each specified day of the month.
	 *
	 * @param array $recurring_properties The properties of the recurring campaign.
	 * @param int   $campaign_id          The ID of the campaign.
	 * @param int   $total_recipients     The total number of recipients for the campaign.
	 *
	 * @since 1.7.0
	 */
	public function schedule_monthly_recurring( $recurring_properties, $campaign_id, $total_recipients ) {
		$start_date = isset( $recurring_properties['schedule'][ 'startDate' ] ) ? $recurring_properties['schedule'][ 'startDate' ] : '';
		// Convert start date to a DateTime object.

		$recurring_at   = isset( $recurring_properties['schedule'][ 'recurringAt' ] ) ? $recurring_properties['schedule'][ 'recurringAt' ] : '';
		$recurring_days = isset( $recurring_properties['schedule'][ 'recurringOn' ] ) ? $recurring_properties['schedule'][ 'recurringOn' ] : array();

		if ( !empty( $recurring_days ) ) {
			foreach ( $recurring_days as $day ) {
				$start_date_obj = new DateTime( $start_date );

				// Get the day part of the start date.
				$start_day = $start_date_obj->format( 'd' );
				// Get the number of days in the current month.
				$days_in_month = cal_days_in_month( CAL_GREGORIAN, $start_date_obj->format( 'm' ), $start_date_obj->format( 'Y' ) );
				// Check if the given day is less than the start day.
				if ( $day < $start_day || $day > $days_in_month ) {
					// If it is, increment the month by 1.
					$start_date_obj->modify( '+1 month' );
				}

				// Set the day part to the new value.
				$start_date_obj->setDate( $start_date_obj->format( 'Y' ), $start_date_obj->format( 'm' ), $day );

				// Get the final date.
				$new_date = $start_date_obj->format( 'Y-m-d' );

				$schedule_date = $new_date . ' ' . $recurring_at;

				$timestamp       = $this->get_recurring_starting_timestamp( $schedule_date );
				$recurring_every = isset( $recurring_properties['schedule'][ 'recurringEvery' ] ) ? $recurring_properties['schedule'][ 'recurringEvery' ] : '';
				// Calculate the interval in seconds.
				$interval_in_seconds = $recurring_every * MONTH_IN_SECONDS;
				$this->schedule_asynchronous_recurring_action( $campaign_id, $total_recipients, $recurring_properties, $timestamp, $interval_in_seconds );
			}
		}
	}

	/**
	 * Schedule recurring campaign emails based on conditions.
	 *
	 * @param array $args An array containing campaign and scheduling information.
	 * @since 1.7.0
	 */
	public function mint_recurring_campaign_schedule( $args ) {
		/**
		 * Signals that processing tasks should be executed after a campaign has started.
		 *
		 * This action hook is triggered after a campaign has been initiated. Developers can hook into
		 * this action to perform additional tasks or operations based on the provided arguments.
		 *
		 * @since 1.7.0
		 *
		 * @param array $args An array of arguments related to the campaign start. The structure of this array
		 *                    may include campaign-specific information needed for further processing.
		 */
		do_action( 'mailmint_after_campaign_start', $args );

		// Check if the conditions for recurring campaigns are satisfied.
		$is_published = $this->is_recurring_conditions_satisfied( $args );
		if ( !$is_published ) {
			return;
		}
		$is_end = $this->is_recurring_end_date_satisfied( $args );
		if ( !$is_end ) {
			return;
		}
		global $wpdb;
		$email_broadcast_table = $wpdb->prefix . EmailSchema::$table_name;
		$email_settings        = get_option( '_mrm_email_settings', Email::default_email_settings() );
		$campaign_id           = ! empty( $args[ 'campaign_id' ] ) ? $args[ 'campaign_id' ] : null;
		$campaign_email        = ! empty( $args[ 'email' ] ) ? $args[ 'email' ] : array();
		$campaign_email_id     = ! empty( $campaign_email[ 'id' ] ) ? $campaign_email[ 'id' ] : null;
		$per_batch             = ! empty( $args[ 'per_batch' ] ) ? $args[ 'per_batch' ] : 500;
		$offset                = ! empty( $args[ 'offset' ] ) ? $args[ 'offset' ] : 0;
		$reply_name            = ! empty( $email_settings[ 'reply_name' ] ) ? $email_settings[ 'reply_name' ] : '';
		$reply_email           = ! empty( $email_settings[ 'reply_email' ] ) ? $email_settings[ 'reply_email' ] : '';
		$sender_email          = ! empty( $email_settings[ 'sender_email' ] ) ? $email_settings[ 'sender_email' ] : '';
		$sender_name           = ! empty( $email_settings[ 'sender_name' ] ) ? $email_settings[ 'sender_name' ] : '';

		$total_recipients = isset( $args[ 'total_recipients' ] ) ? $args[ 'total_recipients' ] : 0;

		$per_batch = 500; // Set the number of emails to process in each batch.

		// Calculate the number of batches.
		$total_batches = ceil( $total_recipients / $per_batch );

		for ( $batch = 0; $batch < $total_batches; $batch++ ) {
			// Calculate offset for the current batch.
			$offset = $batch * $per_batch;

			// Get recipients for the current batch.
			$recipients_emails = CampaignModel::get_campaign_recipients_email( $campaign_id, $offset, $per_batch );

			// Process emails in $recipients_emails array.
			if ( is_array( $recipients_emails ) && ! empty( $recipients_emails ) ) {
				$sender_name   = ! empty( $campaign_email[ 'sender_name' ] ) ? $campaign_email[ 'sender_name' ] : $sender_name;
				$sender_email  = ! empty( $campaign_email[ 'sender_email' ] ) ? $campaign_email[ 'sender_email' ] : $sender_email;
				$reply_name    = ! empty( $campaign_email[ 'reply_name' ] ) ? $campaign_email[ 'reply_name' ] : $reply_name;
				$reply_email   = ! empty( $campaign_email[ 'reply_email' ] ) ? $campaign_email[ 'reply_email' ] : $reply_email;
				$headers       = Helper::prepare_email_headers( $sender_name, $sender_email, $reply_name, $reply_email );

				foreach ( $recipients_emails as $email ) {
					if ( isset( $email[ 'id' ], $email[ 'email' ] ) && $email[ 'id' ] && $email[ 'email' ] ) {
						$email_hash = MrmCommon::get_rand_email_hash( $email[ 'email' ], $campaign_id );

						$wpdb->insert( //phpcs:ignore
							$email_broadcast_table,
							array(
								'campaign_id'   => $campaign_id,
								'email_id'      => $campaign_email_id,
								'contact_id'    => $email[ 'id' ],
								'email_address' => $email[ 'email' ],
								'email_headers' => wp_json_encode($headers),
								'status'        => 'scheduled',
								'email_type'    => 'campaign',
								'email_hash'    => $email_hash,
								'scheduled_at'  => current_time( 'mysql' ),
								'created_at'    => current_time( 'mysql' ),
							)
						);

						usleep( 5000 ); // 5 milliseconds sleep
					}
				}
			}

			if ( !( $batch < $total_batches ) ) {
				// Exit the loop.
				break;
			}
		}

		/**
		 * Signals that the scheduling process for a single email in a campaign has been processed.
		 *
		 * This action hook is triggered after the scheduling process for a single email within a campaign
		 * has been completed. Other parts of the code can hook into this action to perform additional tasks
		 * or operations based on the provided campaign and email ID parameters.
		 *
		 * @since 1.0.0
		 *
		 * @param int $campaign_id        The ID of the campaign for which the email was scheduled.
		 * @param int $campaign_email_id  The ID of the scheduled email within the campaign.
		 */
		do_action( 'mailmint_single_email_scheduling_processed', (int) $campaign_id, (int) $campaign_email_id );

		CampaignModel::update_campaign_email_status( $campaign_id, $campaign_email_id, 'scheduled' );

		/**
		 * Signals the completion of scheduling emails for a recurring campaign.
		 *
		 * This action hook is triggered when all email scheduling tasks for a recurring campaign have been completed.
		 * Developers can hook into this action to perform additional tasks or operations after email scheduling is finished.
		 *
		 * @since 1.7.0
		 *
		 * @param string $schedule_group The unique identifier for the scheduling group associated with the recurring campaign.
		 *                               It helps identify and manage the scheduled tasks for this campaign.
		 * @param int    $campaign_id    The ID of the recurring campaign for which email scheduling has been completed.
		 */
		do_action( 'mailmint_campaign_emails_scheduling_completed', 'mailmint-recurring-campaign-schedule-' . $campaign_id );
	}

	/**
	 * Check if the conditions for recurring campaigns are satisfied.
	 *
	 * @param array $args Arguments containing recurring properties.
	 *
	 * @return bool Whether the conditions are satisfied.
	 * @since 1.7.0
	 */
	public function is_recurring_conditions_satisfied( $args ) {
		$conditions = isset( $args['recurring_properties']['conditions'] ) ? $args['recurring_properties']['conditions'] : array();
		if ( empty( $conditions ) ) {
			return true;
		}

		if (array_key_exists('enableConditions', $args['recurring_properties']['schedule'])) {
			if (empty($args['recurring_properties']['schedule']['enableConditions'])) {
				return true;
			}
		}
		// Initialize the result to false.
		$result = false;

		// Loop through the outer conditions (OR operator).
		foreach ( $conditions as $outer_condition ) {

			// Initialize the inner result to true for OR operator.
			$inner_result = true;

			// Loop through the steps array for each outer condition (AND operator).
			foreach ( $outer_condition['steps'] as $step ) {
				$publish_item    = $step['publishItem'];
				$publish_with_in = $step['publishWithIn'];

				// Check if the condition is satisfied for each step.
				$inner_result = $inner_result && $this->is_published_within_days( $publish_item, $publish_with_in );

				// If inner result is false, break the inner loop.
				if ( !$inner_result ) {
					break;
				}
			}

			// Update the overall result for OR operator.
			$result = $result || $inner_result;

			// If overall result is true, break the outer loop.
			if ( $result ) {
				break;
			}
		}

		return $result;
	}

	/**
	 * Check if there are published posts of a given post type within the specified number of days.
	 *
	 * @param string $post_type The post type to check.
	 * @param int    $days     The number of days within which to check for published posts.
	 *
	 * @return bool Whether there are published posts within the specified days.
	 * @since 1.7.0
	 */
	public function is_published_within_days( $post_type, $days ) {
		// Set up the date query.
		$date_query = array(
			'after'     => gmdate( 'Y-m-d', strtotime( "-$days days" ) ),
			'inclusive' => true,
		);
		// Set up the query arguments.
		$args = array(
			'post_type'      => $post_type,
			'posts_per_page' => 1,
			'date_query'     => array( $date_query ),
		);

		// Get the posts.
		$posts = get_posts( $args );

		// Check if any posts were found.
		return !empty( $posts );
	}

	/**
	 * Checks if the recurring campaign has reached its specified end date.
	 *
	 * This method evaluates whether the current date is on or before the specified end date of the recurring campaign.
	 * If the end date is set to 'never_end' or the current date is within the specified range, it returns true.
	 * Otherwise, it updates the campaign status to 'draft' and unschedules any associated campaign actions.
	 *
	 * @since 1.7.0
	 *
	 * @param array $args Array of arguments.
	 *
	 * @return bool True if the recurring campaign has not reached its end date or has no end date, false otherwise.
	 */
	public function is_recurring_end_date_satisfied( $args ) {
		$end_date     = isset( $args['recurring_properties']['schedule']['endDate'] ) ? $args['recurring_properties']['schedule']['endDate'] : 'never_end';
		$campaign_id  = isset( $args[ 'campaign_id' ] ) ? $args[ 'campaign_id' ] : null;
		$current_date = current_time( 'Y-m-d' );

		if ( 'never_end' === $end_date || strtotime( $current_date ) <= strtotime( $end_date ) ) {
			return true;
		}

		CampaignModel::update_campaign_status( $campaign_id, 'draft' );
		CampaignModel::unschedule_campaign_actions( $campaign_id );
	}

	/**
	 * Maps and extracts relevant data from a Bricks form submission based on automation settings.
	 *
	 * This method is used to map and extract specific data from a Bricks form submission based on
	 * the settings configured in the associated automation step.
	 *
	 * @param array $data      The original data array containing the Bricks form submission data.
	 * @param array $step_data Additional data related to the automation step.
	 *
	 * @return array The modified data array with mapped and extracted Bricks form fields.
	 * @since 1.14.1
	 */
	public function mint_process_bricks_form_fields_map( $data, $step_data ) {
		if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
			$step_data     = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			$form_settings = isset( $step_data['settings']['bricks_form_settings'] ) ? $step_data['settings']['bricks_form_settings'] : array();
			$mapping_data  = isset( $form_settings['mapping'] ) ? $form_settings['mapping'] : array();
			$form_fields   = array();

			foreach ($mapping_data as $mapping) {
				$form_fields[ $mapping['target'] ] = $mapping['source'];
			}

			$form_data  = isset( $data['data']['form_data'] ) ? $data['data']['form_data'] : array();
			$map_fields = array();

			foreach ( $form_fields as $key => $value ) {
				if ( isset( $form_data[ $value ] ) ) {
					$map_fields[ $key ] = $form_data[ $value ];
				}
			}

			unset( $data['data']['form_data'] );
			// Dynamically map $map_fields to $data['data'].
			foreach ( $map_fields as $key => $value ) {
				// Ensure the key is formatted properly for $data['data'].
				if( 'email' === $key ) {
					$data['data']['user_email'] = $value;
				} else {
					$data['data'][ $key ] = $value;
				}
			}
			return $data;
		}
		return $data;
	}

	/**
	 * Maps and extracts relevant data from a Fluent Form submission based on automation settings.
	 *
	 * This method is used to map and extract specific data from a Fluent Form submission based on
	 * the settings configured in the associated automation step.
	 *
	 * @param array $data      The original data array containing the Fluent Form submission data.
	 * @param array $step_data Additional data related to the automation step.
	 *
	 * @return array The modified data array with mapped and extracted Fluent Form fields.
	 * @since 1.15.1
	 */
	public function mint_process_fluent_form_fields_map( $data, $step_data ) {
		if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
			$step_data     = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			$form_settings = isset( $step_data['settings']['fluentform_settings'] ) ? $step_data['settings']['fluentform_settings'] : array();
			$mapping_data  = isset( $form_settings['mapping'] ) ? $form_settings['mapping'] : array();
			$form_fields   = array();

			foreach ($mapping_data as $mapping) {
				$form_fields[ $mapping['target'] ] = $mapping['source'];
			}

			$map_fields = array();

			foreach ( $form_fields as $key => $value ) {
				$map_fields[$key] = Arr::get_multidimensional($data, $value);
			}

			unset( $data['data']['form-data'] );
			// Dynamically map $map_fields to $data['data'].
			foreach ( $map_fields as $key => $value ) {
				// Ensure the key is formatted properly for $data['data'].
				if( 'email' === $key ) {
					$data['data']['user_email'] = $value;
				} else {
					$data['data'][ $key ] = $value;
				}
			}
			return $data;
		}
		return $data;
	}


		/**
	 * Maps and extracts relevant data from a WPForms Form submission based on automation settings.
	 *
	 * This method is used to map and extract specific data from a WPForms Form submission based on
	 * the settings configured in the associated automation step.
	 *
	 * @param array $data      The original data array containing the WPForms Form submission data.
	 * @param array $step_data Additional data related to the automation step.
	 *
	 * @return array The modified data array with mapped and extracted WPForms Form fields.
	 * @since 1.17.0
	 */
	public function mint_process_wpforms_form_fields_map( $data, $step_data ) {
		if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {

			$step_data     = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			$form_settings = isset( $step_data['settings']['wpforms_settings'] ) ? $step_data['settings']['wpforms_settings'] : array();
			$mapping_data  = isset( $form_settings['mapping'] ) ? $form_settings['mapping'] : array();
			$wp_form_data  = isset($data['data']['form_data']) ? $data['data']['form_data'] : array();
			
			$form_fields   = array();

			foreach ($mapping_data as $mapping) {
				$form_fields[ $mapping['target'] ] = $mapping['source'];
			}

			$map_fields = [];

			foreach ($form_fields as $key => $id_to_match) {
				foreach ($wp_form_data as $item) {
					if (isset($item['id']) && (int)$item['id'] == (int)$id_to_match) {
						$map_fields[$key] = $item['value'];
						break;
					}
				}
			}

			unset( $data['data']['form_data'] );
			foreach ( $map_fields as $key => $value ) {
				if( 'email' === $key ) {
					$data['data']['user_email'] = $value;
				} else {
					$data['data'][ $key ] = $value;
				}
			}
			return $data;
		}
		return $data;
	}

	/**
	 * Maps and extracts relevant data from a Gravity Form submission based on automation settings.
	 *
	 * This method is used to map and extract specific data from a Gravity Form submission based on
	 * the settings configured in the associated automation step.
	 *
	 * @param array $data      The original data array containing the Gravity Form submission data.
	 * @param array $step_data Additional data related to the automation step.
	 *
	 * @return array The modified data array with mapped and extracted Gravity Form fields.
	 * @since 1.15.1
	 */
	public function mint_process_gravity_form_fields_map( $data, $step_data ) {
		if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
			$step_data     = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			$form_settings = isset( $step_data['settings']['gform_settings'] ) ? $step_data['settings']['gform_settings'] : array();
			$mapping_data  = isset( $form_settings['mapping'] ) ? $form_settings['mapping'] : array();
			$form_fields   = array();

			foreach ($mapping_data as $mapping) {
				$form_fields[ $mapping['target'] ] = $mapping['source'];
			}

			$form_data  = isset( $data['data']['entry'] ) ? $data['data']['entry'] : array();
			$map_fields = array();

			foreach ( $form_fields as $key => $value ) {
				if ( isset( $form_data[ $value ] ) ) {
					$map_fields[ $key ] = $form_data[ $value ];
				}
			}

			unset( $data['data']['entry'] );
			// Dynamically map $map_fields to $data['data'].
			foreach ( $map_fields as $key => $value ) {
				// Ensure the key is formatted properly for $data['data'].
				if( 'email' === $key ) {
					$data['data']['user_email'] = $value;
				} else {
					$data['data'][ $key ] = $value;
				}
			}
			return $data;
		}
		return $data;
	}

	/**
	 * Maps and extracts relevant data from a JetFormBuilder submission based on automation settings.
	 *
	 * This method is used to map and extract specific data from a JetFormBuilder submission based on
	 * the settings configured in the associated automation step.
	 *
	 * @param array $data      The original data array containing the JetFormBuilder submission data.
	 * @param array $step_data Additional data related to the automation step.
	 *
	 * @return array The modified data array with mapped and extracted Jet Form Builder fields.
	 * @since 1.16.4
	 */
	public function mint_process_mint_jet_form_builder_fields_map( $data, $step_data ) {
		if ( isset( $step_data['automation_id'], $step_data['automation_id'] ) ) {
			$step_data     = HelperFunctions::get_step_data( $step_data['automation_id'], $step_data['step_id'] );
			$form_settings = isset( $step_data['settings']['jetform_settings'] ) ? $step_data['settings']['jetform_settings'] : array();
			$mapping_data  = isset( $form_settings['mapping'] ) ? $form_settings['mapping'] : array();
			$form_data     = isset( $data['data']['form_data'] ) ? $data['data']['form_data'] : array();

			if ( empty( $mapping_data ) ) {
				foreach ( $form_data as $key => $value ) {
					// Ensure the key is formatted properly for $data['data'].
					if( 'email' === $key ) {
						$data['data']['user_email'] = $value;
					} else {
						$data['data'][ $key ] = $value;
					}
				}
				unset( $data['data']['form_data'] );
				return $data;
			}
			$form_fields   = array();

			foreach ($mapping_data as $mapping) {
				$form_fields[ $mapping['target'] ] = $mapping['source'];
			}

			$map_fields = array();

			foreach ( $form_fields as $key => $value ) {
				if ( isset( $form_data[ $value ] ) ) {
					$map_fields[ $key ] = $form_data[ $value ];
				}
			}

			unset( $data['data']['form_data'] );

			// Dynamically map $map_fields to $data['data'].
			foreach ( $map_fields as $key => $value ) {
				// Ensure the key is formatted properly for $data['data'].
				if( 'email' === $key ) {
					$data['data']['user_email'] = $value;
				} else {
					$data['data'][ $key ] = $value;
				}
			}
			return $data;
		}
		return $data;
	}

	/**
	 * Clear transients related to pro plugin updates.
	 *
	 * This function deletes the 'update_plugins' site transient and clears the pro update check transient.
	 * It also triggers an action to check update transients.
	 * 
	 * @since 1.16.4
	 */
	public function clear_pro_plugin_transients(){
		delete_site_transient('update_plugins');
		$this->clear_pro_update_check_transient();
		ob_start();
		do_action('mailmint_check_update_transient');
		ob_get_clean();
	}

	/**
	 * Clear the pro update check transient.
	 *
	 * This function deletes the transient and its timeout from the options table in the database.
	 * 
	 * @since 1.16.4
	 */
	private function clear_pro_update_check_transient(){
		global $wpdb;
		$sql   = "DELETE a, b FROM $wpdb->options a, $wpdb->options b
        WHERE a.option_name LIKE %s
        AND a.option_name NOT LIKE %s
        AND b.option_name = CONCAT( '_site_transient_timeout_', SUBSTRING( a.option_name, 17 ) )";
		$wpdb->query($wpdb->prepare($sql, $wpdb->esc_like('_site_transient_mailmint-check_for_plugin_update') . '%', $wpdb->esc_like('_site_transient_timeout_') . '%'));
	}
}
