<?php
/**
 * Class WooCommerceNewOrder
 *
 * Handles the WooCommerce "woocommerce_new_order" hook.
 * This class is responsible for handling actions related to new orders.
 *
 * @package MailMintPro\Mint\Internal\AbandonedCart
 * @since 1.5.0
 */

namespace MailMintPro\Mint\Internal\AbandonedCart\Hooks\WooCommerce;

use MailMintPro\Mint\Internal\AbandonedCart\Helper\Common;
use MailMintPro\Mint\Internal\AbandonedCart\Helper\Model;

/**
 * Class WooCommerceNewOrder
 *
 * Handles the WooCommerce "woocommerce_new_order" hook.
 * This class is responsible for handling actions related to new orders.
 */
class WooCommerceNewOrder {

	/**
	 * WooCommerceNewOrder constructor.
	 *
	 * Initializes the WooCommerceNewOrder class by setting up the action hook for "woocommerce_new_order".
	 *
	 * @param string $key The key associated with the action hook.
	 *
	 * @since 1.5.0
	 */
	public function __construct( $key ) {
		add_action( $key, array( $this, 'handle_new_order' ), 10, 2 );
	}

	/**
	 * Handles the action triggered by the "woocommerce_new_order" hook.
	 *
	 * This method is called when a new order is created.
	 * It receives the order ID and the order instance as parameters.
	 *
	 * @param int    $order_id The ID of the new order.
	 * @param object $order    The order instance.
	 *
	 * @since 1.5.0
	 */
	public function handle_new_order( $order_id, $order ) {
		// Code logic to handle the action of a new order.
		$email = $order->get_billing_email();

		// No email associated with the order, nothing to handle.
		if ( empty( $email ) ) {
			return;
		}

		$cart_details = Model::get_cart_details_by_key( 'email', $email, 'pending' );
		// No cart details found, nothing to handle.
		if ( empty( $cart_details ) ) {
			return;
		}

		$created_at = ! empty( $cart_details['created_at'] ) ? $cart_details['created_at'] : '';
		$id         = ! empty( $cart_details['id'] ) ? $cart_details['id'] : '';

		// Cart is not within the wait time, no further action needed.
		if ( ! Common::is_within_wait_time_minutes( $created_at ) ) {
			return;
		}

		Model::delete( $id );
	}
}


