<?php
/**
 * Class MailMint
 *
 * Handles the Mail Mint abandoned cart related hooks.
 * This class is responsible for handling actions related to adding items to the cart.
 *
 * @package MailMintPro\Mint\Internal\AbandonedCart
 * @since 1.5.0
 */

namespace MailMintPro\Mint\Internal\AbandonedCart\Hooks\WooCommerce;

use MailMintPro\Mint\Internal\AbandonedCart\Helper\Common;
use MailMintPro\Mint\Internal\AbandonedCart\Helper\Model;
use MintMail\App\Internal\Automation\HelperFunctions;
use WC_Order;

/**
 * Class MailMint
 *
 * Handles the Mail Mint abandoned cart related hooks.
 */
class MailMint {

	/**
	 * Holds the details of the restored cart.
	 *
	 * @var array $restored_cart_detail The array containing the restored cart details.
	 * @access protected
	 * @since 1.5.0
	 */
	protected $restored_cart_detail = array();

	/**
	 * MailMint constructor.
	 *
	 * Initializes the MailMint class by setting up the action hook for Mail Mint abandoned cart related hooks.
	 *
	 * @param string $key The key associated with the action hook.
	 *
	 * @since 1.5.0
	 */
	public function __construct( $key ) {
		add_filter( 'mint_abandoned_cart_updated_after_automation_trigger', array( $this, 'handle_abandoned_cart_updated_after_automation_trigger' ), 10, 2 );
		add_action( 'wp', array( $this, 'handle_abandoned_cart_recovery' ), 10 );
		// prefill the checkout fields after the cart is restored.
		add_filter( 'woocommerce_billing_fields', array( $this, 'prefill_billing_fields_with_restored_data' ), 20 );
		add_filter( 'woocommerce_shipping_fields', array( $this, 'prefill_shipping_fields_with_restored_data' ), 20 );
		add_filter( 'mint_automation_email_body', array( $this, 'handle_replace_abandoned_carts_placeholder' ), 10, 2 );

		// Handle the WooCommerce checkout order processed event.
		add_action( 'woocommerce_checkout_order_processed', array( $this, 'handle_woocommerce_checkout_order_processed' ) );
		add_action( 'woocommerce_store_api_checkout_order_processed', array( $this, 'handle_woocommerce_checkout_order_processed' ) );

		add_filter( 'mint_test_email_preview', array( $this, 'replace_abandoned_carts_placeholder_on_test_email' ), 10, 1 );
		add_filter( 'mailmint_before_automation_step_process', array( $this, 'handle_mailmint_before_automation_step_process' ), 10, 1 );
		add_filter( 'mailmint_before_automation_sequence_process', array( $this, 'handle_mailmint_before_automation_sequence_process' ), 10, 1 );

		/**
		 * Conditionally adds a filter to modify the display of prices when a multi-currency plugin is active.
		 *
		 * Checks if the constant WOOMULTI_CURRENCY_VERSION is defined, and if true, adds a filter to the wc_price hook.
		 * The wc_price hook is a WooCommerce filter used for formatting prices.
		 * When the filter is executed, the maybe_multi_currency_converted_price method of the current class instance is called.
		 *
		 * @since 1.6.1
		 *
		 * @see maybe_multi_currency_converted_price
		 */
		if ( defined( 'WOOMULTI_CURRENCY_VERSION' ) ) {
			add_filter( 'wc_price', array( $this, 'maybe_multi_currency_converted_price' ), 9999, 5 );
		}

		add_filter( 'mailmint_default_checkout_nice_names', array( $this, 'adjust_checkout_nice_names_on_block' ) );
		add_action( 'woocommerce_after_checkout_form', array( $this, 'store_checkout_page_id_on_cookie' ) );
		add_action( 'woocommerce_blocks_enqueue_checkout_block_scripts_after', array( $this, 'store_checkout_page_id_on_cookie' ) );
	}

	/**
	 * Handles storing the checkout page ID on a cookie.
	 *
	 * This method is hooked into the 'woocommerce_after_checkout_form'
	 * and 'woocommerce_blocks_enqueue_checkout_block_scripts_after' actions.
	 *
	 * @return void
	 * @since 1.18.5
	 */
	public function store_checkout_page_id_on_cookie(){
		$page_id = get_the_ID();
		setcookie( 'mint_checkout_page_id', $page_id, time() + 3600, '/' );
	}

	/**
	 * Adjust the checkout nice names on block.
	 *
	 * This method is hooked into the mailmint_default_checkout_nice_names filter to customize
	 * the nice names of the checkout fields based on the block type.
	 *
	 * @param array $nice_names The array containing the nice names of the checkout fields.
	 *
	 * @return array The modified array of nice names for the checkout fields.
	 * @since 1.15.4
	 */
	public function adjust_checkout_nice_names_on_block( $nice_names ){
		if( Common::is_checkout_block() ) {
			$nice_names['billing-first_name'] = __( 'First Name', 'mailmint-pro' );
			$nice_names['billing-last_name'] = __( 'Last Name', 'mailmint-pro' );
			$nice_names['billing-address_1'] = __( 'Address Line 1', 'mailmint-pro' );
			$nice_names['billing-address_2'] = __( 'Address Line 2', 'mailmint-pro' );
			$nice_names['billing-city'] = __( 'City', 'mailmint-pro' );
			$nice_names['billing-postcode'] = __( 'Postcode', 'mailmint-pro' );
			$nice_names['billing-country'] = __( 'Country', 'mailmint-pro' );
			$nice_names['billing-state'] = __( 'State', 'mailmint-pro' );
			$nice_names['billing-phone'] = __( 'Phone', 'mailmint-pro' );
			$nice_names['shipping-first_name'] = __( 'First Name', 'mailmint-pro' );
			$nice_names['shipping-last_name'] = __( 'Last Name', 'mailmint-pro' );
			$nice_names['shipping-company'] = __( 'Company Name', 'mailmint-pro' );
			$nice_names['shipping-address_1'] = __( 'Address Line 1', 'mailmint-pro' );
			$nice_names['shipping-address_2'] = __( 'Address Line 2', 'mailmint-pro' );
			$nice_names['shipping-city'] = __( 'City', 'mailmint-pro' );
			$nice_names['shipping-postcode'] = __( 'Postcode', 'mailmint-pro' );
			$nice_names['shipping-country'] = __( 'Country', 'mailmint-pro' );
			$nice_names['shipping-state'] = __( 'State', 'mailmint-pro' );
		}
		return $nice_names;
	}

	/**
	 * Modify the display of prices when a multi-currency plugin is active.
	 *
	 * This method is hooked into the wc_price filter to customize the formatting of prices based on multi-currency conditions.
	 *
	 * @since 1.6.1
	 *
	 * @param string $return            The formatted price HTML string.
	 * @param float  $price             The price to be formatted.
	 * @param array  $args              The arguments passed to wc_price.
	 * @param float  $unformatted_price The unformatted price.
	 * @param float  $original_price    The original price before any modifications.
	 *
	 * @return string The modified HTML string for displaying the price.
	 */
	public function maybe_multi_currency_converted_price( $return, $price, $args, $unformatted_price, $original_price ) {
		$negative        = $price < 0;
		$formatted_price = ( $negative ? '-' : '' ) . sprintf( $args['price_format'], '<span class="woocommerce-Price-currencySymbol">' . get_woocommerce_currency_symbol( $args['currency'] ) . '</span>', $price );
		$return          = '<span class="woocommerce-Price-amount amount"><bdi>' . $formatted_price . '</bdi></span>';

		if ( $args['ex_tax_label'] && wc_tax_enabled() ) {
			$return .= ' <small class="woocommerce-Price-taxLabel tax_label">' . WC()->countries->ex_tax_or_vat() . '</small>';
		}
		return $return;
	}

	/**
	 * Handles the action after an abandoned cart is updated following an automation trigger.
	 *
	 * @param array $data       The data associated with the automation trigger.
	 * @param array $automation The automation details.
	 * @return bool             Returns true if the action is successfully handled, false otherwise.
	 *
	 * @since 1.5.0
	 */
	public function handle_abandoned_cart_updated_after_automation_trigger( $data, $automation ) {
		if ( !is_array( $data ) && empty( $data ) ) {
			return false;
		}
		if ( isset( $data['data']['session_key'] ) ) {
			$automation_id = isset( $automation['id'] ) ? $automation['id'] : '';

			// Ensure $automation_id is not empty before using it.
			if ( ! empty( $automation_id ) ) {
				$trigger_data          = HelperFunctions::get_next_step( $automation_id );
				$trigger_id            = isset( $trigger_data['step_id'] ) ? $trigger_data['step_id'] : '';
				$trigger_automation_id = isset( $trigger_data['automation_id'] ) ? $trigger_data['automation_id'] : null;

				$step_data   = HelperFunctions::get_step_data( $trigger_automation_id, $trigger_id );
				$items       = isset( $data['data']['items'] ) ? $data['data']['items'] : array();
				$product_ids = array();
				$terms       = array();
				foreach ( $items as $item_key => $item ) {
					$product_id    = !empty( $item[ 'variation_id' ] ) ? (int) $item[ 'variation_id' ] : $item[ 'product_id' ];
					$product_ids[] = $product_id;
					$terms[]       = get_the_terms( $product_id, 'product_cat' );
				}

				$is_automation_exist = $this->is_automation_exist( $data );
				if ( !$is_automation_exist ) {
					Model::update_status( 'automation_id', $automation_id, array( 'session_key' => $data['data']['session_key'] ) );
				} else {
					if ( !empty( $data['data']['abandoned_id'] ) ) {
						$this->update_manual_run_automation_ids( $data['data']['abandoned_id'], $automation_id );
					}
				}

				// Checking product type.
				$option_type = !empty( $step_data['settings']['product_settings']['option_type'] ) ? $step_data['settings']['product_settings']['option_type'] : 'choose-all';
				// This condition is need for old user .
				if ( !isset( $step_data['settings']['product_settings']['option_type'] ) && !empty( $products['settings']['product_settings']['products'] ) ) {
					return $this->check_product_exists( $step_data, $product_ids );
				}
				// Set condition for choose type.
				if ( 'choose-product' ===$option_type ) {
					return $this->check_product_exists( $step_data, $product_ids );
				}
				if ( 'choose-category' ===$option_type ) {
					return $this->check_category_exists( $step_data, $terms );
				}
				return true;
			}
		}
		// Return false if none of the conditions are met.
		return false;
	}
	/**
	 * Update the manual run automation IDs in the cart's meta data.
	 *
	 * This function adds a new automation ID to the existing list, removes duplicates,
	 * and updates the cart's meta data with the updated list.
	 *
	 * @param int    $abandoned_id            The ID of the abandoned cart.
	 * @param string $automation_id           The ID of the automation to add.
	 *
	 * @return void
	 * @since 1.5.5
	 */
	private function update_manual_run_automation_ids( $abandoned_id, $automation_id ) {
		// Get the existing manual run automation IDs from the cart's meta data.
		$get_meta = Model::get_cart_meta( $abandoned_id, 'manual_run_automation_ids' );

		// Add the new automation ID to the existing list.
		$get_meta[] = intval( $automation_id );

		// Remove duplicates by converting the array to a set and back to an array.
		$get_meta = array_unique( $get_meta );

		// Update the cart's meta data with the updated list.
		Model::update_cart_meta( $abandoned_id, 'manual_run_automation_ids', maybe_serialize( $get_meta ) );
	}

	/**
	 * Check if an automation exists for a given abandoned cart.
	 *
	 * This function checks if an automation exists for the provided abandoned cart ID.
	 *
	 * @param array $data The data array containing the abandoned cart information.
	 *
	 * @return bool True if an automation exists, false otherwise.
	 * @since 1.5.5
	 */
	private function is_automation_exist( $data ) {
		if ( !empty( $data['data']['abandoned_id'] ) ) {
			$cart = Model::get_cart_by_id( $data['data']['abandoned_id'] );
			if ( !empty( $cart ) && !is_null( $cart['automation_id'] ) ) {
				return true;
			}
			return false;
		}
		return false;
	}

	/**
	 * Checks if any value from the provided array exists in the 'value' keys of the products array.
	 *
	 * @param array $products The products array containing the product settings.
	 * @param array $values   The array of values to check against the product values.
	 *
	 * @return bool True if at least one value exists in the product array, false otherwise.
	 * @since 1.5.0
	 */
	private function check_product_exists( $products, $values ) {
		$product_values = array_column( $products['settings']['product_settings']['products'], 'value' );

		return empty( $product_values ) || count( array_intersect( $values, $product_values ) ) > 0;
	}
	/**
	 * Checks if any value from the provided array exists in the 'value' keys of the products array.
	 *
	 * @param array $step_data The products array containing the product settings.
	 * @param array $terms   The array of values to check against the product values.
	 *
	 * @return bool True if at least one value exists in the product array, false otherwise.
	 * @since 1.5.5
	 */
	private function check_category_exists( $step_data, $terms ) {
		$category_values = array_column( $step_data['settings']['product_settings']['category'], 'value' );
		$term_value      = array_merge( ...array_map( 'array_values', $terms ) );
		$term_id         = array_column( $term_value, 'term_id' );
		return empty( $category_values ) || count( array_intersect( $term_id, $category_values ) ) > 0;
	}

	/**
	 * Summary: Handles abandoned cart recovery.
	 *
	 * Description: This function handles the recovery of an abandoned cart. It retrieves the cart detail based on the token,
	 * restores the cart items, and redirects the user to the checkout or cart page based on the presence of checkout data.
	 *
	 * @access public
	 *
	 * @return bool
	 * @since 1.5.0
	 */
	public function handle_abandoned_cart_recovery() {
		if ( isset( $_GET['mint-cart-restore-test'] ) && 'yes' === $_GET['mint-cart-restore-test'] ) { //phpcs:ignore
			$this->redirect_to_test_checkout();
		}

		if ( ! isset( $_GET['mint-ac-token'] ) || empty( $_GET['mint-ac-token'] ) || wp_doing_ajax() || is_admin() ) { //phpcs:ignore
			return false;
		}

		$token = sanitize_text_field( $_GET['mint-ac-token'] ); //phpcs:ignore

		$cart_detail = Model::get_abandoned_cart_detail_by_token_and_status( $token, 'abandoned' );
		$cart_detail = Common::get_formatted_cart_detail( $cart_detail );
		$cart_object = WC()->cart;
		$wc_session  = WC()->session;

		$restored = $this->restore_abandoned_cart_items( $cart_detail, $cart_object, $wc_session );

		if ( false === $restored ) {
			$this->handle_cart_restore_fail();
		}

		// Retrieves the checkout data from the provided array.
		$checkout_data    = Common::get_checkout_data( $this->restored_cart_detail );
		$checkout_page_id = isset( $cart_detail['checkout_page_id'] ) ? $cart_detail['checkout_page_id'] : null;

		if ( is_array( $checkout_data ) && count( $checkout_data ) > 0 ) {
			$this->redirect_to_checkout( $checkout_page_id );
		}

		$this->redirect_to_cart();
	}

	/**
	 * Summary: Restores abandoned cart items.
	 *
	 * Description: Restores the abandoned cart items by adding them to the provided cart object.
	 *
	 * @access private
	 *
	 * @param array              $cart_detail  The abandoned cart detail to restore.
	 * @param WC_Cart Object     $cart_object  The cart object to add the items to.
	 * @param WC_Session_Handler $wc_session   The WooCommerce session object.
	 *
	 * @return bool True if the abandoned cart items are successfully restored, false otherwise.
	 * @since 1.5.0
	 */
	private function restore_abandoned_cart_items( $cart_detail, $cart_object, $wc_session ) {
		if ( empty( $cart_detail ) ) {
			return false;
		}

		$cart_items = isset( $cart_detail['items'] ) ? maybe_unserialize( $cart_detail['items'] ) : array();
		$email      = isset( $cart_detail['email'] ) ? $cart_detail['email'] : '';

		$checkout_data = isset( $cart_detail['checkout_data'] ) ? maybe_unserialize( $cart_detail['checkout_data'] ) : array();

		$billing_email = isset( $checkout_data['fields']['billing_email'] ) ? $checkout_data['fields']['billing_email'] : '';
		if( empty( $billing_email ) ){
			$checkout_data['fields']['billing_email'] = $email;
		}

		$cart_detail['checkout_data'] = maybe_serialize( $checkout_data );
		
		if ( ! is_array( $cart_items ) || empty( $cart_items ) ) {
			return false;
		}

		$cart_object->empty_cart();

		/**
		 * Summary: Triggers the 'mint_before_abandoned_cart_restored' action hook before restoring an abandoned cart.
		 *
		 * Description: This code section triggers the 'mint_before_abandoned_cart_restored' action hook,
		 * allowing developers to perform actions or modifications before an abandoned cart is restored.
		 *
		 * @param array $cart_detail The abandoned cart detail being restored.
		 * @since 1.5.0
		 */
		do_action( 'mint_before_abandoned_cart_restored', $cart_detail );

		// Adds restored customer data to the cart.
		if( Common::is_checkout_block() ){
			$this->add_restored_customer_data_to_cart($cart_detail, $cart_object);
		}

		// Adds restored items to the cart.
		$this->add_restored_item_to_cart( $cart_items, $cart_object );

		// Adds restored coupons to the cart.
		$coupons = isset( $cart_detail['abandoned_cart_meta']['coupons'] ) ? $cart_detail['abandoned_cart_meta']['coupons'] : array();
		$this->add_restored_coupon_to_cart( $coupons, $cart_object );

		// Clear show notices for added coupons or products.
		if ( ! is_null( $wc_session ) ) {
			$wc_session->set( 'wc_notices', array() );
		}

		// If any order_id is found for this abandoned row, then set this order_id in woocommerce session.
		$order_id = isset( $cart_detail['order_id'] ) ? $cart_detail['order_id'] : 0;
		if ( absint( $order_id ) > 0 ) {
			$order = wc_get_order( $order_id );
			if ( $order instanceof WC_Order && ! is_null( $wc_session ) && in_array( $order->get_status(), array( 'pending', 'wc-pending' ) ) ) { //phpcs:ignore
				$wc_session->set( 'order_awaiting_payment', $order_id );
			}
		}

		$this->restored_cart_detail = $cart_detail;
		if ( ! is_null( $wc_session ) ) {
			$wc_session->set( 'mint_restored_cart_detail', $cart_detail );
		}

		/**
		 * Summary: Triggers the 'mint_abandoned_cart_restored' action hook after an abandoned cart has been restored.
		 *
		 * Description: This code section triggers the 'mint_abandoned_cart_restored' action hook,
		 * allowing developers to perform actions or modifications after an abandoned cart has been successfully restored.
		 *
		 * @param array $cart_detail The abandoned cart detail being restored.
		 * @since 1.5.0
		 */
		do_action( 'mint_abandoned_cart_restored', $cart_detail );

		return true;
	}

	/**
	 * Summary: Adds restored items to the cart.
	 *
	 * Description: Adds the restored items to the cart by iterating through the provided cart items array.
	 *
	 * @access private
	 *
	 * @param array          $cart_items  An array containing the restored cart items.
	 * @param WC_Cart Object $cart_object The cart object to add the items to.
	 *
	 * @return void
	 * @since 1.5.0
	 */
	private function add_restored_item_to_cart( $cart_items, $cart_object ) {
		foreach ( $cart_items as $item_key => $item_data ) {
			$product_id     = 0;
			$quantity       = 0;
			$variation_id   = 0;
			$variation_data = array();

			if ( isset( $item_data['product_id'] ) ) {
				$product_id = $item_data['product_id'];
				unset( $item_data['product_id'] );
			}
			if ( isset( $item_data['quantity'] ) ) {
				$quantity = $item_data['quantity'];
				unset( $item_data['quantity'] );
			}
			if ( isset( $item_data['variation_id'] ) ) {
				$variation_id = $item_data['variation_id'];
				unset( $item_data['variation_id'] );
			}
			if ( isset( $item_data['variation'] ) ) {
				$variation_data = $item_data['variation'];
				unset( $item_data['variation'] );
			}

			$cart_object->add_to_cart( $product_id, $quantity, $variation_id, $variation_data, $item_data );
		}
	}

	/**
	 * Summary: Adds restored coupons to the cart object.
	 *
	 * Description: Adds the restored coupons to the provided cart object if they are not already applied.
	 *
	 * @access private
	 *
	 * @param array          $coupons      An array containing the restored coupon codes.
	 * @param WC_Cart Object $cart_object  The cart object to add the coupons to.
	 *
	 * @return void
	 * @since 1.5.0
	 */
	private function add_restored_coupon_to_cart( $coupons, $cart_object ) {
		if ( is_array( $coupons ) && count( $coupons ) > 0 ) {
			$coupons = array_keys( $coupons );
			foreach ( $coupons as $coupon_code ) {
				if ( ! $cart_object->has_discount( $coupon_code ) ) {
					$cart_object->add_discount( $coupon_code );
				}
			}
		}
	}

	/**
	 * Summary: Adds restored customer data to the cart object.
	 *
	 * Description: Adds the restored customer data to the provided cart object.
	 *
	 * @access private
	 *
	 * @param array          $cart_detail  The abandoned cart detail containing the customer data.
	 * @param WC_Cart Object $cart_object  The cart object to add the customer data to.
	 *
	 * @return void
	 * @since 1.5.0
	 */
	private function add_restored_customer_data_to_cart( $cart_detail, $cart_object ) {
		$checkout_data = isset( $cart_detail['checkout_data'] ) ? maybe_unserialize( $cart_detail['checkout_data'] ) : array();
		$fields        = isset( $checkout_data['fields'] ) ? $checkout_data['fields'] : array();

		if (isset($fields['email']) || isset($fields['billing_email'])) {
			$email = $fields['email'] ?? $fields['billing_email'];
			WC()->customer->set_billing_email(sanitize_email($email));
		}

		if (isset($fields['shipping-first_name'])) {
			WC()->customer->set_shipping_first_name(sanitize_text_field($fields['shipping-first_name']));
		}

		if (isset($fields['shipping-last_name'])) {
			WC()->customer->set_shipping_last_name(sanitize_text_field($fields['shipping-last_name']));
		}

		if (isset($fields['shipping-address_1'])) {
			WC()->customer->set_shipping_address_1(sanitize_text_field($fields['shipping-address_1']));
		}

		if (isset($fields['shipping-address_2'])) {
			WC()->customer->set_shipping_address_2(sanitize_text_field($fields['shipping-address_2']));
		}

		if (isset($fields['shipping-city'])) {
			WC()->customer->set_shipping_city(sanitize_text_field($fields['shipping-city']));
		}

		if (isset($fields['shipping-state'])) {
			WC()->customer->set_shipping_state(sanitize_text_field($fields['shipping-state']));
		}

		if (isset($fields['shipping-postcode'])) {
			WC()->customer->set_shipping_postcode(sanitize_text_field($fields['shipping-postcode']));
		}

		if (isset($fields['shipping-country'])) {
			WC()->customer->set_shipping_country(sanitize_text_field($fields['shipping-country']));
		}

		if (isset($fields['shipping-phone'])) {
			WC()->customer->set_billing_phone(sanitize_text_field($fields['shipping-phone']));
		}

		if (isset($fields['billing-first_name'])) {
			WC()->customer->set_billing_first_name(sanitize_text_field($fields['billing-first_name']));
		}

		if (isset($fields['billing-last_name'])) {
			WC()->customer->set_billing_last_name(sanitize_text_field($fields['billing-last_name']));
		}

		if (isset($fields['billing-address_1'])) {
			WC()->customer->set_billing_address_1(sanitize_text_field($fields['billing-address_1']));
		}

		if (isset($fields['billing-address_2'])) {
			WC()->customer->set_billing_address_2(sanitize_text_field($fields['billing-address_2']));
		}

		if (isset($fields['billing-city'])) {
			WC()->customer->set_billing_city(sanitize_text_field($fields['billing-city']));
		}

		if (isset($fields['billing-state'])) {
			WC()->customer->set_billing_state(sanitize_text_field($fields['billing-state']));
		}

		if (isset($fields['billing-postcode'])) {
			WC()->customer->set_billing_postcode(sanitize_text_field($fields['billing-postcode']));
		}

		if (isset($fields['billing-country'])) {
			WC()->customer->set_billing_country(sanitize_text_field($fields['billing-country']));
		}

		if (isset($fields['billing-phone'])) {
			WC()->customer->set_billing_phone(sanitize_text_field($fields['billing-phone']));
		}
	}

	/**
	 * Summary: Prefills billing fields with restored data.
	 *
	 * Description: This function pre-fills the billing fields with the restored data from the cart, if available.
	 *
	 * @access public
	 *
	 * @param array $address_fields An array containing the billing address fields.
	 *
	 * @return array The modified array of billing address fields.
	 * @since 1.5.0
	 */
	public function prefill_billing_fields_with_restored_data( $address_fields ) {
		$session = WC()->session;
		return Common::prefill_fields_with_restored_data( $address_fields, 'shipping', $session );
	}

	/**
	 * Summary: Prefills shipping fields with restored data.
	 *
	 * Description: This function pre-fills the shipping fields with the restored data from the cart, if available.
	 *
	 * @access public
	 *
	 * @param array $address_fields An array containing the shipping address fields.
	 *
	 * @return array The modified array of shipping address fields.
	 * @since 1.5.0
	 */
	public function prefill_shipping_fields_with_restored_data( $address_fields ) {
		$session = WC()->session;
		return Common::prefill_fields_with_restored_data( $address_fields, 'billing', $session );
	}

	/**
	 * Summary: Replaces the abandoned carts placeholders in the email body.
	 *
	 * Description: Replaces the placeholders for cart recovery URL and cart items in the email body with actual values.
	 *
	 * @access public
	 *
	 * @param string $email_body The email body.
	 * @param array  $data        An array containing the data for the email.
	 *
	 * @return string The updated email body with replaced placeholders.
	 * @since 1.5.0
	 */
	public function handle_replace_abandoned_carts_placeholder( $email_body, $data ) {
		// Get session key and automation ID from data array with null coalescing operator.
		$session_key   = isset( $data['data']['session_key'] ) ? $data['data']['session_key'] : '';
		$automation_id = isset( $data['automation_id'] ) ? $data['automation_id'] : '';
		$cart_details  = Model::get_cart_details_by_key( 'session_key', $session_key, 'abandoned' );
		$cart_details  = Common::get_abandoned_cart_totals( $cart_details );

		$checkout_id = Common::get_woocommerce_checkout_page_id();

		// Generate cart recovery URL and cart items table block.
		$cart_url    = Common::get_cart_recovery_url( $cart_details, $automation_id, $checkout_id );
		$table_block = Common::generate_cart_items_table_block_from_placeholder( $cart_details );
		$email_body  = Common::generate_cart_items_from_cart_block( $cart_details, $email_body );
		$email_body  = Common::generate_cart_items_from_cart_table_block( $cart_details, $email_body );

		// Replace cart.recovery_url and cart.items placeholders.
		$email_body = str_replace( '{{cart.recovery_url}}', $cart_url, $email_body );
		$email_body = str_replace( '{{cart.items}}', $table_block, $email_body );
		return $email_body;
	}

	/**
	 * Summary: Handles the failure to restore the cart.
	 *
	 * Description: This function displays a notice to the user indicating that the cart could not be restored,
	 * and redirects them to the cart page.
	 *
	 * @access private
	 *
	 * @return void
	 * @since 1.5.0
	 */
	private function handle_cart_restore_fail() {
		wc_add_notice( 'Your cart could not be restored, it may have expired.', 'notice' );

		$url = add_query_arg(
			array(
				'mint-cart-restored' => 'fail',
			),
			wc_get_page_permalink( 'cart' )
		);

		wp_safe_redirect( $url );
		exit;
	}

	/**
	 * Summary: Redirects to the checkout page after successful cart restoration.
	 *
	 * Description: This function redirects the user to the checkout page after successfully restoring the cart,
	 * and adds a query parameter to indicate the success.
	 *
	 * @param int $checkout_page_id The ID of the checkout page.
	 * @access private
	 *
	 * @return void
	 * @since 1.5.0
	 */
	private function redirect_to_checkout( $checkout_page_id ) {
		$url = get_permalink( $checkout_page_id ? $checkout_page_id : Common::get_woocommerce_checkout_page_id() );

		$url = add_query_arg(
			array(
				'mint-cart-restored' => 'success',
			),
			$url
		);

		wp_safe_redirect( $url );
		exit;
	}

	/**
	 * Summary: Redirects to the cart page after successful cart restoration.
	 *
	 * Description: This function redirects the user to the cart page after successfully restoring the cart,
	 * and adds a query parameter to indicate the success.
	 *
	 * @access private
	 *
	 * @return void
	 * @since 1.5.0
	 */
	private function redirect_to_cart() {
		$url = wc_get_page_permalink( 'cart' );

		$url = add_query_arg(
			array(
				'mint-cart-restored' => 'success',
			),
			$url
		);

		wp_safe_redirect( $url );
		exit;
	}

	/**
	 * Summary: Handles the WooCommerce checkout order processed event.
	 *
	 * Description: This function is triggered when a WooCommerce order is processed during checkout. It retrieves the restored cart detail from the session, updates the status of the recovered cart, and inserts the order ID as meta data.
	 *
	 * @access public
	 *
	 * @param int $order_id The ID of the processed order.
	 * @return void
	 * @since 1.5.0
	 */
	public function handle_woocommerce_checkout_order_processed( $order_id ) {
		$session = WC()->session;
		$data    = $session->get( 'mint_restored_cart_detail' );
		$token   = isset( $data['token'] ) ? $data['token'] : '';
		$id      = isset( $data['id'] ) ? $data['id'] : '';
		/**
		 * Execute the 'mailmint_before_abandoned_cart_recovered' action hook.
		 *
		 * This function triggers the 'mailmint_before_abandoned_cart_recovered' action,
		 * allowing other functions or plugins to perform additional tasks after an abandoned cart is recovered.
		 *
		 * @param mixed $data Optional. Additional data or parameters passed to the hooked functions. Default is null.
		 *                    Developers can use this parameter to pass custom data to the hooked functions.
		 * @return void
		 * @since 1.5.0
		 */
		do_action( 'mailmint_before_abandoned_cart_recovered', $data );
		$updated_status = Model::update_status(
			'status',
			'recovered',
			array(
				'id'    => $id,
				'token' => $token,
			)
		);
		if ( $updated_status ) {
			Model::insert_meta( $id, 'order_id', $order_id );
			/**
			 * Execute the 'mailmint_after_abandoned_cart_recovered' action hook.
			 *
			 * This function triggers the 'mailmint_after_abandoned_cart_recovered' action,
			 * allowing other functions or plugins to perform additional tasks after an abandoned cart is recovered.
			 *
			 * @param mixed $data Optional. Additional data or parameters passed to the hooked functions. Default is null.
			 *                    Developers can use this parameter to pass custom data to the hooked functions.
			 * @return void
			 * @since 1.5.0
			 */
			do_action( 'mailmint_after_abandoned_cart_recovered', $data );
		}
	}

	/**
	 * Summary: Replaces the abandoned carts placeholders with dummy data in the test email.
	 *
	 * Description: This function replaces the placeholders {{cart.recovery_url}} and {{cart.items}}
	 * with dummy recovery URL and cart item details respectively in the provided data.
	 *
	 * @access public
	 *
	 * @param string $data The test email content to replace placeholders in.
	 * @return string The modified test email content with replaced placeholders.
	 * @since 1.5.0
	 */
	public function replace_abandoned_carts_placeholder_on_test_email( $data ) {
		$cart_url    = Common::get_dummy_recovery_url();
		$cart_detail = Common::get_dummy_cart_detail();
		$cart_detail = Common::generate_cart_items_table_block_from_placeholder( $cart_detail );

		$data = str_replace(
			array( '{{cart.recovery_url}}', '{{cart.items}}' ),
			array( $cart_url, $cart_detail ),
			$data
		);
		return $data;
	}

	/**
	 * Summary: Redirects to the test checkout page with randomly added products.
	 *
	 * Description: This private method generates a list of random products and adds them to the WooCommerce cart.
	 *
	 * @access private
	 *
	 * @since 1.5.0
	 */
	private function redirect_to_test_checkout() {
		$args            = array(
			'posts_per_page' => 2,
			'orderby'        => 'rand',
			'post_type'      => 'product',
			'fields'         => 'ids',
		);
		$random_products = get_posts( $args );

		// Exit if no random products found.
		if ( ! is_array( $random_products ) || empty( $random_products ) ) {
			return;
		}

		WC()->cart->empty_cart();

		foreach ( $random_products as $product_id ) {
			$product = wc_get_product( $product_id );

			// Skip if product not found.
			if ( ! $product ) {
				continue;
			}

			WC()->cart->add_to_cart( $product_id, 1 );
		}

		$url = wc_get_page_permalink( 'checkout' );
		$url = add_query_arg(
			array(
				'mint-cart-restored' => 'success',
			),
			$url
		);

		// Clear show notices for added products.
		if ( ! is_null( WC()->session ) ) {
			WC()->session->set( 'wc_notices', array() );
		}

		wp_safe_redirect( wp_sanitize_redirect( esc_url_raw( $url ) ) );
		exit;
	}

	/**
	 * Callback function executed before an automation step is processed.
	 *
	 * @param array $data The data passed to the automation step.
	 * @return array The modified data or an empty array based on certain conditions.
	 * @since 1.5.0
	 */
	public function handle_mailmint_before_automation_step_process( $data ) {
		if ( empty( $data['data']['abandoned_id'] ) ) {
			return $data;
		}
		$abandoned_id  = !empty( $data['data']['abandoned_id'] ) ? $data['data']['abandoned_id'] : 0;
		$automation_id = !empty( $data['automation_id'] ) ? $data['automation_id'] : 0;

		$automation_data = Model::get_automation_data( $automation_id );
		if ( isset( $automation_data['trigger_name'] ) && 'wc_abandoned_cart_recovered' === $automation_data['trigger_name'] ) {
			return $data;
		}
		$cart_details = Model::get_cart_by_id( $abandoned_id );

		if ( !empty( $cart_details['status'] ) && 'recovered' === $cart_details['status'] ) {
			return array();
		}
		return $data;
	}
	/**
	 * Callback function executed before an automation step is processed.
	 *
	 * @param array $data The data passed to the automation step.
	 * @return array The modified data or an empty array based on certain conditions.
	 * @since 1.5.0
	 */
	public function handle_mailmint_before_automation_sequence_process( $data ) {
		if ( empty( array( 'abandoned_id' ) ) ) {
			return $data;
		}
		$abandoned_id  = !empty( $data['abandoned_id'] ) ? $data['abandoned_id'] : 0;
		$automation_id = !empty( $data['automation_id'] ) ? $data['automation_id'] : 0;

		$get_automation = Model::get_automation_data( $automation_id );
		if ( isset( $get_automation['trigger_name'] ) && 'wc_abandoned_cart_recovered' === $get_automation['trigger_name'] ) {
			return $data;
		}
		$cart_details = Model::get_cart_by_id( $abandoned_id );

		if ( !empty( $cart_details['status'] ) && 'recovered' === $cart_details['status'] ) {
			return array();
		}
		return $data;
	}
}

