<?php
/**
 * Class Common
 *
 * Provides common helper functions related to abandoned cart functionality.
 *
 * @package MailMintPro\Mint\Internal\AbandonedCart\Helper
 * @since 1.5.0
 */

namespace MailMintPro\Mint\Internal\AbandonedCart\Helper;

use MailMintPro\Mint\DataBase\Tables\AbandonedCart;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataBase\Tables\AutomationLogSchema;
use Mint\MRM\DataBase\Tables\AutomationSchema;
use MRM\Common\MrmCommon;

/**
 * Class Common
 *
 * Provides common helper functions related to abandoned cart functionality.
 */
class Model {
	/**
	 * Get the fully qualified abandoned carts table name.
	 *
	 * @return string The fully qualified abandoned carts table name.
	 * @since 1.5.0
	 */
	protected static function get_table_name() {
		global $wpdb;
		return $wpdb->prefix . AbandonedCart::$table_name;
	}

	/**
	 * Summary: Retrieves the meta table name.
	 *
	 * Description: Retrieves the table name for the meta data associated with the AbandonedCart class.
	 *
	 * @access protected
	 *
	 * @global wpdb $wpdb The WordPress database object.
	 *
	 * @return string The meta table name.
	 * @since 1.5.0
	 */
	protected static function get_meta_table_name() {
		global $wpdb;
		return $wpdb->prefix . AbandonedCart::$meta_table_name;
	}

	/**
	 * Retrieve cart details from the database based on a specific key-value pair and status.
	 *
	 * This function queries the database table for cart details that match the given key-value pair and status.
	 *
	 * @param string $key     The key to search for in the database table.
	 * @param string $value   The value to match against the specified key.
	 * @param string $status  The status of the cart (default: 'pending').
	 *
	 * @return array|false     An associative array of cart details if found, or false if no matching cart is found.
	 * @since 1.5.0
	 */
	public static function get_cart_details_by_key( $key, $value, $status = 'pending' ) {
		global $wpdb;
		$table  = $wpdb->prefix . AbandonedCart::$table_name;
		$result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE {$key} = %s AND status = %s", array( $value, $status ) ), ARRAY_A ); //phpcs:ignore
		if ( is_array( $result ) && !empty( $result ) ) {
			return $result;
		}
		return false;
	}

	/**
	 * Insert or update abandoned cart data.
	 *
	 * This function inserts or updates the abandoned cart data based on the provided $data array.
	 *
	 * @param array $data The abandoned cart data.
	 *
	 * @return bool
	 * @since 1.5.0
	 */
	public static function insert_or_update( $data ) {
		// Set initial values.
		$email         = isset( $data['email'] ) ? $data['email'] : '';
		$settings      = Common::get_abandoned_cart_settings();
		$disable_roles = isset($settings['disable_roles']) ? $settings['disable_roles'] : array();

		// Check if the user role is in the disabled roles list.
		if (! empty($disable_roles)) {
			$user = get_user_by('email', $email);
			if ($user) {
				$user_roles = $user->roles;

				foreach ($user_roles as $role) {
					if (in_array($role, $disable_roles, true)) {
						return false;
					}
				}
			}
		}

		if (isset($_COOKIE['mint_ab_cart_skip_track']) && $_COOKIE['mint_ab_cart_skip_track'] == 'yes') {
			return rest_ensure_response(
				array(
					'success' => true,
					'message' => __('Cart tracking has been opted out successfully.', 'mailmint-pro'),
				)
			);
		}

		$data = Common::get_abandoned_cart_totals( $data );

		// Check if cart details exist based on email.
		$cart_detail = self::get_cart_details_by_key_and_status( 'email', $email, array( 'pending' ) );

		if ( isset( $cart_detail['id'] ) ) {
			// Update existing cart.
			$data['status']     = $cart_detail['status'];
			$data['updated_at'] = current_time( 'mysql' );
			return self::update( $data, $cart_detail['id'] );
		} else {
			// Insert new cart.
			$session_key           = WC()->session->get_customer_id();
			$data['token']         = Common::create_abandoned_cart_token( 32 );
			$data['session_key']   = $session_key;
			$data['created_at']    = current_time( 'mysql' );
			$data['checkout_data'] = maybe_serialize( Common::prepare_checkout_data( $data ) );
			$abandoned_cart_id     = self::insert( $data );

			// set user data in action Scheduler.
			$enqueue_data =array(
				'data' =>array(
					'id'          => $abandoned_cart_id,
					'email'       => $email,
					'session_key' => $session_key,
				),
			);
			Common::enqueue_user_after_add_to_cart( $enqueue_data, $abandoned_cart_id );

			// set cart creation data in action Scheduler.
			$enqueue_data = array(
				'data' =>array(
					'id'          => $abandoned_cart_id,
					'email'       => $email,
					'session_key' => $session_key,
					'created_at'  => current_time( 'mysql' ),
				),
			);
			Common::enqueue_cart_creation_time( $enqueue_data, $abandoned_cart_id );
		}
	}

	/**
	 * Insert data into the database table.
	 *
	 * @param array $data  The data to be inserted into the table.
	 *
	 * @return false|int   False on failure to insert, or the number of rows affected on success.
	 * @since 1.5.0
	 */
	public static function insert( $data ) {
		if ( empty( $data ) || !is_array( $data ) ) {
			return false;
		}
		global $wpdb;
        $inserted =  $wpdb->insert( self::get_table_name(), $data ); //phpcs:ignore
		if ( $inserted ) {
			return $wpdb->insert_id;
		}
		return $inserted;
	}

	/**
	 * Update data in the database table.
	 *
	 * @param array $data  The data to be updated in the table.
	 * @param int   $id    The ID of the row to be updated.
	 *
	 * @return false|int   False on failure to update, or the number of rows affected on success.
	 * @since 1.5.0
	 */
	public static function update( $data, $id ) {
		global $wpdb;
		return $wpdb->update( //phpcs:ignore
			self::get_table_name(),
			$data,
			array(
				'ID' => $id,
			)
		);
	}
	/**
	 * Update the status of a record in the database table.
	 *
	 * @param string $column    The column name to update.
	 * @param mixed  $value     The new value for the column.
	 * @param array  $condition An array defining the conditions for updating the record.
	 *                          Example: ['id' => 123, 'status' => 'pending'].
	 * @return int|false The number of rows updated, or false on failure.
	 */
	public static function update_status( $column, $value, $condition ) {
		if ( !$column || !$value || !$condition || !is_array( $condition ) ) {
			return false;
		}
		global $wpdb;
		// Prepare the data array with the column and its new value.
		$data = array(
			$column => $value,
		);

		// Perform the database update operation.
		return $wpdb->update( //phpcs:ignore
			self::get_table_name(), // The table name.
			$data,                  // The data to be updated.
			$condition              // The conditions for updating the record.
		);
	}


	/**
	 * Delete a record from the database table by ID.
	 *
	 * @param int $id The ID of the record to be deleted.
	 * @return bool|int False if the ID is empty or the deletion fails. Otherwise, the number of rows deleted.
	 * @since 1.5.0
	 */
	public static function delete( $id ) {
		if ( empty( $id ) ) {
			return false;
		}

		global $wpdb;
		return $wpdb->delete( self::get_table_name(), array( 'ID' => $id ) ); //phpcs:ignore
	}

	/**
	 * Summary: Inserts a meta data for a cart.
	 *
	 * Description: Inserts a meta data with the specified key and value for the cart with the given ID.
	 *
	 * @access public
	 *
	 * @global wpdb $wpdb The WordPress database object.
	 *
	 * @param int    $cart_id The ID of the cart.
	 * @param string $key     The meta key.
	 * @param mixed  $value   The meta value.
	 *
	 * @return false|int The number of rows affected if the meta data is inserted successfully, false otherwise.
	 * @since 1.5.0
	 */
	public static function insert_meta( $cart_id, $key, $value ) {
		global $wpdb;

		$data = array(
			'meta_key'          => $key, //phpcs:ignore
			'meta_value'        => $value, //phpcs:ignore
			'abandoned_cart_id' => $cart_id,
		);

		return $wpdb->insert( self::get_meta_table_name(), $data ); //phpcs:ignore
	}

	/**
	 * Summary: Updates a meta data for a cart.
	 *
	 * Description: Updates the meta data with the specified key and value for the cart meta with the given ID.
	 *
	 * @access public
	 *
	 * @global wpdb $wpdb The WordPress database object.
	 *
	 * @param int    $meta_id The ID of the meta data.
	 * @param string $key     The meta key.
	 * @param mixed  $value   The meta value.
	 *
	 * @return false|int The number of rows affected if the meta data is updated successfully, false otherwise.
	 * @since 1.5.0
	 */
	public static function update_meta( $meta_id, $key, $value ) {
		// Check if $meta_id, $key, and $value are valid.
		if ( empty( $meta_id ) || empty( $key ) ) {
			return false;
		}

		global $wpdb;
		return $wpdb->update( //phpcs:ignore
			self::get_meta_table_name(),
			array( 'meta_value' => $value ), //phpcs:ignore
			array(
				'ID'       => $meta_id,
				'meta_key' => $key, //phpcs:ignore
			)
		);
	}

	/**
	 * Retrieves all abandoned data based on status.
	 *
	 * @param string $status     The comma-separated status string.
	 * @param int    $offset     The offset for pagination.
	 * @param int    $limit      The limit for pagination.
	 * @param string $search     The search term.
	 * @param string $order_by   The column to order by.
	 * @param string $order_type The order type (ASC or DESC).
	 *
	 * @return array The formatted abandoned data.
	 *
	 * @since 1.5.0
	 */
	public static function get_all_abandoned_data_by_status( $status = 'pending,abandoned', $offset = 0, $limit = 10, $search = '', $order_by = 'email', $order_type = 'DESC' ) {
		global $wpdb;
		$abandoned_cart      = $wpdb->prefix . 'mint_abandoned_carts';
		$abandoned_cart_meta = $wpdb->prefix . 'mint_abandoned_carts_meta';
		$search_terms        = null;
		if ( ! empty( $search ) ) {
			$search       = $wpdb->esc_like( $search );
			$search_terms = "AND $abandoned_cart.email LIKE '%%$search%%'";
		}
		$status_formatted = Common::format_status( $status );
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder
		if ( 'total' === $order_by ) {
			$order_by = $wpdb->prepare( 'CAST(%1$s.%2$s AS DECIMAL(10,2))', $abandoned_cart, $order_by );
		} else {
			$order_by = $wpdb->prepare( '%1$s.%2$s', $abandoned_cart, $order_by );
		}
		$query = $wpdb->prepare(
			"SELECT $abandoned_cart.*, GROUP_CONCAT($abandoned_cart_meta.meta_key) AS meta_key, GROUP_CONCAT($abandoned_cart_meta.meta_value) AS meta_value
                FROM $abandoned_cart
                LEFT JOIN $abandoned_cart_meta ON $abandoned_cart.id = $abandoned_cart_meta.abandoned_cart_id
                WHERE $abandoned_cart.status in ( $status_formatted ) $search_terms
                 GROUP BY $abandoned_cart.id
                ORDER BY $order_by $order_type
                LIMIT %d, %d",
			array( $offset, $limit )
		);
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// phpcs:enable WordPress.DB.PreparedSQLPlaceholders.UnquotedComplexPlaceholder
        $results            = $wpdb->get_results( $query, ARRAY_A ); //phpcs:ignore
		$get_formatted_data = self::formatted_abandoned_data( $results );
		return $get_formatted_data;
	}

	/**
	 * Formats the retrieved abandoned data.
	 *
	 * @param array $results The retrieved abandoned data.
	 *
	 * @return array The formatted abandoned data.
	 *
	 * @since 1.5.0
	 */
	public static function formatted_abandoned_data( $results ) {
		if ( !is_array( $results ) ) {
			return array();
		}
		$final_results = array();
		foreach ( $results as $row ) {
			$id = $row['id'];
			// If the id is not already present in the final results array, create a new entry.
			if ( !isset( $final_results[ $id ] ) ) {
				$final_results[ $id ] = array(
					'id'            => $id,
					'email'         => $row['email'],
					'user_id'       => $row['user_id'],
					'automation_id' => $row['automation_id'],
					'status'        => $row['status'],
					'items'         => maybe_unserialize( $row['items'] ), // No need to serialize here.
					'provider'      => $row['provider'],
					'total'         => $row['total'],
					'token'         => $row['token'],
					'session_key'   => $row['session_key'],
					'checkout_data' => maybe_unserialize( $row['checkout_data'] ), // No need to serialize here.
					'created_at'    => gmdate( 'F d, Y h:i a', strtotime( $row['created_at'] ) ),
					'updated_at'    => gmdate( 'F d, Y h:i a', strtotime( $row['updated_at'] ) ),
					'meta'          => array(),
				);
			}

			// Add the meta key-value pair to the corresponding entry in the final results array.
			if ( !empty( $row['meta_key'] ) && !empty( $row['meta_value'] ) ) {
				$meta_key   = explode( ',', $row['meta_key'] );
				$meta_value = explode( ',', $row['meta_value'] );
				$keys       = array_map( 'trim', $meta_key );
				$values     = array_map( 'trim', $meta_value );
				$result     = array();
				foreach ( $keys as $index => $key ) {
					$result[ $key ] = $values[ $index ];
				}
				$final_results[ $id ]['meta'] = $result;
			}
		}
		return array_values( $final_results );
	}


	/**
	 * Formats the retrieved abandoned cart data.
	 *
	 * @param array $get_recoverable The retrieved abandoned cart data.
	 *
	 * @return array The formatted abandoned cart data.
	 *
	 * @since 1.5.0
	 */
	public static function get_formatted_result( $get_recoverable ) {
		$result = array();
		if ( !is_array( $get_recoverable ) ) {
			return $result;
		}
		foreach ( $get_recoverable as $item ) {
			$result[] = array(
				'id'                        => ! is_null( $item['id'] ) ? $item['id'] : 0,
				'automation_id'             => ! is_null( $item['automation_id'] ) ? $item['automation_id'] : 0,
				'manual_run_automation_ids' => !empty( $item['meta']['manual_run_automation_ids'] ) ? maybe_unserialize( $item['meta']['manual_run_automation_ids'] ) : array(),
				'email'                     => ! is_null( $item['email'] ) ? $item['email'] : '',
				'preview'                   => Common::get_preview_data( $item, $item['checkout_data'] ),
				'created_at'                => $item['created_at'],
				'status'                    => ! is_null( $item['status'] ) ? $item['status'] : '',
				'items'                     => Common::get_items( $item['items'] ),
				'total'                     => ! is_null( $item['total'] ) ? $item['total'] : 0,
				'currency'                  => Common::get_currency(),
				'order'                     => Common::get_order_id( $item ),
				'user_id'                   => ! empty( $item['user_id'] ) ? $item['user_id'] : 0,
				'mint_contact_id'           => ContactModel::get_id_by_email( ! empty( $item['email'] ) ? $item['email'] : '' ),
				'checkout_data'             => isset( $item['checkout_data'] ) ? $item['checkout_data'] : '',
			);
		}
		return $result;
	}
	/**
	 * Retrieves all abandoned cart data based on status and parameters.
	 *
	 * @param array  $params The parameters for retrieving abandoned cart data.
	 * @param string $status The status of abandoned carts to retrieve.
	 *
	 * @return array The retrieved abandoned cart data.
	 *
	 * @since 1.5.0
	 */
	public static function get_all_abandoned_cart( $params, $status ) {
		$page     = isset( $params['page'] ) ? absint( $params['page'] ) : 1;
		$per_page = isset( $params['per-page'] ) ? absint( $params['per-page'] ) : 25;
		$offset   = ( $page - 1 ) * $per_page;

		$order_by   = isset( $params['order-by'] ) ? strtolower( $params['order-by'] ) : 'email';
		$order_type = isset( $params['order-type'] ) ? strtolower( $params['order-type'] ) : 'asc';

		// valid order by fields and types.
		$allowed_order_by_fields = array( 'email', 'created_at', 'total' );
		$allowed_order_by_types  = array( 'asc', 'desc' );

		// validate order by fields or use default otherwise.
		$order_by   = in_array( $order_by, $allowed_order_by_fields, true ) ? $order_by : 'email';
		$order_type = in_array( $order_type, $allowed_order_by_types, true ) ? $order_type : 'asc';

		// Form Search keyword.
		$search         = isset( $params['search'] ) ? sanitize_text_field( $params['search'] ) : '';
		$get_cart_data  = self::get_all_abandoned_data_by_status( $status, $offset, $per_page, $search, $order_by, $order_type );
		$abandoned_data = self::get_formatted_result( $get_cart_data );
		$total          = Common::get_count_abandoned_cart_data();
		$result         = array(
			'total'          => $total,
			'current_page'   => $page,
			'abandoned_data' => $abandoned_data,
		);
		return $result;
	}


	/**
	 * Retrieves the count of abandoned cart data for different statuses.
	 *
	 * @return array An array containing the counts for different abandoned cart statuses.
	 * @since 1.5.0
	 */
	public static function get_abandoned_cart_status_count() {
		global $wpdb;
		$abandoned_cart = $wpdb->prefix . 'mint_abandoned_carts';
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$get_total = $wpdb->get_row( //phpcs:ignore
			$wpdb->prepare(
				"SELECT COUNT(CASE WHEN status = %s THEN 1 END) AS abandoned,
            COUNT(CASE WHEN status = %s THEN 1 END) AS pending,
            COUNT(CASE WHEN status = %s THEN 1 END) AS recovered,
            COUNT(CASE WHEN status = %s THEN 1 END) AS lost
            FROM $abandoned_cart",
				'abandoned',
				'pending',
				'recovered',
				'lost'
			),
			ARRAY_A
		);
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		if ( !empty( $get_total ) ) {
			return $get_total;
		}
		return array();
	}

	/**
	 * Delete multiple Abandoned
	 *
	 * @param array $abandoned_ids abandoned id.
	 *
	 * @return bool
	 * @since 1.5.0
	 */
	public static function destroy_all( $abandoned_ids ) {
		global $wpdb;
		$abandoned_cart      = $wpdb->prefix . 'mint_abandoned_carts';
		$abandoned_cart_meta = $wpdb->prefix . 'mint_abandoned_carts_meta';
		if ( is_array( $abandoned_ids ) && count( $abandoned_ids ) > 0 ) {
			$placeholders = implode( ', ', array_fill( 0, count( $abandoned_ids ), '%d' ) );
			$query                = $wpdb->prepare( "DELETE FROM $abandoned_cart WHERE id IN($placeholders)", $abandoned_ids ); //phpcs:ignore
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$delete_abandoned_ids = $wpdb->query( $query ); //phpcs:ignore
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			if ( $delete_abandoned_ids ) {
				$query = $wpdb->prepare( "DELETE FROM $abandoned_cart_meta WHERE abandoned_cart_id IN($placeholders)", $abandoned_ids ); //phpcs:ignore
                // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$wpdb->query( $query ); //phpcs:ignore
                // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			}
			return $delete_abandoned_ids;
		}
		return false;
	}


	/**
	 * Summary: Retrieves cart details based on key, value, and statuses.
	 *
	 * Description: Retrieves the cart details from the database based on the specified key, value, and statuses.
	 *
	 * @access public
	 *
	 * @param string $key         The key to match in the query.
	 * @param mixed  $value       The value to match against the provided key.
	 * @param array  $statuses    Optional. An array of statuses to match.
	 *
	 * @return array|false Returns an associative array containing the cart details if found, or false if not found.
	 * @since 1.5.0
	 */
	public static function get_cart_details_by_key_and_status( $key, $value, $statuses = array() ) {
		global $wpdb;
		$table = $wpdb->prefix . AbandonedCart::$table_name;

		$status_conditions = empty( $statuses ) ? '1=1' : implode( ' OR ', array_fill( 0, count( $statuses ), 'status = %s' ) );
		$status_values     = array_merge( array( $value ), $statuses );

		$query = $wpdb->prepare(
			"SELECT * FROM {$table} WHERE {$key} = %s AND ({$status_conditions})", //phpcs:ignore
			$status_values
		);

		$result = $wpdb->get_row( $query, ARRAY_A ); //phpcs:ignore

		return $result ? $result : false;
	}

	/**
	 * Summary: Updates the meta data for a cart.
	 *
	 * Description: Updates the meta data with the specified key and value for the cart with the given ID.
	 *
	 * @access public
	 *
	 * @param int    $cart_id    The ID of the cart.
	 * @param string $meta_key   The meta key.
	 * @param mixed  $meta_value The meta value.
	 *
	 * @return bool|int The meta ID if the meta data is updated successfully, false otherwise.
	 * @since 1.5.0
	 */
	public static function update_cart_meta( $cart_id, $meta_key, $meta_value ) {
		$meta_id = self::is_cart_meta_exist( $cart_id, $meta_key );
		if ( ! $meta_id ) {
			return self::insert_meta( $cart_id, $meta_key, $meta_value );
		}

		return self::update_meta( $meta_id, $meta_key, $meta_value );
	}

	/**
	 * Summary: Checks if a cart meta exists.
	 *
	 * Description: Checks if a cart meta with the specified meta key exists for the given cart ID.
	 *
	 * @access public
	 *
	 * @global wpdb $wpdb The WordPress database object.
	 *
	 * @param int    $cart_id  The ID of the cart.
	 * @param string $meta_key The meta key.
	 *
	 * @return bool|int The meta ID if the cart meta exists, false otherwise.
	 * @since 1.5.0
	 */
	public static function is_cart_meta_exist( $cart_id, $meta_key ) {
		global $wpdb;
		$table   = $wpdb->prefix . AbandonedCart::$meta_table_name;
		$meta_id = $wpdb->get_col( $wpdb->prepare( 'SELECT id FROM %1s WHERE meta_key = %s AND abandoned_cart_id = %d', $table, $meta_key, $cart_id ) ); //phpcs:ignore
		return $meta_id ? $meta_id[0] : false;
	}

	/**
	 * Summary: Get cart meta data.
	 *
	 * Description: Checks if a cart meta with the specified meta key exists for the given cart ID.
	 *
	 * @access public
	 *
	 * @global wpdb $wpdb The WordPress database object.
	 *
	 * @param int    $cart_id  The ID of the cart.
	 * @param string $meta_key The meta key.
	 *
	 * @return array The meta ID if the cart meta exists, false otherwise.
	 * @since 1.5.5
	 */
	public static function get_cart_meta( $cart_id, $meta_key ) {
		global $wpdb;
		$table     = $wpdb->prefix . AbandonedCart::$meta_table_name;
		$meta_data = $wpdb->get_row( $wpdb->prepare( 'SELECT meta_value FROM %1s WHERE meta_key = %s AND abandoned_cart_id = %d', $table, $meta_key, $cart_id ),ARRAY_A ); //phpcs:ignore
		if ( !empty( $meta_data['meta_value'] ) ) {
			return maybe_unserialize( $meta_data['meta_value'] );
		}
		return array();
	}

	/**
	 * Summary: Retrieves abandoned cart details by token and status.
	 *
	 * Description: Retrieves the abandoned cart details from the database based on the specified token and status.
	 *
	 * @access public
	 *
	 * @global wpdb $wpdb The WordPress database object.
	 *
	 * @param string $token  The token of the abandoned cart.
	 * @param string $status The status of the abandoned cart.
	 *
	 * @return array An array containing the abandoned cart details.
	 * @since 1.5.0
	 */
	public static function get_abandoned_cart_detail_by_token_and_status( $token, $status ) {
		global $wpdb;

		// Define table names.
		$abandoned_carts_table = $wpdb->prefix . AbandonedCart::$table_name;

		$abandoned_carts_meta_table = $wpdb->prefix . AbandonedCart::$meta_table_name;

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		// Prepare and execute the query.
		$query = $wpdb->prepare(
			"
							SELECT carts.*, meta.meta_key, meta.meta_value
							FROM {$abandoned_carts_table} AS carts
							LEFT JOIN {$abandoned_carts_meta_table} AS meta ON carts.id = meta.abandoned_cart_id
							WHERE carts.token = %s AND carts.status = %s 
						",
			$token,
			$status
		);
		// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return $wpdb->get_results( $query, ARRAY_A ); //phpcs:ignore
	}

	/**
	 * Retrieves automation logs by automation ID and email with pagination.
	 *
	 * @param int    $automation_id The ID of the automation.
	 * @param string $email         The email to match.
	 * @param int    $offset        The offset for pagination.
	 * @param int    $limit         The number of logs to retrieve.
	 * @return array The automation log data matching the criteria.
	 * @since 1.5.0
	 */
	public static function get_automation_log_by_id_and_email( $automation_id, $email, $offset = 0, $limit = 10 ) {
		global $wpdb;
		$table_name = $wpdb->prefix . AutomationLogSchema::$table_name;
		$query      = $wpdb->prepare(
			'SELECT * FROM %1s WHERE automation_id = %d AND email = %s ORDER BY id DESC LIMIT %d offset %d', //phpcs:ignore
			array( $table_name, $automation_id, $email, $limit, $offset )
		);
		$result     = $wpdb->get_results( $query, ARRAY_A ); //phpcs:ignore

		if ( !empty( $result ) ) {
			return $result;
		}
		return array();
	}

	/**
	 * Retrieves the total count of automation logs by automation ID and email.
	 *
	 * @param int    $automation_id The ID of the automation.
	 * @param string $email         The email to match.
	 * @return int The total count of automation logs matching the criteria.
	 * @since 1.5.0
	 */
	public static function get_automation_log_count( $automation_id, $email ) {
		global $wpdb;
		$table_name = $wpdb->prefix . AutomationLogSchema::$table_name;

		$count_query = $wpdb->prepare(
			'SELECT COUNT(*) AS total FROM %1s WHERE automation_id = %d AND email = %s', //phpcs:ignore
			array( $table_name, $automation_id, $email )
		);
		$total_count = $wpdb->get_var( $count_query ); //phpcs:ignore

		return $total_count;
	}

	/**
	 * Retrieves all abandoned automations based on the provided parameters with pagination.
	 *
	 * @param array $params The parameters for retrieving abandoned automations.
	 * @return array The abandoned automations data including pagination information.
	 * @since 1.5.0
	 */
	public static function get_all_abandoned_automations( $params ) {
		$page          = !empty( $params['page'] ) ? absint( $params['page'] ) : 1;
		$per_page      = !empty( $params['per-page'] ) ? absint( $params['per-page'] ) : 10;
		$automation_id = !empty( $params['automation_id'] ) ? $params['automation_id'] : 0;
		$email         = !empty( $params['email'] ) ? $params['email'] : '';
		$offset        = ( $page - 1 ) * $per_page;

		if ( !$automation_id || !$email ) {
			return array();
		}

		$log_data   = self::get_automation_log_by_id_and_email( $automation_id, $email, $offset, $per_page );
		$total_logs = self::get_automation_log_count( $automation_id, $email );

		return array(
			'current_page' => $page,
			'per_page'     => $per_page,
			'total'        => $total_logs,
			'data'         => Common::formatted_automation_log( $log_data ),
		);
	}

	/**
	 * Retrieves the abandoned cart data by ID.
	 *
	 * @param int $abandoned_id The ID of the abandoned cart.
	 * @return array|false The abandoned cart data if found, or false if not found.
	 * @since 1.5.0
	 */
	public static function get_cart_by_id( $abandoned_id ) {
		if ( !$abandoned_id ) {
			return array();
		}
		global $wpdb;
		$table  = $wpdb->prefix . AbandonedCart::$table_name;
        $result = $wpdb->get_row( $wpdb->prepare( "SELECT id,email,automation_id,status FROM {$table} WHERE id = %d", array( $abandoned_id, ) ), ARRAY_A ); //phpcs:ignore
		if ( is_array( $result ) && !empty( $result ) ) {
			return $result;
		}
		return false;
	}

	/**
	 * Retrieves the abandoned cart report data.
	 *
	 * @param string $filter The filter to be applied (e.g., 'all', 'today', 'last_7_days').
	 * @return array The abandoned cart report data.
	 * @since 1.5.0
	 */
	public static function get_abandoned_cart_report( $filter = 'all' ) {
		$abandoned_table = self::get_table_name();

		$cart_total                      = self::get_cart_total_report( $abandoned_table, $filter );
		$cart_total['potential_revenue'] = isset( $cart_total['potential_revenue'] ) ? MrmCommon::price_format_with_wc_currency( $cart_total['potential_revenue'] ) : MrmCommon::price_format_with_wc_currency( 0 );
		$cart_total['recovered_revenue'] = isset( $cart_total['recovered_revenue'] ) ? MrmCommon::price_format_with_wc_currency( $cart_total['recovered_revenue'] ) : MrmCommon::price_format_with_wc_currency( 0 );

		$revenue_bar_chart  = self::get_revenue_for_bar_chart( $filter );
		$recovery_rate      = self::get_recovery_rate( $filter );
		$revenue_line_chart = self::get_revenue_for_line_chart( $filter );

		return array(
			'cart_total'         => $cart_total,
			'currency'           => Common::get_currency(),
			'revenue_bar_chart'  => $revenue_bar_chart,
			'recovery_rate'      => $recovery_rate,
			'revenue_line_chart' => $revenue_line_chart,
		);
	}
	/**
	 * Retrieves the revenue recovery rate.
	 *
	 * @param string $filter The filter to be applied (e.g., 'all', 'today', 'last_7_days').
	 * @return float The revenue recovery rate as a percentage.
	 * @since 1.5.0
	 */
	public static function get_recovery_rate( $filter ) {
		$abandoned_table = self::get_table_name();
		$get_total       = self::get_cart_total_report( $abandoned_table, $filter );

		$abandoned = isset( $get_total['abandoned'] ) ? intval( $get_total['abandoned'] ) : 0;
		$pending   = isset( $get_total['pending'] ) ? intval( $get_total['pending'] ) : 0;
		$recovered = isset( $get_total['recovered'] ) ? intval( $get_total['recovered'] ) : 0;
		$lost      = isset( $get_total['lost'] ) ? intval( $get_total['lost'] ) : 0;

		$total_abandoned = intval( $abandoned + $pending + $lost );

		$recovery_rate = 0;

		if ( 0 === $recovered ) {
			return 0;
		}

		$total_abandoned += $recovered;
		$recovery_rate    = number_format( ( $recovered / $total_abandoned ) * 100, 2 );
		return $recovery_rate;
	}
	/**
	 * Retrieves revenue data for the bar chart based on the provided filter.
	 *
	 * @param string $filter The filter to be applied (e.g., 'all', 'today', 'last_7_days').
	 * @return array The revenue data for the bar chart.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_bar_chart( $filter ) {
		$get_revenue  = 'get_revenue_for_bar_chart_' . strtolower( $filter );
		$revenue_data = self::$get_revenue();
		$max_revenue  = array_reduce(
			$revenue_data,
			function ( $carry, $revenues ) {
				$potential_revenue = isset( $revenues['potential_revenue'] ) ? $revenues['potential_revenue'] : 0;
				$recovered_revenue = isset( $revenues['recovered_revenue'] ) ? $revenues['recovered_revenue'] : 0;
				return max( $carry, $potential_revenue, $recovered_revenue );
			},
			PHP_INT_MIN
		);

		return array(
			'data'        => $revenue_data,
			'max_revenue' => $max_revenue,
		);
	}
	/**
	 * Get revenue data for the line chart based on the provided filter.
	 *
	 * @param string $filter The filter to be applied ('all', 'monthly', 'weekly', or 'yearly').
	 * @return array An array containing abandoned, recovered, and lost revenue data for the line chart.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_line_chart( $filter ) {
		$get_revenue  = 'get_revenue_for_line_chart_' . strtolower( $filter );
		$revenue_data = self::$get_revenue();
		$abandoned    = !empty( $revenue_data['abandoned'] ) ? $revenue_data['abandoned'] : array();
		$recovered    = !empty( $revenue_data['recovered'] ) ? $revenue_data['recovered'] : array();
		$lost         = !empty( $revenue_data['lost'] ) ? $revenue_data['lost'] : array();
		$max_value    = max(
			max( array_values( $abandoned ) ),
			max( array_values( $recovered ) ),
			max( array_values( $lost ) )
		);
		return array(
			'labels'      => array_keys( $abandoned ),
			'max_revenue' => $max_value,

			'abandoned'   => array(
				'values' => array_values( $abandoned ),
			),
			'recovered'   => array(
				'values' => array_values( $recovered ),
			),
			'lost'        => array(
				'values' => array_values( $lost ),
			),
		);
	}

	/**
	 * Retrieves the cart total report based on the provided filter.
	 *
	 * @param string $table_name The name of the table to query.
	 * @param string $filter The filter to be applied (e.g., 'all', 'today', 'last_7_days').
	 * @return array The cart total report data.
	 * @since 1.5.0
	 */
	public static function get_cart_total_report( $table_name, $filter ) {
		global $wpdb;
		$condition = Common::get_where_query( $filter );
        $get_total = $wpdb->get_row(  //phpcs:ignore
			$wpdb->prepare(
				'SELECT
            COUNT(CASE WHEN status = %s THEN 1 END) AS abandoned,
            COUNT(CASE WHEN status = %s THEN 1 END) AS pending,
            COUNT(CASE WHEN status = %s THEN 1 END) AS recovered,
            COUNT(CASE WHEN status = %s THEN 1 END) AS lost,
            SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS potential_revenue,
            SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered_revenue
        FROM %1s WHERE %1s', array( 'abandoned', 'pending', 'recovered', 'lost', 'abandoned', 'recovered', $table_name, $condition )), ARRAY_A);  //phpcs:ignore

		if ( !empty( $get_total ) ) {
			return $get_total;
		}
		return array();
	}
	/**
	 * Retrieves revenue data for the bar chart for all months.
	 *
	 * @return array The revenue data for the bar chart with months labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_line_chart_all() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();

		$query  = "SELECT DATE_FORMAT(created_at, '%b') AS label";
		$query .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS abandoned';
		$query .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered';
		$query .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS lost';
		$query .= ' FROM %1s ';
		$query .= 'WHERE created_at BETWEEN DATE_SUB(NOW(), INTERVAL 5 YEAR) AND DATE_ADD(NOW(), INTERVAL 1 YEAR) ';
		$query .= "GROUP BY DATE_FORMAT(created_at, '%b') ";
		$query .= "ORDER BY DATE_FORMAT(created_at, '%b') ASC";
        $result = $wpdb->get_results( $wpdb->prepare( $query, array(  'abandoned', 'recovered','lost' ,$abandoned_cat_table) ), ARRAY_A ); //phpcs:ignore
		$months = Common::get_months_array();
		return Common::get_formated_line_chart( $result, $months );
	}

	/**
	 * Retrieves revenue data for the bar chart for the current week.
	 *
	 * @return array The revenue data for the bar chart with daily labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_line_chart_weekly() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();
		$week_start_end      = get_weekstartend( current_time( 'mysql' ), get_option( 'start_of_week', 1 ) );
		if ( ! empty( $week_start_end[ 'start' ] ) && ! empty( $week_start_end[ 'end' ] ) ) {
			$start_of_week = date_i18n( 'Y-m-d', $week_start_end['start'] );
			$end_of_week   = date_i18n( 'Y-m-d', $week_start_end['end'] );

			$query     = "SELECT DATE_FORMAT(created_at, '%b %e') AS label";
            $query .= $wpdb->prepare( ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS abandoned', array( 'abandoned' ) ); //phpcs:ignore
            $query .= $wpdb->prepare( ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered', array( 'recovered' ) ); //phpcs:ignore
            $query .= $wpdb->prepare( ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS lost', array( 'lost' ) ); //phpcs:ignore
            $query .= $wpdb->prepare( " FROM %1s ", array( $abandoned_cat_table ) ); //phpcs:ignore
			$query    .= "WHERE DATE_FORMAT(created_at, '%Y-%m-%d') >= ";
            $query .= $wpdb->prepare( '%s ', array( $start_of_week ) ); //phpcs:ignore
			$query    .= "AND DATE_FORMAT(created_at, '%Y-%m-%d') <= ";
            $query .= $wpdb->prepare( '%s ', array( $end_of_week ) ); //phpcs:ignore
			$query    .= "GROUP BY DATE_FORMAT(created_at, '%b %e') ";
			$query    .= "ORDER BY DATE_FORMAT(created_at, '%c %e') ASC";
            $result         = $wpdb->get_results( $query, ARRAY_A ); //phpcs:ignore
			$week_days = Common::get_week_days( $start_of_week );
			return Common::get_formated_line_chart( $result, $week_days );
		}
		return array();
	}
	/**
	 * Retrieves revenue data for the bar chart for the current month.
	 *
	 * @return array The revenue data for the bar chart with daily labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_line_chart_monthly() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();

		$query        = "SELECT DATE_FORMAT(created_at,  '%b %e') AS label";
		$query       .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS abandoned';
		$query       .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered';
		$query       .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS lost ';
		$query       .= ' FROM %1s ';
		$query       .= 'WHERE (EXTRACT(YEAR_MONTH FROM created_at) = EXTRACT(YEAR_MONTH FROM now())) ';
		$query       .= "GROUP BY DATE_FORMAT(created_at,  '%b %e') ";
		$query       .= "ORDER BY DATE_FORMAT(created_at,  '%b %e') ASC";
        $result = $wpdb->get_results( $wpdb->prepare( $query, array(  'abandoned', 'recovered', 'lost' ,$abandoned_cat_table) ), ARRAY_A ); //phpcs:ignore
		$monthly_days = Common::get_monthly_days();
		return Common::get_formated_line_chart( $result, $monthly_days );
	}
	/**
	 * Retrieves revenue data for the bar chart for the current year.
	 *
	 * @return array The revenue data for the bar chart with monthly labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_line_chart_yearly() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();

		$query  = "SELECT DATE_FORMAT(created_at, '%b') AS label";
		$query .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS abandoned';
		$query .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered';
		$query .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS lost ';
		$query .= ' FROM %1s ';
		$query .= 'WHERE YEAR(created_at) = YEAR(now()) ';
		$query .= "GROUP BY DATE_FORMAT(created_at, '%b') ";
		$query .= "ORDER BY DATE_FORMAT(created_at, '%b') ASC";
        $result = $wpdb->get_results( $wpdb->prepare( $query, array(  'abandoned', 'recovered', 'lost' ,$abandoned_cat_table) ), ARRAY_A ); //phpcs:ignore
		$months = Common::get_months_array();
		return Common::get_formated_line_chart( $result, $months );
	}

	/**
	 * Retrieves revenue data for the bar chart for all months.
	 *
	 * @return array The revenue data for the bar chart with months labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_bar_chart_all() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();

		$query          = "SELECT DATE_FORMAT(created_at, '%b') AS label";
		$query         .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS potential_revenue';
		$query         .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered_revenue';
		$query         .= ' FROM %1s ';
		$query         .= 'WHERE created_at BETWEEN DATE_SUB(NOW(), INTERVAL 5 YEAR) AND DATE_ADD(NOW(), INTERVAL 1 YEAR) ';
		$query         .= "GROUP BY DATE_FORMAT(created_at, '%b') ";
		$query         .= "ORDER BY DATE_FORMAT(created_at, '%b') ASC";
        $result = $wpdb->get_results( $wpdb->prepare( $query, array(  'abandoned', 'recovered' ,$abandoned_cat_table) ), ARRAY_A ); //phpcs:ignore
		$formatted_data = Common::format_data_with_label( $result );
		$months         = Common::get_months_multi_dimensional_array();
		return array_merge( $months, $formatted_data );
	}

	/**
	 * Retrieves revenue data for the bar chart for the current week.
	 *
	 * @return array The revenue data for the bar chart with daily labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_bar_chart_weekly() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();
		$week_start_end      = get_weekstartend( current_time( 'mysql' ), get_option( 'start_of_week', 1 ) );
		if ( ! empty( $week_start_end[ 'start' ] ) && ! empty( $week_start_end[ 'end' ] ) ) {
			$start_of_week = date_i18n( 'Y-m-d', $week_start_end['start'] );
			$end_of_week   = date_i18n( 'Y-m-d', $week_start_end['end'] );

			$query          = "SELECT DATE_FORMAT(created_at, '%b %e') AS label";
            $query .= $wpdb->prepare( ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS potential_revenue', array( 'abandoned' ) ); //phpcs:ignore
            $query .= $wpdb->prepare( ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered_revenue', array( 'recovered' ) ); //phpcs:ignore
            $query .= $wpdb->prepare( " FROM %1s ", array( $abandoned_cat_table ) ); //phpcs:ignore
			$query         .= "WHERE DATE_FORMAT(created_at, '%Y-%m-%d') >= ";
            $query .= $wpdb->prepare( '%s ', array( $start_of_week ) ); //phpcs:ignore
			$query         .= "AND DATE_FORMAT(created_at, '%Y-%m-%d') <= ";
            $query .= $wpdb->prepare( '%s ', array( $end_of_week ) ); //phpcs:ignore
			$query         .= "GROUP BY DATE_FORMAT(created_at, '%b %e') ";
			$query         .= "ORDER BY DATE_FORMAT(created_at, '%c %e') ASC";
			$result         = $wpdb->get_results( $query, ARRAY_A ); //phpcs:ignore
			$formatted_data = Common::format_data_with_label( $result );
			$week_days      = Common::get_week_days( $start_of_week );

			return array_merge( $week_days, $formatted_data );
		}
		return array();
	}
	/**
	 * Retrieves revenue data for the bar chart for the current month.
	 *
	 * @return array The revenue data for the bar chart with daily labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_bar_chart_monthly() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();

		$query          = "SELECT DATE_FORMAT(created_at,  '%b %e') AS label";
		$query         .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS potential_revenue';
		$query         .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered_revenue';
		$query         .= ' FROM %1s ';
		$query         .= 'WHERE (EXTRACT(YEAR_MONTH FROM created_at) = EXTRACT(YEAR_MONTH FROM now())) ';
		$query         .= "GROUP BY DATE_FORMAT(created_at,  '%b %e') ";
		$query         .= "ORDER BY DATE_FORMAT(created_at,  '%b %e') ASC";
        $result = $wpdb->get_results( $wpdb->prepare( $query, array(  'abandoned', 'recovered' ,$abandoned_cat_table) ), ARRAY_A ); //phpcs:ignore
		$formatted_data = Common::format_data_with_label( $result );
		$monthly_days   = Common::get_monthly_days();
		return array_merge( $monthly_days, $formatted_data );
	}
	/**
	 * Retrieves revenue data for the bar chart for the current year.
	 *
	 * @return array The revenue data for the bar chart with monthly labels.
	 * @since 1.5.0
	 */
	public static function get_revenue_for_bar_chart_yearly() {
		global $wpdb;
		$abandoned_cat_table = self::get_table_name();

		$query          = "SELECT DATE_FORMAT(created_at, '%b') AS label";
		$query         .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS potential_revenue';
		$query         .= ', SUM(CASE WHEN status = %s THEN total ELSE 0 END) AS recovered_revenue';
		$query         .= ' FROM %1s ';
		$query         .= 'WHERE YEAR(created_at) = YEAR(now()) ';
		$query         .= "GROUP BY DATE_FORMAT(created_at, '%b') ";
		$query         .= "ORDER BY DATE_FORMAT(created_at, '%b') ASC";
        $result = $wpdb->get_results( $wpdb->prepare( $query, array(  'abandoned', 'recovered' ,$abandoned_cat_table) ), ARRAY_A ); //phpcs:ignore
		$formatted_data = Common::format_data_with_label( $result );
		$months         = Common::get_months_multi_dimensional_array();
		return array_merge( $months, $formatted_data );
	}

	/**
	 * Get automation data based on the provided automation ID.
	 *
	 * @param int $automation_id The ID of the automation to retrieve data for.
	 * @return array|false An array containing automation data or false if no data is found.
	 * @since 1.5.0
	 */
	public static function get_automation_data( $automation_id ) {
		if ( !$automation_id ) {
			return false;
		}
		global $wpdb;
		$automation_table = $wpdb->prefix . AutomationSchema::$table_name;
        $results          = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $automation_table WHERE id = %d and status = %s", array($automation_id, 'active' )), ARRAY_A ); //phpcs:ignore
		if ( !empty( $results ) ) {
			return $results;
		}
		return array();
	}
	/**
	 * Retrieves all active abandoned cart automation records with a specific trigger name.
	 *
	 * This function queries the database to fetch active abandoned cart automation records
	 * that match the provided trigger name.
	 *
	 * @return array Associative array containing the id and name of matching automation records.
	 * @since 1.5.5
	 */
	public static function get_all_abandoned_cart_automation() {
		global $wpdb;
		$automation_table = $wpdb->prefix . AutomationSchema::$table_name;
        $results          = $wpdb->get_results( $wpdb->prepare( "SELECT id as automation_id, name as automation_name  FROM $automation_table WHERE trigger_name = %s and status = %s ORDER BY id DESC", array('wc_abandoned_cart', 'active' )), ARRAY_A ); //phpcs:ignore
		if ( !empty( $results ) ) {
			return $results;
		}
		return array();
	}

	/**
	 * Checks if an abandoned cart with the specified ID exists.
	 *
	 * This function queries the database to check whether an abandoned cart record
	 * with the given ID exists in the database.
	 *
	 * @param int $id The ID of the abandoned cart to check.
	 * @return bool True if an abandoned cart with the specified ID exists, false otherwise.
	 * @since 1.5.5
	 */
	public static function is_abandoned_cart_exist( $id ) {
		global $wpdb;
		$table  = $wpdb->prefix . AbandonedCart::$table_name;
        $result = $wpdb->get_row( $wpdb->prepare( "SELECT id FROM $table WHERE id = %d", array( $id ) ), ARRAY_A ); //phpcs:ignore
		if ( is_array( $result ) && !empty( $result ) ) {
			return true;
		}
		return false;
	}
	/**
	 * Runs the abandoned cart automation manually for a specific abandoned cart ID.
	 *
	 * This function triggers the abandoned cart automation process for a given abandoned cart ID.
	 * It checks if the abandoned cart exists, and if so, it fires the 'mailmint_after_cart_abandoned'
	 * action hook.
	 *
	 * @param int $abandoned_id The ID of the abandoned cart to run automation for.
	 * @param int $automation_id The ID of the triggered automation to run automation for.
	 * @return bool True if the abandoned cart exists and automation was triggered, false otherwise.
	 * @since 1.5.5
	 */
	public static function run_automation_manually( $abandoned_id, $automation_id ) {
		if ( !$abandoned_id || !$automation_id ) {
			return false;
		}
		$is_cart_exist = self::is_abandoned_cart_exist( $abandoned_id );
		if ( $is_cart_exist ) {
			/**
			 * Executes the 'mailmint_manually_run_abandoned_data' action hook, triggering manual processing
			 * of abandoned data for a specific abandoned item and automation.
			 *
			 * This action is typically used to manually initiate the processing of abandoned data, which might
			 * involve sending notifications, updating records, or performing other tasks related to abandoned items.
			 *
			 * @since 1.0.0
			 *
			 * @param int $abandoned_id The ID of the abandoned item to be processed.
			 * @param int $automation_id The ID of the automation associated with the abandoned item.
			 */
			do_action( 'mailmint_manually_run_abandoned_data', $abandoned_id, $automation_id );
			return true;
		}
		return false;
	}

	/**
	 * Retrieves the abandoned cart details by ID.
	 *
	 * This function queries the database to fetch the details of an abandoned cart record
	 * with the specified ID.
	 *
	 * @param int    $abandoned_id The ID of the abandoned cart to retrieve details for.
	 * @param string $status       The status of the abandoned cart to retrieve.
	 * @return array|false An array containing the abandoned cart details if found, or false if not found.
	 * @since 1.5.5
	 */
	public static function get_cart_details_by_id($abandoned_id, $status = ''){
		global $wpdb;
		$table  = $wpdb->prefix . AbandonedCart::$table_name;

		// Base query and parameters.
		$query = "SELECT * FROM {$table} WHERE id = %d";
		$params = [$abandoned_id];

		// Add status condition if status is not empty.
		if ( !empty( $status ) ) {
			$query .= " AND status = %s";
			$params[] = $status;
		}

		// Prepare and execute the query.
		$result = $wpdb->get_row( $wpdb->prepare( $query, $params ), ARRAY_A ); //phpcs:ignore
		if (is_array($result) && !empty($result)) {
			return $result;
		}
		return array();
	}
}
