<?php
/**
 * Manage lead magnet schema
 *
 * @desc Manage plugin's assets
 * @package /app/Database/Schemas
 * @since 1.11.0
 */

namespace MailMintPro\Mint\DataBase\Tables;

use Mint\MRM\Interfaces\Schema;

/**
 * Class LeadMagnet
 *
 * @desc Manage plugin's assets
 * @package /app/Database/Schemas
 * @since 1.11.0
 */
class LeadMagnet implements Schema {

	/**
	 * Lead magnets table name.
	 *
	 * @var string
	 * @since 1.11.0
	 */
	public static $table_name = 'mint_lead_magnets';

	/**
	 * Lead magnet download tracking table name.
	 *
	 * @var string
	 * @since 1.11.0
	 */
	public static $tracking_table_name = 'mint_lead_magnet_download_tracking';

	/**
	 * Get the SQL statements for creating the necessary tables.
	 *
	 * @return string
	 * @since 1.11.0
	 */
	public function get_sql() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		// Create lead magnets table.
		$lead_magnets_table = $wpdb->prefix . self::$table_name;
		$this->create_lead_magnets_table( $lead_magnets_table, $charset_collate );

		// Create lead magnets download tracking table.
		$tracking_table = $wpdb->prefix . self::$tracking_table_name;
		$this->create_lead_magnet_download_tracking_table( $tracking_table, $charset_collate );
	}

	/**
	 * Create lead magnets table
	 *
	 * @param string $table Table name
	 * @param string $charset_collate Charset collate
	 * @since 1.11.0
	 */
	private function create_lead_magnets_table( $table, $charset_collate ) {
		$sql = $this->lead_magnets_sql( $table, $charset_collate );
		dbDelta( $sql );
	}

	/**
	 * Create lead magnet download trackings table
	 *
	 * @param string $table Table name
	 * @param string $charset_collate Charset collate
	 * @since 1.11.0
	 */
	private function create_lead_magnet_download_tracking_table( $table, $charset_collate ) {
		$sql = $this->lead_magnet_download_track_sql( $table, $charset_collate );
		dbDelta( $sql );
	}

	/**
	 * Get lead magnets SQL
	 *
	 * @param string $table Table name
	 * @param string $charset_collate Charset collate
	 * @return string
	 * @since 1.11.0
	 */
	public function lead_magnets_sql( $table, $charset_collate ) {
		$sql = "CREATE TABLE IF NOT EXISTS {$table} (
            `id` BIGINT UNSIGNED NOT NULL PRIMARY KEY AUTO_INCREMENT,
            `title` VARCHAR(192) NOT NULL,
			`description` TEXT NULL,
            `hash` VARCHAR(90) NOT NULL,
            `attachment_id` BIGINT UNSIGNED NOT NULL,
            `url` VARCHAR(192) NOT NULL,
            `created_at` TIMESTAMP NULL,
            `updated_at` TIMESTAMP NULL,
             INDEX `hash_index` (`hash` ASC)
        ) $charset_collate;";

		return $sql;
	}

	/**
	 * Get lead magnets SQL
	 *
	 * @param string $table Table name
	 * @param string $charset_collate Charset collate
	 * @return string
	 * @since 1.11.0
	 */
	public function lead_magnet_download_track_sql( $table, $charset_collate ) {
		$sql = "CREATE TABLE IF NOT EXISTS {$table} (
            `id` BIGINT UNSIGNED NOT NULL PRIMARY KEY AUTO_INCREMENT,
            `email` VARCHAR(192) NOT NULL,
            `token` VARCHAR(90) NOT NULL,
            `is_used` BOOLEAN NOT NULL DEFAULT FALSE,
			`expires_at` DATETIME NULL,
            `ip_address` VARCHAR(192) NULL,
            `created_at` TIMESTAMP NULL,
            `updated_at` TIMESTAMP NULL,
             INDEX `token_index` (`token` ASC)
        ) $charset_collate;";

		return $sql;
	}
}
