<?php
/**
 * Class AbandonedCart
 *
 * Represents the AbandonedCart schema, responsible for creating the necessary database tables
 * related to abandoned carts and their metadata.
 *
 * @package /app/Database/Schemas
 * @since 1.5.0
 */

namespace MailMintPro\Mint\DataBase\Tables;

require_once ABSPATH . 'wp-admin/includes/upgrade.php';

use Mint\MRM\Interfaces\Schema;

/**
 * Class AbandonedCart
 *
 * Represents the AbandonedCart schema, responsible for creating the necessary database tables
 * related to abandoned carts and their metadata.
 *
 * @package /app/Database/Schemas
 * @since 1.5.0
 */
class AbandonedCart implements Schema {

	/**
	 * The name of the abandoned carts table.
	 *
	 * @var string
	 * @since 1.5.0
	 */
	public static $table_name = 'mint_abandoned_carts';

	/**
	 * The name of the abandoned carts meta table.
	 *
	 * @var string
	 * @since 1.5.0
	 */
	public static $meta_table_name = 'mint_abandoned_carts_meta';

	/**
	 * Get the SQL statements for creating the necessary tables.
	 *
	 * This function generates the SQL statements for creating the abandoned carts table
	 * and the abandoned carts meta table. It uses the global $wpdb object to retrieve
	 * the table prefixes and charset collation.
	 *
	 * @return void
	 * @since 1.5.0
	 */
	public function get_sql() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		// Create abandoned carts table.
		$abandoned_carts_table = $wpdb->prefix . self::$table_name;
		$this->create_abandoned_carts_table( $abandoned_carts_table, $charset_collate );

		// Create abandoned carts meta table.
		$abandoned_carts_meta_table = $wpdb->prefix . self::$meta_table_name;
		$this->create_abandoned_carts_meta_table( $abandoned_carts_meta_table, $charset_collate );
	}

	/**
	 * Create the abandoned carts table.
	 *
	 * This function creates the abandoned carts table if it doesn't exist already.
	 * The table structure includes columns for various cart-related information such as email, status, items, etc.
	 *
	 * @param string $table           The name of the table to create.
	 * @param string $charset_collate The charset and collation to use for the table.
	 *                                This should be generated using the `$wpdb->get_charset_collate()` method.
	 * @return void
	 * @since 1.5.0
	 */
	private function create_abandoned_carts_table( $table, $charset_collate ) {
		$sql = $this->abandoned_carts_sql( $table, $charset_collate );

		dbDelta( $sql );
	}

	/**
	 * Generates the SQL statement to create the abandoned carts meta table.
	 *
	 * @param string $table The name of the table to be created.
	 * @param string $charset_collate The character set and collation for the table.
	 * @return string The SQL statement for creating the abandoned carts meta table.
	 * @since 1.5.0
	 */
	public function abandoned_carts_sql( $table, $charset_collate ) {
		$sql = "CREATE TABLE IF NOT EXISTS {$table} (
            `id` BIGINT UNSIGNED NOT NULL PRIMARY KEY AUTO_INCREMENT,
            `email` VARCHAR(255),
			`user_id` BIGINT(20),
            `automation_id` BIGINT(20),
            `status` VARCHAR(50) NOT NULL,
            `items` LONGTEXT,
			`provider` VARCHAR(32),
            `total` VARCHAR(32),
            `token` VARCHAR(32),
            `session_key` VARCHAR(32),
		    `checkout_data` LONGTEXT,
            `created_at` TIMESTAMP NULL,
            `updated_at` TIMESTAMP NULL,
            INDEX `abandoned_cart_id` (`id` ASC),
            INDEX `abandoned_cart_email` (`email` ASC)
        ) $charset_collate;";

		return $sql;
	}

	/**
	 * Create the abandoned carts meta table.
	 *
	 * This function creates the abandoned carts meta table if it doesn't exist already.
	 * The table structure includes columns for storing meta information related to abandoned carts.
	 *
	 * @param string $table           The name of the table to create.
	 * @param string $charset_collate The charset and collation to use for the table.
	 *                                This should be generated using the `$wpdb->get_charset_collate()` method.
	 * @return void
	 * @since 1.5.0
	 */
	private function create_abandoned_carts_meta_table( $table, $charset_collate ) {
		$sql = $this->abandoned_carts_meta_sql( $table, $charset_collate );
		dbDelta( $sql );
	}

	/**
	 * Generates the SQL statement to create the abandoned carts meta table.
	 *
	 * @param string $table The name of the table to be created.
	 * @param string $charset_collate The character set and collation for the table.
	 * @return string The SQL statement for creating the abandoned carts meta table.
	 * @since 1.5.0
	 */
	public function abandoned_carts_meta_sql( $table, $charset_collate ) {
		$sql = "CREATE TABLE IF NOT EXISTS {$table} (
            `id` BIGINT UNSIGNED NOT NULL PRIMARY KEY AUTO_INCREMENT,
            `abandoned_cart_id` BIGINT(20),
            `meta_key` VARCHAR(255),
            `meta_value` LONGTEXT,
            `created_at` TIMESTAMP NULL,
            `updated_at` TIMESTAMP NULL,
            INDEX `abandoned_cart_id` (`abandoned_cart_id` ASC)
         ) $charset_collate;";
		return $sql;
	}
}
