<?php
/**
 * Webhook Class for Maintain webhook Table data
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2022-08-09 11:03:17
 * @modify date 2022-08-09 11:03:17
 * @package /app
 */

namespace MailMintPro\Mint\DataBase\Models;

use Exception;
use MailMintPro\Mint\DataBase\Tables\WebHook;
use MailMintPro\Mint\DataStores\WebHookData;
use MailMintPro\Mint\Internal\Traits\Singleton;
use Mint\MRM\DataBase\Tables\CustomFieldSchema;
/**
 * MintMail WebHook class.
 *
 * @since 1.0.0
 */
class WebHookModel {
	use Singleton;


	/**
	 * SQL query to create a new note
	 *
	 * @param WebHookData $webhook Webhook Object.
	 * @return false|int|string
	 * @since 1.0.0
	 */
	public static function insert( WebHookData $webhook ) {
		global $wpdb;
		$webhook_table = $wpdb->prefix . WebHook::$table_name;
		try {
			$wpdb->insert(
				$webhook_table,
				array(
					'name'          => $webhook->get_name(),
					'hash'          => $webhook->get_hash(),
					'settings'      => $webhook->get_settings(),
					'topic'         => $webhook->get_topic(),
					'failure_count' => $webhook->get_failure_count(),
					'created_at'    => current_time( 'mysql' ),
				)
			); // db call ok.
			$insert_id = ! empty( $wpdb->insert_id ) ? $wpdb->insert_id : '';
			return $insert_id;
		} catch ( Exception $e ) {
			return false;
		}
	}

	/**
	 * SQL query to update a note.
	 *
	 * @param WebHookData $webhook Get Webhook object.
	 * @param int         $webhook_id Get Wen Hook ID.
	 * @return bool
	 * @since 1.0.0
	 */
	public static function update( WebHookData $webhook, $webhook_id ) {
		global $wpdb;
		$table = $wpdb->prefix . WebHook::$table_name;

		try {
			$wpdb->update(
				$table,
				array(
					'name'          => $webhook->get_name(),
					'hash'          => $webhook->get_hash(),
					'settings'      => $webhook->get_settings(),
					'topic'         => $webhook->get_topic(),
					'failure_count' => $webhook->get_failure_count(),
					'updated_at'    => current_time( 'mysql' ),
				),
				array(
					'id' => $webhook_id,
				)
			); // db call ok. // no-cache ok.
			return true;
		} catch ( Exception $e ) {
			return false;
		}
	}

	/**
	 * Delete a note from the database
	 *
	 * @param mixed $id Note id.
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public static function destroy( $id ) {
		global $wpdb;
		$table = $wpdb->prefix . WebHook::$table_name;

		try {
			$wpdb->delete( $table, array( 'id' => $id ) ); // db call ok. ; no-cache ok.
			return true;
		} catch ( \Exception $e ) {
			return false;
		}
	}


	/**
	 * SQL query to get all notes for a contact with pagination
	 *
	 * @param int    $offset offset.
	 * @param int    $limit limit.
	 * @param string $search search Parameter.
	 * @param string $order_by search Parameter.
	 * @param string $order_type search Parameter.
	 *
	 * @return array|bool
	 * @since 1.0.0
	 */
	public static function get_all( $offset = 0, $limit = 10, $search = '', $order_by = 'id', $order_type = 'DESC' ) {
		global $wpdb;
		$table_name   = $wpdb->prefix . WebHook::$table_name;
		$search_terms = null;

		// Search notes by title.
		if ( ! empty( $search ) ) {
			$search       = $wpdb->esc_like( $search );
			$search_terms = "WHERE name LIKE '%%$search%%'";
		}
		// Return forms in list view.
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$select_query = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name {$search_terms} ORDER BY $order_by $order_type LIMIT %d, %d", array( $offset, $limit ) ), ARRAY_A ); // db call ok. ; no-cache ok.
		$count_query  = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) as total FROM $table_name" ) ); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$count       = (int) $count_query;
		$total_pages = ceil( $count / $limit );

		return array(
			'data'        => $select_query,
			'total_pages' => $total_pages,
			'count'       => $count,
		);
	}

	/**
	 * Run SQL Query to get a single note information
	 *
	 * @param mixed $id Note ID.
	 *
	 * @return object
	 * @since 1.0.0
	 */
	public static function get( $id ) {
		global $wpdb;
		$table_name = $wpdb->prefix . WebHook::$table_name;
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return $wpdb->get_row( $wpdb->prepare( "SELECT *  FROM {$table_name} WHERE id = %d", array( $id ) ), ARRAY_A ); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Delete multiple forms
	 *
	 * @param array $webhook_ids webhook_ids id.
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public static function destroy_all( $webhook_ids ) {
		global $wpdb;
		$form_table = $wpdb->prefix . WebHook::$table_name;
		if ( is_array( $webhook_ids ) && count( $webhook_ids ) > 0 ) {
			$forms_ids = implode( ',', array_map( 'intval', $webhook_ids ) );
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			return $wpdb->query( "DELETE FROM $form_table WHERE id IN($webhook_ids)" ); // db call ok. ; no-cache ok.
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		}
		return false;
	}
	/**
	 * Run SQL Query to get a single note information
	 *
	 * @param mixed $hash Webhook Hash.
	 *
	 * @return object
	 * @since 1.0.0
	 */
	public static function get_hook_by_hash( $hash ) {
		global $wpdb;
		$table_name = $wpdb->prefix . WebHook::$table_name;
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$result = $wpdb->get_row( $wpdb->prepare( "SELECT  * FROM {$table_name} WHERE hash = %s", array( $hash ) ), ARRAY_A ); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$result['settings'] = isset( $result['settings'] ) ? maybe_unserialize( $result['settings'] ) :array();
		return $result;
	}

	/**
	 * Get Custom Fields.
	 *
	 * @return array
	 */
	public static function get_custom_fields() {
		global $wpdb;
		$fields_table = $wpdb->prefix . CustomFieldSchema::$table_name;
		// Return field froups for list view.
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$select_query = $wpdb->prepare( "SELECT title, slug FROM $fields_table " );
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // phpcs:disable WordPress.DB.PreparedSQL.NotPrepared
		$results = $wpdb->get_results( $select_query, ARRAY_A ); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.NotPrepared
		return array(
			'data' => $results,
		);
	}
}
