<?php

/**
 * LinkTriggerModel class for handling database operations related to link triggers.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-07-10 11:03:17
 * @modify date 2024-07-10 11:03:17
 * @package MailMintPro\Mint\DataBase\Models
 */

namespace MailMintPro\Mint\DataBase\Models;

use MailMintPro\Mint\DataBase\Tables\LinkTrigger;

/**
 * LinkTriggerModel class for handling database operations related to link triggers.
 *
 * This class includes methods for creating, updating, and retrieving link triggers from the database.
 * 
 * @since 1.15.0
 */
class LinkTriggerModel {

	/**
	 * Retrieves the table name for the link triggers.
	 * 
	 * @return string The table name for the link triggers.
	 * @since 1.15.0
	 */
    protected static function _table() {
		global $wpdb;
		return $wpdb->prefix . LinkTrigger::$table_name;
	}

	/**
	 * Retrieves the ID of the last inserted row.
	 * 
	 * @return int The ID of the last inserted row.
	 * @since 1.15.0
	 */
    protected static function insert_id() {
		global $wpdb;
		return $wpdb->insert_id;
	}

	/**
	 * Creates a new link trigger in the database.
	 * 
	 * @param array $data The data to insert into the database.
	 * @return int|void The ID of the newly created link trigger, or void if no data provided.
	 * @since 1.15.0
	 */
    public static function create( $data ) {
		if ( empty( $data ) ) {
			return;
		}

		self::insert( $data );
		$link_id = absint( self::insert_id() );

		$hash = md5( $link_id . time() );

		self::update_link_trigger_data( $link_id, [
			'hash' => $hash
		] );

		return $link_id;
	}

	/**
	 * Inserts data into the link triggers table.
	 * 
	 * @param array $data The data to insert into the table.
	 * @return void
	 * @since 1.15.0
	 */
	protected static function insert( $data ) {
		global $wpdb;
		$wpdb->insert( self::_table(), $data );
	}

	/**
	 * Updates the data for a link trigger in the database.
	 * 
	 * @param int   $id   The ID of the link trigger to update.
	 * @param array $data The data to update for the link trigger.
	 * @return bool True if the link trigger was updated successfully, false otherwise.
	 * @since 1.15.0
	 */
	public static function update_link_trigger_data( $id, $data ) {
		if ( ! is_array( $data ) ) {
			return false;
		}

		return ! ! self::update( $data, array(
			'id' => absint( $id ),
		) );
	}

	/**
	 * Updates data in the link triggers table.
	 * 
	 * @param array $data  The data to update in the table.
	 * @param array $where The conditions to match for updating the data.
	 * @return int The number of rows updated.
	 * @since 1.15.0
	 */
    protected static function update( $data, $where ) {
		global $wpdb;
		return $wpdb->update( self::_table(), $data, $where );
	}

	/**
	 * Deletes a link trigger from the database.
	 *
	 * @param array $ids The IDs of the link triggers to delete.
	 * @return bool True if the link trigger was deleted successfully, false otherwise.
	 * @since 1.15.0
	 */
	public static function destroy_all( $ids ){
		global $wpdb;
		$ids   = implode(',', array_map('intval', $ids));
		$query = $wpdb->prepare( 'DELETE FROM %1s WHERE `id` IN(%1s)', self::_table(), $ids ); //phpcs:ignore
		$wpdb->query( $query ); //phpcs:ignore
		return true;
	}

	/**
     * Retrieves all link triggers with pagination, search, and sorting functionality.
     *
     * @param int    $offset     The offset for pagination.
     * @param int    $limit      The number of items per page.
     * @param string $search     The search term for filtering by title.
     * @param string $order_by   The column to order by.
     * @param string $order_type The order type ('ASC' or 'DESC').
     * 
     * @return array The array containing the triggers, total pages, and total count.
     * @since 1.15.0
     */
	public static function get_all( $offset = 0, $limit = 10, $search = '', $order_by = 'id', $order_type = 'DESC' ) {
		global $wpdb;
		$search_terms = '';

        // Sanitize and validate inputs.
        $offset     = absint( $offset );
        $limit      = absint( $limit );
        $order_by   = sanitize_key( $order_by );
        $order_type = strtoupper( $order_type );
        $order_type = in_array( $order_type, array( 'ASC', 'DESC' ), true ) ? $order_type : 'DESC';

		// Prepare search term if provided.
        if ( ! empty( $search ) ) {
            $search = '%' . $wpdb->esc_like( $search ) . '%';
            $search_terms = $wpdb->prepare( "WHERE title LIKE %s", $search );
        }

        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // Prepare and execute the select query.
        $select_query = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM %1s {$search_terms} ORDER BY $order_by $order_type LIMIT %d, %d", self::_table(), $offset, $limit ), ARRAY_A ); // db call ok. ; no-cache ok.

        // Prepare and execute the count query.
        $count_query = $wpdb->get_var( $wpdb->prepare("SELECT COUNT(*) as total FROM %1s {$search_terms}", self::_table() ) ); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        $count       = (int) $count_query;
        $total_pages = ceil( $count / $limit );

        return array(
            'triggers'    => $select_query,
            'total_pages' => $total_pages,
            'count'       => $count,
        );
	}

	/**
	 * Retrieves a single link trigger by ID.
	 *
	 * @param int $id The ID of the link trigger to retrieve.
	 * @return array|null The link trigger data as an associative array, or null if not found.
	 * 
	 * @since 1.15.0
	 */
	public static function get( $id ) {
		global $wpdb;
		return $wpdb->get_row( $wpdb->prepare( "SELECT *  FROM %1s WHERE id = %d", array( self::_table(), $id ) ), ARRAY_A ); // db call ok. ; no-cache ok.
	}

	/**
	 * Retrieves a single link trigger by hash.
	 *
	 * @param string $slug The hash of the link trigger to retrieve.
	 * @return array|null The link trigger data as an associative array, or null if not found.
	 * 
	 * @since 1.16.0
	 */
	public static function get_by_slug( $slug ) {
		global $wpdb;
		$row = $wpdb->get_row( $wpdb->prepare( "SELECT *  FROM %1s WHERE hash = %s", array( self::_table(), $slug ) ), ARRAY_A ); // db call ok. ; no-cache ok.
	
		if ( $row && isset( $row['data'] ) ) {
			$data = unserialize($row['data']);
			unset($row['data']);
			$row = array_merge($row, $data);
		}

		return $row;
	}
}
