<?php
/**
 * LeadMagnetModel class to manage database interactions related to lead magnets.
 *
 * Class LeadMagnetModel
 *
 * @package MailMintPro\Mint\DataBase\Models
 */

namespace MailMintPro\Mint\DataBase\Models;

use Exception;
use MailMintPro\Mint\DataBase\Tables\LeadMagnet;

/**
 * Manages all database interactions related to lead magnets within the MailMintPro application.
 *
 * The LeadMagnetModel class provides an abstraction layer over the database operations involving lead magnets,
 * encapsulating all CRUD operations (Create, Read, Update, Delete).
 * 
 * @since 1.11.0
 */
class LeadMagnetModel {

	/**
     * Inserts a new lead magnet record into the database.
     *
     * @param array $magnet Array of lead magnet attributes to be inserted.
     * @return mixed Returns the insert ID on success, or false on failure.
	 * 
	 * @since 1.11.0
     */
	public static function insert( $magnet ) {
		global $wpdb;
		$magnet_table = $wpdb->prefix . LeadMagnet::$table_name;

		$magnet['created_at'] = current_time( 'mysql' );
		$wpdb->insert( $magnet_table, $magnet ); // db call ok.
		$insert_id = ! empty( $wpdb->insert_id ) ? $wpdb->insert_id : false;
		return $insert_id;
	}

	/**
     * Updates an existing lead magnet record in the database.
     *
     * @param array $magnet Array of lead magnet attributes to be updated.
     * @param int $magnet_id The ID of the lead magnet to update.
     * @return bool Returns true on success, or false on failure.
	 * 
	 * @since 1.11.0
     */
	public static function update( $magnet, $magnet_id ) {
		global $wpdb;
		$table = $wpdb->prefix . LeadMagnet::$table_name;

		try {
			$magnet['updated_at'] = current_time( 'mysql' );
			$wpdb->update( $table, $magnet,array( 'id' => $magnet_id ) ); // db call ok. // no-cache ok.
			return true;
		} catch ( Exception $e ) {
			return false;
		}
	}

	/**
     * Deletes a lead magnet record from the database.
     *
     * @param int $id The ID of the lead magnet to delete.
     * @return int|false The number of rows affected, or false on error.
	 * 
	 * @since 1.11.0
     */
	public static function destroy( $id ) {
		global $wpdb;
		$table = $wpdb->prefix . LeadMagnet::$table_name;

		return $wpdb->delete( $table, array( 'id' => $id ) ); // db call ok. ; no-cache ok.
	}

	/**
     * Retrieves a paginated list of lead magnets based on search criteria.
     *
     * @param int $offset Starting point for records to return.
     * @param int $limit Number of records to return.
     * @param string $search Search term for filtering the results.
     * @param string $order_by Column to sort the results by.
     * @param string $order_type Direction of the sort order.
     * @return array Contains data array, total pages, and count of total records.
	 * 
	 * @since 1.11.0
     */
	public static function get_all( $offset = 0, $limit = 10, $search = '', $order_by = 'id', $order_type = 'DESC' ) {
		global $wpdb;
		$table_name   = $wpdb->prefix . LeadMagnet::$table_name;
		$search_terms = null;

		// Search notes by title.
		if ( ! empty( $search ) ) {
			$search       = $wpdb->esc_like( $search );
			$search_terms = "WHERE title LIKE '%%$search%%'";
		}

		// Return forms in list view.
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$select_query = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table_name {$search_terms} ORDER BY $order_by $order_type LIMIT %d, %d", array( $offset, $limit ) ), ARRAY_A ); // db call ok. ; no-cache ok.
		$count_query  = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) as total FROM $table_name" ) ); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

		$count       = (int) $count_query;
		$total_pages = ceil( $count / $limit );

		return array(
			'data'        => $select_query,
			'total_pages' => $total_pages,
			'count'       => $count,
		);
	}

	/**
     * Retrieves a single lead magnet by ID.
     *
     * @param int $id The ID of the lead magnet to retrieve.
     * @return array|null Lead magnet data as an associative array, or null if not found.
	 * 
	 * @since 1.11.0
     */
	public static function get( $id ) {
		global $wpdb;
		$table_name = $wpdb->prefix . LeadMagnet::$table_name;
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return $wpdb->get_row( $wpdb->prepare( "SELECT *  FROM {$table_name} WHERE id = %d", array( $id ) ), ARRAY_A ); // db call ok. ; no-cache ok.
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
     * Retrieves tracking information for a specific token.
     *
     * @param string $token The token to retrieve tracking info for.
     * @return array|null Tracking data as an associative array, or null if not found.
	 * 
	 * @since 1.11.0
     */
	public static function get_tracking_info( $token ) {
		global $wpdb;
		$tracking_table = $wpdb->prefix . LeadMagnet::$tracking_table_name;
        $query          = $wpdb->prepare( "SELECT email, token, is_used FROM {$tracking_table} WHERE token = %s", $token );
        return $wpdb->get_row($query, ARRAY_A);
    }

	/**
     * Retrieves the attachment ID associated with a given hash.
     *
     * @param string $lead_hash The hash to retrieve the attachment ID for.
     * @return int|null The attachment ID, or null if not found.
	 * 
	 * @since 1.11.0
     */
	public static function get_attachment_id( $lead_hash ) {
		global $wpdb;
		$table_name = $wpdb->prefix . LeadMagnet::$table_name;
        $query      = $wpdb->prepare( "SELECT attachment_id FROM {$table_name} WHERE hash = %s", $lead_hash );
        return $wpdb->get_var( $query );
    }

	/**
     * Marks a token as used in the tracking table.
     *
     * @param string $token The token to mark as used.
     * @return int Updated row ID.
	 * 
	 * @since 1.11.0
     */
	public static function mark_as_used( $token ) {
		global $wpdb;
		$tracking_table = $wpdb->prefix . LeadMagnet::$tracking_table_name;
        return $wpdb->update( $tracking_table, array( 'is_used' => 1 ), array( 'token' => $token ) );
    }
}
