<?php
/**
 * DatabaseMigrator class.
 *
 * @package MailMintPro\Mint\DataBase\Migration
 * @namespace MailMintPro\Mint\DataBase\Migration
 * @author [MRM Team]
 * @email [support@wpfunnels.com]
 */

namespace MailMintPro\Mint\DataBase\Migration;

use MailMintPro\Mint\DataBase\Tables\AbandonedCart;
use MailMintPro\Mint\DataBase\Tables\LeadMagnet;
use MailMintPro\Mint\DataBase\Tables\LinkTrigger;
use MailMintPro\Mint\Internal\Traits\Singleton;

/**
 * DatabaseMigrator class
 *
 * Manages database migrations.
 *
 * @package MailMintPro\Mint\DataBase\Migration
 * @namespace MailMintPro\Mint\DataBase\Migration
 *
 * @version 1.5.0
 */
class DatabaseMigrator {

	use Singleton;

	/**
	 * Existing database version
	 *
	 * @var string
	 *
	 * @since 1.5.0
	 */
	private $current_db_version;

	/**
	 * New database version
	 *
	 * @var array
	 *
	 * @since 1.5.0
	 */
	private $update_db_versions;

	/**
	 * Initialize class functionalities
	 *
	 * @return void
	 *
	 * @since 1.5.0
	 */
	public function init() {
		$this->update_db_versions = array(
			'1.0.1' => array(
				'create_abandoned_cart_table',
			),
			'1.11.0' => array(
				'create_lead_magnet_download_tracking_table',
			),
			'1.12.0' => array(
				'create_lead_magnets_table',
			),
			'1.16.0' => array(
				'create_link_triggers_table',
			),
		);

		$this->current_db_version = get_option( 'mail_mint_pro_db_version', null );

		if ( !is_null( $this->current_db_version ) ) {
			$this->upgrade_database_tables();
		}
	}

	/**
	 * Upgrade all required database
	 *
	 * @return void
	 *
	 * @since 1.5.0
	 */
	public function upgrade_database_tables() {
		$update_versions = $this->get_db_update_versions();

		foreach ( $update_versions as $version => $callbacks ) {
			if ( version_compare( $this->current_db_version, $version, '<' ) ) {
				foreach ( $callbacks as $callback ) {
					if ( method_exists( $this, $callback ) ) {
						$this->$callback();
					}
				}
			}
		}

		$this->update_db_version();
	}

	/**
	 * Get update database versions
	 *
	 * @return mixed|void
	 *
	 * @since 1.5.0
	 */
	public function get_db_update_versions() {
		return apply_filters( 'mail_mint_pro_db_version_for_update', $this->update_db_versions );
	}

	/**
	 * Update database version
	 *
	 * @return void
	 * @since 1.5.0
	 */
	public function update_db_version() {
		update_option(
			'mail_mint_pro_db_version',
			apply_filters( 'mail_mint_pro_db_version', MAIL_MINT_PRO_DB_VERSION ),
			false
		);
	}

	/**
	 * Creates the abandoned cart table and its associated meta table.
	 *
	 * @since 1.5.0
	 */
	public function create_abandoned_cart_table() {
		$this->create_abandoned_cart_main_table();
		$this->create_abandoned_carts_meta_table();
	}

	/**
	 * Creates the abandoned cart main table.
	 *
	 * @global wpdb $wpdb WordPress database object.
	 * @since 1.5.0
	 */
	public function create_abandoned_cart_main_table() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		// Create abandoned carts table.
		$abandoned_carts_table  = $wpdb->prefix . 'mint_abandoned_carts';
		$abandoned_carts_object = new AbandonedCart();
		$abandoned_carts        = $abandoned_carts_object->abandoned_carts_sql( $abandoned_carts_table, $charset_collate );

		return $wpdb->query( $abandoned_carts ); //phpcs:ignore
	}

	/**
	 * Creates the abandoned carts meta table.
	 *
	 * @global wpdb $wpdb WordPress database object.
	 * @since 1.5.0
	 */
	public function create_abandoned_carts_meta_table() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		$abandoned_carts_meta_table = $wpdb->prefix . 'mint_abandoned_carts_meta';
		$abandoned_carts_object     = new AbandonedCart();
		$abandoned_carts_meta       = $abandoned_carts_object->abandoned_carts_meta_sql( $abandoned_carts_meta_table, $charset_collate );

		return $wpdb->query( $abandoned_carts_meta ); //phpcs:ignore
	}

	/**
	 * Creates the lead magnets table.
	 *
	 * @since 1.11.0
	 */
	public function create_lead_magnets_table() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		$lead_magnets_table  = $wpdb->prefix . 'mint_lead_magnets';
		$lead_magnets_object = new LeadMagnet();
		$lead_magnets        = $lead_magnets_object->lead_magnets_sql( $lead_magnets_table, $charset_collate );

		return $wpdb->query( $lead_magnets ); //phpcs:ignore
	}

	/**
	 * Creates the lead magnet download tracking table.
	 *
	 * @since 1.11.0
	 */
	public function create_lead_magnet_download_tracking_table() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		$lead_magnets_table  = $wpdb->prefix . 'mint_lead_magnet_download_tracking';
		$lead_magnets_object = new LeadMagnet();
		$lead_magnets        = $lead_magnets_object->lead_magnet_download_track_sql( $lead_magnets_table, $charset_collate );

		return $wpdb->query( $lead_magnets ); //phpcs:ignore
	}

	/**
	 * Creates the link triggers table.
	 *
	 * @since 1.15.0
	 */
	public function create_link_triggers_table() {
		global $wpdb;
		$charset_collate = $wpdb->get_charset_collate();

		$link_triggers_table  = $wpdb->prefix . 'mint_link_triggers';
		$link_triggers_object = new LinkTrigger();
		$link_triggers        = $link_triggers_object->link_triggers_sql( $link_triggers_table, $charset_collate );

		return $wpdb->query( $link_triggers ); //phpcs:ignore
	}
}
