<?php
/**
 * Mail Mint
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2022-08-09 11:03:17
 * @modify date 2022-08-09 11:03:17
 * @package /app/API/
 */

namespace MailMintPro\MintAdmin\API;

use MailMintPro\Mint\Internal\Traits\Singleton;
use MRM\Common\MrmCommon;

defined( 'ABSPATH' ) || exit;

/**
 * [Register REST API routes after plugin has been activated]]
 *
 * @desc Manage Campaign email builder related API
 * @package /app/API/Routes
 * @since 1.0.0
 */
class Server {

	use Singleton;

	/**
	 * REST API namespaces and endpoints.
	 *
	 * @var array
	 * @since 1.0.0
	 */
	protected $routes = array();


	/**
	 * Hook into WordPress ready to init the REST API as needed.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// rest api endpoints.
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ), 10 );
		add_action( 'wp_loaded', array( $this, 'rest_api_includes' ), 5 );
	}

	/**
	 * Register REST API after plugin activation
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function rest_api_init() {
		// Codes needs to be audited. Need to follow WP way here.

		foreach ( $this->get_rest_namespaces() as $namespace => $controllers ) {
			foreach ( $controllers as $controller_name => $route_class ) {
				$route_class_name                               = '\MailMintPro\\Mint\\Admin\\API\\Routes\\' . $route_class;
				$this->routes[ $namespace ][ $controller_name ] = new $route_class_name();
				$this->routes[ $namespace ][ $controller_name ]->register_routes();
			}
		}
	}

	/**
	 * Includes the necessary files for the REST API functionality.
	 *
	 * This function includes the required files for the REST API to work properly.
	 * It includes the WC Cart Functions and WC Notice Functions files from the WooCommerce plugin.
	 * These files provide important functions and functionality related to the cart and notices.
	 *
	 * @since 1.5.0
	 */
	public function rest_api_includes() {
		if ( MrmCommon::is_wc_active() ) {
			require_once WC_ABSPATH . 'includes/wc-cart-functions.php';
			require_once WC_ABSPATH . 'includes/wc-notice-functions.php';
		}
	}

	/**
	 * Get API namespaces - new namespaces should be registered here.
	 *
	 * @return array List of Namespaces and Main controller classes.
	 * @since 1.0.0
	 */
	protected function get_rest_namespaces() {
		return array(
			'mrm/v1' => $this->get_routes(),
		);
	}

	/**
	 * List of controllers in the mrm/v1 namespace.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	protected function get_routes() {
		return apply_filters(
			'mailmint_pro_rest_api_routes',
			array(
				'segments'                => 'SegmentRoute',
				'analytics'               => 'AnalyticsRoute',
				'contacts'                => 'ContactRoute',
				'custom-fields'           => 'CustomFieldRoute',
				'webhooks'                => 'WebHookRoute',
				'product'                 => 'ProductRoute',
				'email-builder'           => 'EmailBuilderRoute',
				'abandoned-cart-frontend' => 'AbandonedCartFrontendRoute',
				'abandoned-cart-setting'  => 'AbandonedCartSettingRoute',
				'abandoned-cart-admin'    => 'AbandonedCartAdminRoute',
				'contact-export'          => 'ContactExportRoute',
				'contact-profile'         => 'ContactProfileRoute',
				'integration'             => 'IntegrationRoute',
				'lead-magnet'             => 'LeadMagnetRoute',
				'wp-core'                 => 'WPRoute',
				'link-triggers'           => 'LinkTriggerRoute',
				'wc-core'                 => 'WCRoute',
				'form'                    => 'FormRoute',
				'role-management'         => 'RoleManagementRoute',
				'custom-access'           => 'CustomAccessRoute',
				'lms'                     => 'LMSRoute',
				'connector'               => 'ConnectorRoute',
			)
		);
	}

	/**
	 * Gets featured image
	 *
	 * @param object|array $object Image object.
	 * @param string       $field_name Field name in image object.
	 * @param array|object $request Request.
	 *
	 * @return false|mixed
	 */
	public function get_rest_featured_image( $object, $field_name, $request ) {
		if ( $object[ 'featured_media' ] ) {
			$img = wp_get_attachment_image_src( $object[ 'featured_media' ], 'app-thumb' );

			return $img[ 0 ];
		}

		return false;
	}
}
