<?php

/**
 * LinkTriggerRoute class for handling REST API routes related to link triggers.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-07-10 11:03:17
 * @modify date 2024-07-10 11:03:17
 * @package /app/Admin/API/Routes
 */
namespace MailMintPro\Mint\Admin\API\Routes;

use MailMintPro\Mint\Admin\API\Controllers\LinkTriggerController;
use MailMintPro\Mint\Admin\API\Routes\AdminRoute;
use Mint\MRM\Utilities\Helper\PermissionManager;
use WP_REST_Server;

/**
 * LinkTriggerRoute class for handling REST API routes related to link triggers.
 *
 * This class extends the AdminRoute class and is responsible for registering REST API routes
 * related to link triggers. It includes a controller instance for handling the route callbacks.
 * 
 * @since 1.15.0
 */
class LinkTriggerRoute extends AdminRoute {

	/**
	 * Route base.
	 *
	 * @var string
	 * @since 1.15.0
	 */
	protected $rest_base = 'link-triggers';


	/**
	 * Instance of the LinkTriggerController class for handling route callbacks.
	 *
	 * @var object
	 * @since 1.15.0
	 */
	protected $controller;

	/**
	 * Constructor for the LinkTriggerRoute class.
	 *
	 * Initializes the controller instance with an LinkTriggerController object.
	 *
	 * @since 1.15.0
	 */
	public function __construct() {
		$this->controller = new LinkTriggerController();
	}

	/**
	 * Register REST API routes for link triggers.
	 *
	 * @return void
	 * @since 1.15.0
	 */
	public function register_routes() {

		/**
		 * Link trigger create endpoint
		 * Get all link triggers endpoint
		 *
		 * @return void
		 * @since 1.15.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base,
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this->controller, 'create_or_update' ),
					'permission_callback' => PermissionManager::current_user_can('mint_manage_link_triggers'),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'get_all' ),
					'permission_callback' => PermissionManager::current_user_can('mint_manage_link_triggers'),
				),
			)
		);

		/**
		 * Lead magnet get single endpoint
		 *
		 * @return void
		 * @since 1.11.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::EDITABLE,
					'callback'            => array( $this->controller, 'create_or_update' ),
					'permission_callback' => PermissionManager::current_user_can('mint_manage_link_triggers'),
				),
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'get_single' ),
					'permission_callback' => PermissionManager::current_user_can('mint_manage_link_triggers'),
				),
			)
		);

		/**
		 * Lead magnet delete all endpoint.
		 *
		 * @return void
		 * @since 1.15.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/delete',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array($this->controller, 'delete_all'),
					'permission_callback' => PermissionManager::current_user_can('mint_manage_link_triggers'),
				),
			),
		);
	}

}
