<?php
/**
 * Mail Mint
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-19-01 11:03:17
 * @modify date 2024-19-01 11:03:17
 * @package /app/Admin/API/Routes
 */

namespace MailMintPro\Mint\Admin\API\Routes;

use Mint\MRM\Utilities\Helper\PermissionManager;
use MintMailPro\Mint\Admin\API\Controllers\AnalyticsController;
use WP_REST_Server;

/**
 * AnalyticsRoute class for handling REST API routes related to campaign analytics.
 *
 * This class extends the AdminRoute class and is responsible for registering REST API routes
 * related to campaign analytics. It includes a controller instance for handling the route callbacks.
 *
 * @since 1.9.0
 */
class AnalyticsRoute extends AdminRoute {

	/**
	 * The REST base for campaign analytics routes.
	 *
	 * @var string
	 * @since 1.9.0
	 */
	protected $rest_base = 'campaign/analytics';

	/**
	 * Instance of the AnalyticsController class for handling route callbacks.
	 *
	 * @var AnalyticsController
	 * @since 1.9.0
	 */
	protected $controller;

	/**
	 * Constructor for the AnalyticsRoute class.
	 *
	 * Initializes the controller instance with an AnalyticsController object.
	 *
	 * @since 1.9.0
	 */
	public function __construct() {
		$this->controller = new AnalyticsController();
	}

	/**
	 * Register REST API routes for campaign analytics.
	 *
	 * Registers a readable route for retrieving campaign analytics
	 * data based on the provided campaign ID.
	 *
	 * @since 1.9.0
	 */
	public function register_routes() {

		/**
		 * Register REST route for retrieving campaign analytics data.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_analytics_request' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving campaign analytics overview.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/overview/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_analytics_overview' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving campaign emails recipients.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/recipients/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_email_recipients' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving successfully delivered email lists.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/sent-to/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_email_sent_to_lists' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving successfully opened email lists.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/opened/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_email_opened_lists' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving successfully clicked email lists.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/clicked/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_email_clicked_lists' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving successfully bounced email lists.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/bounced/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_email_bounced_lists' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving successfully unsubscribed email lists.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/unsubscribed/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_email_unsubscribed_lists' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);

		/**
		 * Register REST route for retrieving successfully click performance lists.
		 *
		 * @since 1.9.0
		 */
		register_rest_route(
			$this->namespace,
			$this->rest_base . '/(?P<campaign_id>[\d]+)/click-performance/(?P<email_id>[\d]+)',
			array(
				array(
					'methods'             => WP_REST_Server::READABLE,
					'callback'            => array( $this->controller, 'handle_campaign_email_click_performance_lists' ),
					'permission_callback' => PermissionManager::current_user_can('mint_read_campaigns'),
				),
			)
		);
	}
}
