<?php
/**
 * Mail Mint Webhooks functions
 *
 * This class handles Webhook Incomming functions for each WP Actions.
 *
 * @author   Mail Mint Team
 * @category Action
 * @package  MRM
 * @since    1.0.0
 */

namespace MailMintPro\Mint\Admin\API\Controllers;

use Exception;
use MailMintPro\Admin\API\Controllers\BaseController;
use MailMintPro\Mint\DataBase\Models\WebHookModel;
use MailMintPro\Mint\DataStores\WebHookData;
use MailMintPro\Mint\Internal\Traits\Singleton;
use Mint\MRM\DataBase\Models\CustomFieldModel;
use MRM\Common\MrmCommon;
use WP_REST_Request;
/**
 * WebHooks Controller.
 */
class WebHookController extends BaseController {
	use Singleton;

	/**
	 * Webhook object arguments
	 *
	 * @var object
	 * @since 1.0.0
	 */
	public $args;
	/**
	 * Create a new webhook or update a existing segment
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function create_or_update( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$name   = isset( $params['name'] ) ? sanitize_text_field( $params['name'] ) : null;
		if ( empty( $name ) ) {
			return $this->get_error_response( __( 'Title is mandatory', 'mailmint-pro' ), 200 );
		}
		$settings = isset( $params['settings'] ) ? $params['settings'] : array();
		$topic    = isset( $params['topic'] ) ? $params['topic'] : 'contact';
		$hash     = wp_generate_uuid4();
		if ( isset( $params['webhook_id'] ) ) {
			$webhook_data = WebHookModel::get( $params['webhook_id'] );
			if ( !empty( $webhook_data ) ) {
				$hash = isset( $webhook_data['hash'] ) ? $webhook_data['hash'] : '';
			}
		}
		$settings['url'] = site_url( '?mailmint=1&route=webhook&topic=' . $topic . '&hash=' . $hash );
		$this->args      = array(
			'name'          => $name,
			'hash'          => $hash,
			'settings'      => $settings,
			'topic'         => $topic,
			'failure_count' => isset( $params['failure_count'] ) ? $params['failure_count'] : 0,
		);
		try {
			$webhook = new WebHookData( $this->args );

			if ( isset( $params['webhook_id'] ) ) {
				$success = WebHookModel::update( $webhook, $params['webhook_id'] );
			} else {
				$success = WebHookModel::insert( $webhook );
			}

			if ( $success ) {
				return $this->get_success_response( __( 'Webhook has been saved successfully.', 'mailmint-pro' ), 201, $success );
			}
			return $this->get_error_response( __( 'Failed to save', 'mailmint-pro' ), 200 );
		} catch ( Exception $e ) {
			return $this->get_error_response( __( 'WebHook is not valid', 'mailmint-pro' ), 200 );
		}
	}


	/**
	 * Get webhook for list views
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function get_all( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$page     = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page = isset( $params['per-page'] ) ? $params['per-page'] : 25;
		$offset   = ( $page - 1 ) * $per_page;

		$order_by   = isset( $params['order-by'] ) ? strtolower( $params['order-by'] ) : 'id';
		$order_type = isset( $params['order-type'] ) ? strtolower( $params['order-type'] ) : 'desc';

		// valid order by fields and types.
		$allowed_order_by_fields = array( 'name', 'created_at' );
		$allowed_order_by_types  = array( 'asc', 'desc' );

		// validate order by fields or use default otherwise.
		$order_by   = in_array( $order_by, $allowed_order_by_fields, true ) ? $order_by : 'id';
		$order_type = in_array( $order_type, $allowed_order_by_types, true ) ? $order_type : 'desc';

		// Segment Search keyword.
		$search = isset( $params['search'] ) ? sanitize_text_field( $params['search'] ) : '';

		$webhook         = WebHookModel::get_all( $offset, $per_page, $search, $order_by, $order_type );
		$webhook['data'] = array_map(
			function( $webhook ) {
				if ( isset( $webhook['created_at'] ) ) {
					$webhook['created_at'] = date( get_option( 'date_format' ), strtotime( $webhook['created_at'] ) ); //phpcs:ignore
				}
				if ( isset( $webhook['settings'] ) ) {
					$webhook['settings'] = maybe_unserialize( isset( $webhook['settings'] ) ? $webhook['settings'] : array() );
				}
				return $webhook;
			},
			$webhook['data']
		);

		if ( isset( $webhook ) ) {
			return $this->get_success_response( __( 'Query Successfull', 'mailmint-pro' ), 200, $webhook );
		}
		return $this->get_error_response( __( 'Failed to get data', 'mailmint-pro' ), 400 );
	}


	/**
	 * Get a specific segment data (by webhook id)
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function get_single( WP_REST_Request $request ) {
		// Get values from API.
		$params     = MrmCommon::get_api_params_values( $request );
		$webhook_id = isset( $params[ 'webhook_id' ] ) ? $params[ 'webhook_id' ] : '';

		$webhook_data             = WebHookModel::get( $webhook_id );
		$webhook_data['settings'] = maybe_unserialize( $webhook_data['settings'] );
		if ( ! empty( $webhook_data ) ) {
			return $this->get_success_response( __( 'Successfully fetched webhook data!', 'mailmint-pro' ), 200, $webhook_data );
		}

		return $this->get_error_response( __( 'Failed to fetch the webhook data!', 'mailmint-pro' ), 400 );
	}


	/**
	 * Delete a segement
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function delete_single( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$webhook_id = isset( $params['webhook_id'] ) ? $params['webhook_id'] : '';

		$success = WebHookModel::destroy( $webhook_id );
		if ( $success ) {
			return $this->get_success_response( __( 'Webhook has been deleted successfully', 'mailmint-pro' ), 200 );
		}
		return $this->get_error_response( __( 'Failed to delete', 'mailmint-pro' ), 400 );
	}


	/**
	 * Delete multiple groups
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function delete_all( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$webhook_ids = isset( $params['webhook_ids'] ) ? $params['webhook_ids'] : '';
		$success     = WebHookModel::destroy_all( $webhook_ids );
		if ( $success ) {
			return $this->get_success_response( __( 'Webhook has been deleted successfully', 'mailmint-pro' ), 200 );
		}

		return $this->get_error_response( __( 'Failed to delete', 'mailmint-pro' ), 400 );
	}

	/**
	 * Get all custom fields.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function get_custom_fields( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$fields = WebHookModel::get_custom_fields();
		if ( isset( $fields ) ) {
			return $this->get_success_response( __( 'Query Successfull', 'mailmint-pro' ), 200, $fields );
		}
		return $this->get_error_response( __( 'Failed to get data', 'mailmint-pro' ), 400 );
	}

}
