<?php
/**
 * Handles API requests related to WordPress categories, tags, and authors.
 *
 * Provides methods to retrieve and format categories, tags, and authors based on search terms.
 * This class extends the AdminBaseController to utilize common functionalities for admin API controllers.
 *
 * @author   WPFunnels Team
 * @package  MailMintPro\Mint\Admin\API\Controllers
 * @since    1.13.0
 */
namespace MailMintPro\Mint\Admin\API\Controllers;

use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use MRM\Common\MrmCommon;
use WP_REST_Request;
use WP_User_Query;

/**
 * WPController Class
 *
 * Handles API requests related to WordPress categories, tags, and authors.
 *
 * @since 1.13.0
 */
class WPController extends AdminBaseController {
	
    /**
     * Retrieve WordPress categories based on a search term.
     *
     * @param WP_REST_Request $request Request object used to generate the response.
     * @return array|WP_REST_Response|\WP_Error The response containing the categories data.
     * 
     * @since 1.13.0
     */
	public function get_wordpress_categories( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );

		// Check if the 'term' parameter is set.
        if ( isset( $params['term'] ) && !empty( $params['term'] ) ) {
            // Sanitize the search term.
            $term = sanitize_text_field( $params['term'] );
    
            // Query the categories.
            $terms = get_terms( array(
                'taxonomy'   => 'category',
                'name__like' => $term,
                'hide_empty' => false,
            ) );
    
            // Initialize an empty array for formatted categories.
            $categories = array();
    
            // Check if there are any matching categories.
            if ( !is_wp_error( $terms ) && !empty( $terms ) ) {
                foreach ( $terms as $term ) {
                    $categories[] = array(
                        'label' => $term->name,
                        'value' => $term->term_id,
                    );
                }
                return rest_ensure_response( array( 'data' => $categories, 'status' => 'success' ) );
            } else {
                // No categories found or an error occurred.
                return rest_ensure_response( array( 'data' => array(), 'status' => 'success' ) );
            }
        } else {
            // Missing or empty 'term' parameter.
            return rest_ensure_response( array( 'data' => array(), 'status' => 'success' ) );
        }
	}

    /**
     * Retrieve WordPress categories based on a search term.
     *
     * @param WP_REST_Request $request Request object used to generate the response.
     * @return array|WP_REST_Response|\WP_Error The response containing the categories data.
     * @since 1.13.0
     */
	public function get_wordpress_tags( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );

		// Check if the 'term' parameter is set.
        if ( isset( $params['term'] ) && !empty( $params['term'] ) ) {
            // Sanitize the search term.
            $term = sanitize_text_field( $params['term'] );
    
            // Query the tags.
            $terms = get_terms( array(
                'taxonomy'   => 'post_tag',
                'name__like' => $term,
                'hide_empty' => false,
            ) );
    
            // Initialize an empty array for formatted tags.
            $tags = array();
    
            // Check if there are any matching tags.
            if ( !is_wp_error( $terms ) && !empty( $terms ) ) {
                foreach ( $terms as $term ) {
                    $tags[] = array(
                        'label' => $term->name,
                        'value' => $term->term_id,
                    );
                }
                return rest_ensure_response( array( 'data' => $tags, 'status' => 'success' ) );
            } else {
                // No tags found or an error occurred.
                return rest_ensure_response( array( 'data' => array(), 'status' => 'success' ) );
            }
        } else {
            // Missing or empty 'term' parameter.
            return rest_ensure_response( array( 'data' => array(), 'status' => 'success' ) );
        }
	}

    /**
     * Retrieve WordPress authors based on a search term.
     *
     * @param WP_REST_Request $request Request object used to generate the response.
     * @return array|WP_REST_Response|\WP_Error The response containing the authors data.
     * 
     * @since 1.13.0
     */
	public function get_wordpress_authors( WP_REST_Request $request ) {
        $params = MrmCommon::get_api_params_values( $request );
    
        // Initialize an empty array for formatted authors
        $authors = array();
    
        // Query the authors
        $user_query = new WP_User_Query( array(
            'role__in'       => array( 'administrator', 'editor', 'author', 'contributor' ),
            'search'         => isset( $params['term'] ) ? '*' . esc_attr( $params['term'] ) . '*' : '',
            'search_columns' => array(
                'user_login',
                'user_nicename',
                'user_email',
                'display_name',
            ),
        ) );
    
        // Get the results
        $users = $user_query->get_results();

        // Check if there are any matching authors
        if ( !empty( $users ) ) {
            foreach ( $users as $user ) {
                $authors[] = array(
                    'label' => $user->display_name,
                    'value' => $user->ID,
                );
            }
            return rest_ensure_response( array( 'data' => $authors, 'status' => 'success' ) );
        } else {
            // No authors found or an error occurred
            return rest_ensure_response( array( 'data' => array(), 'status' => 'success' ) );
        }
    }

    /**
     * Retrieve WordPress admins emails based on a search term.
     *
     * @param WP_REST_Request $request Request object used to generate the response.
     * @return array|WP_REST_Response|\WP_Error The response containing the authors data.
     * 
     * @since 1.17.0
     */
    public function get_email( WP_REST_Request $request ) {
        $params = $request->get_params();
        $term = isset( $params['term'] ) ? sanitize_text_field( $params['term'] ) : '';
        $args = array(
            'fields'  => array( 'user_email', 'display_name' )
        );
        if ( ! empty( $term ) ) {
            $args['search'] = '*' . $term . '*';
            $args['search_columns'] = array( 'user_email', 'display_name' );
        }
        $admin_users = get_users( $args );
            $admin_emails = array();
        foreach ( $admin_users as $user ) {
            $admin_emails[] = array(
                'email' => $user->user_email
            );
        }
        return rest_ensure_response( array( 'data' => $admin_emails, 'status' => 'success' ) );
    }
    
}
