<?php

/**
 * Handles API requests related to WooCommerce coupons.
 *
 * Provides methods to retrieve and format categories, tags, and authors based on search terms.
 * This class extends the AdminBaseController to utilize common functionalities for admin API controllers.
 *
 * @author   WPFunnels Team
 * @package  MailMintPro\Mint\Admin\API\Controllers
 * @since    1.16.1
 */

namespace MailMintPro\Mint\Admin\API\Controllers;

use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use MRM\Common\MrmCommon;
use WP_REST_Request;

/**
 * WCController Class
 *
 * Handles API requests related to WooCommerce coupons.
 *
 * @since 1.16.1
 */
class WCController extends AdminBaseController{

    /**
     * Retrieves a list of WooCommerce coupons based on the search term.
     *
     * This function handles a REST API request to fetch published WooCommerce coupons
     * that match the provided search term.
     *
     * @param WP_REST_Request $request The REST API request object.
     * @return WP_REST_Response The response object containing the list of coupons.
     * 
     * @since 1.16.1
     */
    public function get_coupons (WP_REST_Request $request ){
        $params       = MrmCommon::get_api_params_values($request);
        $search       = isset($params['search']) ? $params['search'] : '';
        $coupon_codes = array();

        // Get all published coupons (post type 'shop_coupon').
        $args = array(
            'posts_per_page' => -1,
            'orderby'        => 'name',
            'order'          => 'asc',
            'post_type'      => 'shop_coupon',
            'post_status'    => 'publish',
            's'              => $search
        );

        $coupons = get_posts( $args );
        foreach ($coupons as $coupon) {
            if (! empty($coupon->post_title)) {
                $coupon_codes[] = [
                    'value'   => $coupon->ID,
                    'label' => $coupon->post_title,
                ];
            }
        }

        $response['success'] = true;
        $response['coupons'] = $coupon_codes;
        return rest_ensure_response($response);
    }
}
