<?php

/**
 * LinkTriggerController class for handling REST API routes related to link triggers.
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-07-10 11:03:17
 * @modify date 2024-07-10 11:03:17
 * @package /app/Admin/API/Controllers
 */
namespace MailMintPro\Mint\Admin\API\Controllers;

use MailMintPro\Mint\DataBase\Models\LinkTriggerModel;
use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use MRM\Common\MrmCommon;
use WP_REST_Request;
use WP_REST_Response;

/**
 * LinkTriggerController class for handling REST API routes related to link triggers.
 *
 * This class extends the AdminBaseController class and is responsible for handling the route callbacks
 * related to link triggers. It includes methods for creating, updating, retrieving, and deleting link triggers.
 * 
 * @since 1.15.0
 */
class LinkTriggerController extends AdminBaseController {

	/**
	 * Create or update link trigger.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 * @since 1.15.0
	 */
	public function create_or_update( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );

		$title       = isset( $params['title'] ) ? sanitize_text_field( $params['title'] ) : '';
		$description = isset( $params['description'] ) ? sanitize_text_field( $params['description'] ) : '';
		$url		 = isset( $params['url'] ) ? sanitize_text_field( $params['url'] ) : '';
		$data        = isset( $params['data'] ) ? $params['data'] : array();
		$data['url'] = $url;

		$link_trigger = array(
			'title'       => $title,
			'description' => $description,
			'data'        => maybe_serialize( $data ),
			'created_at'  => current_time( 'mysql', 1 ),
			'updated_at'  => current_time( 'mysql', 1 ),
			'created_by'  => get_current_user_id(),
		);

		if ( isset( $params['id'] ) ) {
			$result = LinkTriggerModel::update_link_trigger_data( $params['id'], $link_trigger );
		} else {
			$result = LinkTriggerModel::create( $link_trigger );
		}

		if ( $result ) {
			return new WP_REST_Response( array(
				'status'  => 'success',
				'id'      => $result,
				'message' => __('Link trigger has been saved successfully.', 'mailmint-pro'),
			) );
		}

		return new WP_REST_Response( array(
			'status'  => 'error',
			'message' => __('An error occurred while creating link triggers.', 'mailmint-pro'),
		), 500);
	}


	/**
     * Retrieves all link triggers with pagination and search functionality.
     *
     * @param WP_REST_Request $request The REST API request object.
     * 
     * @return WP_REST_Response The response containing the status, data, and message.
     * @since 1.15.0
     */
	public function get_all( WP_REST_Request $request ) {
        // Retrieve and sanitize API request parameters.
		$params     = MrmCommon::get_api_params_values( $request );
		$page       = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page   = isset( $params['per-page'] ) ? $params['per-page'] : 25;
		$offset     = ( $page - 1 ) * $per_page;
		$order_by   = isset( $params['order-by'] ) ? strtolower( $params['order-by'] ) : 'created_at';
		$order_type = isset( $params['order-type'] ) ? strtolower( $params['order-type'] ) : 'desc';
		$search     = isset( $params['search'] ) ? sanitize_text_field( $params['search'] ) : '';

        // Fetch all link triggers from the model using the provided parameters.
		$response = LinkTriggerModel::get_all( $offset, $per_page, $search, $order_by, $order_type );

		// Prepare created at as WP admin general settings date format.
		if ( isset( $response['triggers'] ) ) {
			$timezone = new \DateTimeZone(wp_timezone_string());

			$response['triggers'] = array_map(
				function ( $trigger )
				use ( $timezone )  {
					if (isset($trigger['created_at'])) {
						// Create the DateTime object from the created_at string.
						$created_at = new \DateTime($trigger['created_at']);

						// Adjust the DateTime object to the WordPress timezone.
						$created_at->setTimezone($timezone);

						// Convert DateTime to Unix timestamp.
						$created_at_timestamp = $created_at->getTimestamp();

						// Get the current time in WordPress timezone.
						$current_time = new \DateTime('now', $timezone);
						$current_time_timestamp = $current_time->getTimestamp();

						// Calculate the human-readable time difference.
						$trigger['created_at'] = human_time_diff($created_at_timestamp, $current_time_timestamp);
					}
					return $trigger;
				},
				$response['triggers']
			);
		}

        if ( is_wp_error( $response ) ) {
            return new WP_REST_Response( array(
                'status'  => 'error',
                'message' => __('An error occurred while retrieving link triggers.', 'mailmint-pro'),
            ), 500);
        }

        // Return a successful response with the retrieved data.
        return new WP_REST_Response( array(
            'status'  => 'success',
			'data'    => $response,
			'message' => __('Link triggers have been retrieved successfully.', 'mailmint-pro'),
        ) ); 
	}

	/**
	 * Retrieves a single link trigger by ID.
	 *
	 * @param WP_REST_Request $request The REST API request object.
	 * 
	 * @return WP_REST_Response The response containing the status, data, and message.
	 * @since 1.15.0
	 */
	public function get_single( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );
		$id     = isset( $params[ 'id' ] ) ? $params[ 'id' ] : '';

		$link_trigger         = LinkTriggerModel::get( $id );
		$link_trigger['data'] = isset( $link_trigger['data'] ) ? maybe_unserialize( $link_trigger['data'] ) : array();

		if ( is_wp_error( $link_trigger ) ) {
			return new WP_REST_Response(array(
				'status'  => 'error',
				'message' => __('An error occurred while retrieving link trigger.', 'mailmint-pro'),
			), 500);
		}

		// Return a successful response with the retrieved data.
		return new WP_REST_Response(array(
			'status'  => 'success',
			'data'    => $link_trigger,
			'message' => __('Link trigger has been retrieve successfully.', 'mailmint-pro'),
		));
	}

	/**
	 * Delete a single link trigger by ID.
	 *
	 * @param WP_REST_Request $request The REST API request object.
	 * 
	 * @return WP_REST_Response The response containing the status and message.
	 * @since 1.15.0
	 */
	public function delete_all( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		if ( isset( $params['ids'] ) ) {
			$success = LinkTriggerModel::destroy_all( $params['ids'] );
			if ( $success ) {
				return new WP_REST_Response(array(
					'status'  => 'success',
					'message' => __('Link triggers has been deleted successfully.', 'mailmint-pro'),
				));
			}
		}
	}
}
