<?php
/**
 * REST API Segment Controller
 *
 * Handles requests to the segments endpoint.
 *
 * @author   MRM Team
 * @category API
 * @package  MRM
 * @since    1.0.0
 */

namespace MailMintPro\Mint\Admin\API\Controllers;

use MailMintPro\Admin\API\Controllers\BaseController;
use MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts;
use Mint\MRM\DataBase\Models\ContactGroupModel;
use WP_REST_Request;
use Exception;
use Mint\MRM\Constants;
use Mint\MRM\DataStores\SegmentData;
use Mint\MRM\DataBase\Models\ContactModel;
use Mint\MRM\DataBase\Models\CustomFieldModel;
use MRM\Common\MrmCommon;
use WP_REST_Response;

/**
 * This is the main class that controls the segments feature. Its responsibilities are:
 *
 * - Create or update a segment
 * - Delete single or multiple segments
 * - Retrieve single or multiple segments
 * - Assign or removes segments from the contact
 *
 * @package Mint\MRM\Admin\API\Controllers
 */
class SegmentController extends BaseController {

	/**
	 * Create a new segment or update a existing segment
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function create_or_update( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		// Segment title validation.
		$title = isset( $params['title'] ) ? sanitize_text_field( $params['title'] ) : null;
		if ( empty( $title ) ) {
			return $this->get_error_response( __( 'Title is mandatory', 'mailmint-pro' ), 200 );
		}

		// Segment filters validation.
		if ( empty( $params['data'] ) || ( is_array( $params['data'] ) && empty( $params['data']['filters'] ) ) ) {
			return $this->get_error_response( __( 'Filters are mandatory.', 'mailmint-pro' ), 200 );
		}

		// Segment object create and insert or update to database.
		try {
			$segment = new SegmentData( $params );

			if ( isset( $params['segment_id'] ) ) {
				$success = ContactGroupModel::update( $segment, $params['segment_id'], 'segments' );
			} else {
				$success = ContactGroupModel::insert( $segment, 'segments' );
			}

			if ( $success ) {
				return $this->get_success_response( __( 'Segment has been saved successfully', 'mailmint-pro' ), 201, $success );
			}
			return $this->get_error_response( __( 'Failed to save', 'mailmint-pro' ), 200 );
		} catch ( Exception $e ) {
			return $this->get_error_response( __( 'Segment is not valid', 'mailmint-pro' ), 200 );
		}
	}


	/**
	 * Get segments for list views
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function get_all( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$page     = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page = isset( $params['per-page'] ) ? $params['per-page'] : 25;
		$offset   = ( $page - 1 ) * $per_page;

		$order_by   = isset( $params['order-by'] ) ? strtolower( $params['order-by'] ) : 'id';
		$order_type = isset( $params['order-type'] ) ? strtolower( $params['order-type'] ) : 'desc';

		// valid order by fields and types.
		$allowed_order_by_fields = array( 'title', 'created_at' );
		$allowed_order_by_types  = array( 'asc', 'desc' );

		// validate order by fields or use default otherwise.
		$order_by   = in_array( $order_by, $allowed_order_by_fields, true ) ? $order_by : 'id';
		$order_type = in_array( $order_type, $allowed_order_by_types, true ) ? $order_type : 'desc';

		// Segment Search keyword.
		$search = isset( $params['search'] ) ? sanitize_text_field( $params['search'] ) : '';

		$segments = ContactGroupModel::get_all( 'segments', $offset, $per_page, $search, $order_by, $order_type );

		// Count contacts groups.
		$segments['count_groups'] = array(
			'segments' => absint( isset( $segments['total_count'] ) ? $segments['total_count'] : '' ),
			'tags'     => ContactGroupModel::get_groups_count( 'tags' ),
			'contacts' => ContactModel::get_contacts_count(),
			'lists'    => ContactGroupModel::get_groups_count( 'lists' ),
		);

		$segments['data'] = array_map(
			function( $segment ) {
				$segment['data']        = maybe_unserialize( isset( $segment['data'] ) ? $segment['data'] : array() );
				$segment['description'] = isset( $segment['data']['description'] ) ? sanitize_text_field( $segment['data']['description'] ) : '';
				if ( isset( $segment['created_at'] ) ) {
					$segment['created_at'] = MrmCommon::date_time_format_with_core( $segment['created_at'] ); //phpcs:ignore
				}
				return $segment;
			},
			$segments['data']
		);

		if ( isset( $segments ) ) {
			return $this->get_success_response( __( 'Query Successfull', 'mailmint-pro' ), 200, $segments );
		}
		return $this->get_error_response( __( 'Failed to get data', 'mailmint-pro' ), 400 );
	}


	/**
	 * Get a specific segment data (by segment id)
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function get_single( WP_REST_Request $request ) {
		// Get values from API.
		$params     = MrmCommon::get_api_params_values( $request );
		$segment_id = isset( $params[ 'segment_id' ] ) ? $params[ 'segment_id' ] : '';

		$segment_data = FilterSegmentContacts::get_segment( $segment_id );

		if ( ! empty( $segment_data ) ) {
			return $this->get_success_response( __( 'Successfully fetched segment contacts!', 'mailmint-pro' ), 200, $segment_data );
		}

		return $this->get_error_response( __( 'Failed to fetch the segment contacts!', 'mailmint-pro' ), 400 );
	}


	/**
	 * Delete a segement
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function delete_single( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$segment_id = isset( $params['segment_id'] ) ? $params['segment_id'] : '';

		$success = ContactGroupModel::destroy( $segment_id );
		if ( $success ) {
			return $this->get_success_response( __( 'Segment has been deleted successfully', 'mailmint-pro' ), 200 );
		}

		return $this->get_error_response( __( 'Failed to delete', 'mailmint-pro' ), 400 );
	}


	/**
	 * Delete multiple groups
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function delete_all( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$segment_ids = isset( $params['segment_ids'] ) ? $params['segment_ids'] : '';
		$success     = ContactGroupModel::destroy_all( $segment_ids );
		if ( $success ) {
			return $this->get_success_response( __( 'Segments has been deleted successfully', 'mailmint-pro' ), 200 );
		}

		return $this->get_error_response( __( 'Failed to delete', 'mailmint-pro' ), 400 );
	}


	/**
	 * Get specific segment contacts
	 *
	 * @param WP_REST_Request $request WP_REST_Request.
	 *
	 * @return array|\WP_Error
	 * @since 1.0.0
	 */
	public function get_preview_contacts( WP_REST_Request $request ) {
		// Get values from API.
		$params          = MrmCommon::get_api_params_values( $request );
		$segment_filters = isset( $params[ 'filters' ] ) ? $params[ 'filters' ] : array();

		$page       = isset( $params[ 'page' ] ) ? $params[ 'page' ] : 1;
		$per_page   = isset( $params[ 'per-page' ] ) ? $params[ 'per-page' ] : 10;
		$count_only = isset( $params[ 'count-only' ] ) ? $params[ 'count-only' ] : false;
		$offset     = ( $page - 1 ) * $per_page;
		$order_by   = isset( $params[ 'order-by' ] ) ? strtolower( $params[ 'order-by' ] ) : 'email';
		$order_type = isset( $params[ 'order-type' ] ) ? strtolower( $params[ 'order-type' ] ) : 'ASC';

		$where_query      = $segment_filters ? FilterSegmentContacts::get_filter_where_query( $segment_filters ) : false;
		$segment_contacts = $where_query ? FilterSegmentContacts::filter_contacts( $where_query, $offset, $per_page, $order_by, $order_type, $count_only ) : false;

		if ( !$count_only && is_array( $segment_contacts ) && !empty( $segment_contacts ) ) {
			foreach ( $segment_contacts[ 'data' ] as $key => $contact ) {
				$contact_id                         = isset( $contact['id'] ) ? $contact['id'] : 0;
				$meta_fields                        = ContactModel::get_meta( $contact_id );
				$contact                            = array_merge( $contact, $meta_fields );
				$contact                            = ContactGroupModel::get_tags_to_contact( $contact );
				$contact                            = ContactGroupModel::get_lists_to_contact( $contact );
				$segment_contacts[ 'data' ][ $key ] = $contact;
			}
		}

		if ( !empty( $segment_contacts ) ) {
			return $this->get_success_response( __( 'Successfully fetched segment contacts!', 'mailmint-pro' ), 200, $segment_contacts );
		}

		return $this->get_error_response( __( 'No contacts matched.', 'mailmint-pro' ), 201 );
	}

	/**
	 * Retrieve a list of available field types, including default, primary, and WooCommerce-specific fields.
	 *
	 * This function compiles a list of available field types by combining default, primary, and WooCommerce-specific fields
	 * if WooCommerce is active.
	 *
	 * @access public
	 *
	 * @return WP_REST_Response The response containing an array of available field types.
	 * @since 1.5.5
	 */
	public function retrieve_available_field_types() {
		$field_types = $this->get_default_field_types();
		$field_types = $this->add_primary_fields( $field_types );
		$field_types = $this->add_custom_fields( $field_types );

		if ( MrmCommon::is_wc_active() && MrmCommon::is_mailmint_pro_active() ) {
			$field_types = $this->add_WooCommerce_fields( $field_types );
		}

		$response['status']  = 'success';
		$response['message'] = __( 'Field types have been retrieved successfully.', 'mailmint-pro' );
		$response['data']    = $field_types;
		return new WP_REST_Response( $response );
	}

	/**
	 * Add WooCommerce-specific fields to the list of available field types.
	 *
	 * This function appends WooCommerce-specific fields to the array of available field types.
	 *
	 * @access public
	 *
	 * @param array $field_types The array of available field types before adding WooCommerce fields.
	 * @return array The updated array of available field types including WooCommerce fields.
	 *
	 * @since 1.5.5
	 */
	public function add_WooCommerce_fields( $field_types ) {
		return array_merge(
			$field_types,
			array(
				'WooCommerce' => array(
					array(
						'label' => __('First Order Date', 'mailmint-pro'),
						'type'  => 'date',
						'value' => 'first_order_date',
					),
					array(
						'label' => __('Last Order Date', 'mailmint-pro'),
						'type'  => 'date',
						'value' => 'last_order_date',
					),
					array(
						'label' => __('Purchased Products', 'mailmint-pro'),
						'type'  => 'wc_select',
						'value' => 'purchased_products',
					),
					array(
						'label' => __('Purchased Tags', 'mailmint-pro'),
						'type'  => 'wc_select',
						'value' => 'purchased_tags',
					),
					array(
						'label' => __('Purchased Categories', 'mailmint-pro'),
						'type'  => 'wc_select',
						'value' => 'purchased_categories',
					),
					array(
						'label' => __('Has Made a Purchase', 'mailmint-pro'),
						'type'  => 'boolean',
						'value' => 'has_made_a_purchase',
					),
					array(
						'label' => __('Total Order Count', 'mailmint-pro'),
						'type'  => 'number',
						'value' => 'total_order_count',
					),
					array(
						'label' => __('Total Order Value', 'mailmint-pro'),
						'type'  => 'number',
						'value' => 'total_order_value',
					),
					array(
						'label' => __('Average Order Value', 'mailmint-pro'),
						'type'  => 'number',
						'value' => 'average_order_value',
					),
					array(
						'label' => __('Used Coupons', 'mailmint-pro'),
						'type'  => 'wc_select',
						'value' => 'used_coupons',
					),
				),
			)
		);
	}

	/**
	 * Add custom fields to the list of available field types.
	 *
	 * This function adds custom fields retrieved from the CustomFieldModel to the array of available field types.
	 *
	 * @access public
	 *
	 * @param array $field_types The array of available field types before adding custom fields.
	 * @return array The updated array of available field types including custom fields.
	 *
	 * @since 1.5.5
	 */
	public function add_custom_fields( $field_types ) {
		$customer_fields = CustomFieldModel::get_custom_fields_to_segment();

		if ( empty( $customer_fields ) ) {
			return $field_types;
		}
		$customer_fields = array_map(
			function ( $customer_field ) {
				$meta                      = maybe_unserialize( $customer_field['meta'] );
				$customer_field['label']   = isset( $meta['label'] ) ? $meta['label'] : '';
				$customer_field['options'] = isset( $meta['options'] ) ? $meta['options'] : '';
				unset( $customer_field['meta'] );
				return $customer_field;
			},
			$customer_fields
		);

		$field_types['Custom'] = $customer_fields;
		return $field_types;
	}

	/**
	 * Add primary fields to the list of available field types.
	 *
	 * This function adds primary fields retrieved from the mint_contact_primary_fields option to the array
	 * of available field types. The fields are categorized based on their predefined segment group.
	 *
	 * @access public
	 *
	 * @param array $field_types The array of available field types before adding primary fields.
	 * @return array The updated array of available field types including primary fields.
	 *
	 * @since 1.5.5
	 */
	public function add_primary_fields( $field_types ) {
		$fields = get_option( 'mint_contact_primary_fields', Constants::$primary_contact_fields );
		$fields = array_merge( ...array_values( $fields ) );

		$primary_fields = array();
		foreach ( $fields as $item ) {
			$slug  = isset( $item['slug'] ) ? $item['slug'] : '';
			$label = isset( $item['meta']['label'] ) ? $item['meta']['label'] : '';
			$type  = isset( $item['segment'] ) ? $item['segment'] : 'address';

			foreach ( Constants::$segment_primary_field_slugs as $group => $slugs ) {
				if ( in_array( $slug, $slugs, true ) ) {
					$primary_fields[ $group ][] = array(
						'label' => $label,
						'type'  => $type,
						'value' => $slug,
					);
					break;
				}
			}
		}
		return array_merge( $primary_fields, $field_types );
	}

	/**
	 * Get the default field types.
	 *
	 * This function returns an array of default field types with the 'Contact Segment' group.
	 * The default field types are retrieved from the Constants::$static_primary_field_types array.
	 *
	 * @access public
	 *
	 * @return array An array of default field types with the 'Contact Segment' group.
	 *
	 * @since 1.5.5
	 */
	public function get_default_field_types() {
		return array(
			'Contact Segment' => Constants::$static_primary_field_types,
		);
	}
}
