<?php

/**
 * Handles API requests related to form export and import.
 *
 * Provides methods to export form data in JSON format and import form data from a JSON file.
 *
 * @author   WPFunnels Team
 * @package  MailMintPro\Mint\Admin\API\Controllers
 * @since    1.17.0
 */

namespace MailMintPro\Mint\Admin\API\Controllers;

use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use Mint\MRM\Utilities\Helper\PermissionManager;
use Mint\Utilities\Arr;
use MRM\Common\MrmCommon;
use WP_REST_Request;
use WP_User_Query;

/**
 * RoleManagementController Class
 *
 * Handles API requests related to form export and import.
 *
 * @since 1.17.0
 */
class RoleManagementController extends AdminBaseController{

    /**
     * Retrieve and paginate WordPress roles with additional metadata.
     *
     * This function handles a REST API request to retrieve WordPress roles.
     *
     * @param WP_REST_Request $request The REST API request object.
     *
     * @return WP_REST_Response The response object containing the roles data.
     * @since 1.17.0
     */
    public function get_roles( WP_REST_Request $request ) {
        $params   = MrmCommon::get_api_params_values( $request );
        $page     = isset( $params['page'] ) ? $params['page'] : 1;
        $per_page = isset( $params['per-page'] ) ? $params['per-page'] : 5;
    
        // Get all roles.
        $wp_roles = wp_roles();
        $roles    = $wp_roles->roles;
    
        // Get the total number of roles
        $total_roles = count( $roles );
    
        // Calculate pagination offsets
        $offset = ( $page - 1 ) * $per_page;
        $paginated_roles = array_slice( $roles, $offset, $per_page, true );
    
        // Enhance roles with additional data
        $roles_with_meta = array_map(function($role_key, $role_data) {   
            // Return enhanced role data
            return [
                'key'         => $role_key,
                'name'        => $role_data['name'],
                'permissions' => PermissionManager::get_role_permissions($role_key)
            ];
        }, array_keys($paginated_roles), $paginated_roles);
    
        // Prepare the response
        $response = array(
            'status'  => 'success',
            'message' => __( 'Roles have been retrieved successfully.', 'mailmint-pro' ),
            'data'    => array(
                'roles'       => array_values( $roles_with_meta ),
                'count'       => $total_roles,
                'total_pages' => ceil( $total_roles / $per_page ),
            ),
        );
    
        return rest_ensure_response( $response );
    }

    /**
     * Retrieve users and their capabilities for a given role.
     *
     * This function handles a REST API request to retrieve users and their capabilities for a given role.
     *
     * @param WP_REST_Request $request The REST API request object.
     *
     * @return WP_REST_Response The response object containing the role details.
     * @since 1.17.0
     */
    public function get_users_and_capabilities( WP_REST_Request $request ){
        $params   = MrmCommon::prepare_request_params( $request );
        $role_key = isset( $params['key'] ) ? $params['key'] : '';

        if ( empty( $role_key ) ) {
            return rest_ensure_response( [
                'status'  => 'failed',
                'message' => __( 'Role key is required.', 'mrm' ),
            ] );
        }
        
        $wp_roles     = wp_roles();
        $roles        = $wp_roles->roles;
        $role_details = $roles[ $role_key ];

        $response = [
            'status' => 'success',
            'message' => __( 'Role details have been retrieved successfully.', 'mailmint-pro' ),
            'data' => [
                'name'         => $role_details['name'],
                'capabilities' => $role_details['capabilities'],
            ],
        ];
        return rest_ensure_response( $response );
    }

    public function assign_permissions( WP_REST_Request $request ){
        $params = MrmCommon::prepare_request_params( $request );
        
        $role        = Arr::get( $params, 'role' );
        $permissions = Arr::get( $params, 'permissions', array() );
        // Filter and keep only keys with truthy values
        $permissions = array_keys(array_filter($permissions));

        // Validate input.
        if ( empty( $role ) || empty( $permissions ) ) {
            return rest_ensure_response( array(
                'status'  => 'failed',
                'message' => 'Permissions are required.',
            ) );
        }

        $role_object = get_role($role);
        if (!$role_object) {
            return rest_ensure_response(array(
                'status' => 'failed',
                'message' => 'Invalid role.',
            ));
        }

        PermissionManager::attach_permissions($role_object, $permissions);

        return rest_ensure_response(array(
            'status' => 'success',
            'message' => 'Permissions added to the role successfully.',
        ));
    }
}
