<?php
/**
 * Mail Mint Webhooks functions
 *
 * This class handles Webhook Incomming functions for each WP Actions.
 *
 * @author   Mail Mint Team
 * @category Action
 * @package  MRM
 * @since    1.0.0
 */

namespace MailMintPro\Mint\Admin\API\Controllers;

use MailMintPro\Admin\API\Controllers\BaseController;
use MailMintPro\Mint\DataBase\Models\LeadMagnetModel;
use MRM\Common\MrmCommon;
use WP_REST_Request;
/**
 * WebHooks Controller.
 */
class LeadMagnetController extends BaseController {

	/**
	 * Webhook object arguments
	 *
	 * @var object
	 * @since 1.0.0
	 */
	public $args;
	/**
	 * Create a new webhook or update a existing segment
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function create_or_update( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );

		$title         = isset( $params['title'] ) ? sanitize_text_field( $params['title'] ) : '';
		$description   = isset( $params['description'] ) ? sanitize_text_field( $params['description'] ) : '';
		$hash          = isset( $params['hash'] ) ? sanitize_text_field( $params['hash'] ) : '';
		$attachment_id = isset( $params['attachment_id'] ) ? sanitize_text_field( $params['attachment_id'] ) : null;
		$url		   = isset( $params['url'] ) ? sanitize_text_field( $params['url'] ) : '';

		$magnet = array(
			'title'       => $title,
			'description' => $description,
			'hash'        => $hash,
			'attachment_id' => $attachment_id,
			'url' => $url,
		);

		if ( isset( $params['magnet_id'] ) ) {
			$success = LeadMagnetModel::update( $magnet, $params['magnet_id'] );
		} else {
			$success = LeadMagnetModel::insert( $magnet );
		}

		if ( $success ) {
			return $this->get_success_response( __( 'Lead Magnet has been saved successfully', 'mailmint-pro' ), 201, $success );
		}
		return $this->get_error_response( __( 'Failed to save', 'mailmint-pro' ), 200 );
	}


	/**
	 * Get webhook for list views
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array|WP_REST_Response|\WP_Error
	 */
	public function get_all( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$page     = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page = isset( $params['per-page'] ) ? $params['per-page'] : 25;
		$offset   = ( $page - 1 ) * $per_page;

		$order_by   = isset( $params['order-by'] ) ? strtolower( $params['order-by'] ) : 'id';
		$order_type = isset( $params['order-type'] ) ? strtolower( $params['order-type'] ) : 'desc';

		// Segment Search keyword.
		$search = isset( $params['search'] ) ? sanitize_text_field( $params['search'] ) : '';

		$magnets = LeadMagnetModel::get_all( $offset, $per_page, $search, $order_by, $order_type );

		if ( isset( $magnets ) ) {
			return $this->get_success_response( __( 'Query Successfull', 'mailmint-pro' ), 200, $magnets );
		}
		return $this->get_error_response( __( 'Failed to get data', 'mailmint-pro' ), 400 );
	}


	/**
	 * Get single lead magnet.
	 * 
	 * @param WP_REST_Request $request Request object used to generate the response.
	 * 
	 * @return array|WP_REST_Response|\WP_Error
	 * @since 1.11.0
	 */
	public function get_single( WP_REST_Request $request ) {
		// Get values from API.
		$params    = MrmCommon::get_api_params_values( $request );
		$magnet_id = isset( $params[ 'magnet_id' ] ) ? $params[ 'magnet_id' ] : '';

		$magnet = LeadMagnetModel::get( $magnet_id );
		if ( ! empty( $magnet ) ) {
			return $this->get_success_response( __( 'Lead magnet has been retrieve successfully.', 'mailmint-pro' ), 200, $magnet );
		}

		return $this->get_error_response( __( 'Failed to fetch the lead magnet data!', 'mailmint-pro' ), 400 );
	}


	/**
	 * Delete a lead magnet.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public function delete_single( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		$magnet_id = isset( $params['magnet_id'] ) ? $params['magnet_id'] : '';

		$success = LeadMagnetModel::destroy( $magnet_id );
		if ( $success ) {
			return $this->get_success_response( __( 'Lead magnet has been deleted successfully.', 'mailmint-pro' ), 200 );
		}
		return $this->get_error_response( __( 'Failed to delete', 'mailmint-pro' ), 400 );
	}

	public function delete_all(WP_REST_Request $request) {}
}
