<?php

namespace MintMailPro\Mint\Admin\API\Controllers\Integration;

use Exception;
use MailMintPro\App\Utilities\Helper\Integration;
use Mint\MRM\DataBase\Models\ContactGroupModel;
use Mint\MRM\DataBase\Models\ContactModel;
use MRM\Common\MrmCommon;
use WP_REST_Request;
use WP_REST_Response;

/**
 * This is the main class that controls the general setting feature. Its responsibilities are:
 *
 * - Create or update general settings
 * - Retrieve general settings from options table
 *
 * @package Mint\MRM\Admin\API\Controllers
 */
class IntegrationController extends IntegrationBaseController {

    /**
     * Update General global settings into wp_options table
     *
     * @param WP_REST_Request $request Request object used to generate the response.
     * @return array|WP_REST_Response
     * @since 1.0.0
     */
    public function connect( WP_REST_Request $request ) {
        $params = MrmCommon::get_api_params_values( $request );

        if( is_array( $params ) && ! empty( $params ) ) {
            $integration = isset( $params['integration'] ) ? $params['integration'] : '';

            // ZeroBounce Integration.
            if ( 'zero_bounce' === $integration ) {
                $api_key = isset( $params['api_key'] ) ? $params['api_key'] : '';
                $email   = isset( $params['email'] ) ? $params['email'] : '';
                $lists   = isset( $params['lists'] ) ? $params['lists'] : array();

                $response = Integration::handle_zero_bounce_request( $api_key, $email );

                if( 200 === $response['response'] && ( isset($response['body']['status']) && 'invalid' !== $response['body']['status'] ) ){
                    $settings    = get_option( '_mint_integration_settings', $this->get_integration_default_settings() );
                    $zero_bounce = isset( $settings['zero_bounce'] ) ? $settings['zero_bounce'] : array();

                    $zero_bounce['api_key']       = $api_key;
                    $zero_bounce['email_address'] = $email;
                    $zero_bounce['is_integrated'] = true;
                    $zero_bounce['lists']         = $lists;

                    $settings['zero_bounce'] = $zero_bounce;
                    update_option( '_mint_integration_settings', $settings );
                    return $this->get_success_response( __( 'ZeroBounce have been successfully connected.', 'mailmint-pro' ) );
                }
            }

            if( 'twilio' === $integration ) {
                $account_sid = isset( $params['account_sid'] ) ? $params['account_sid'] : '';
                $auth_token  = isset( $params['auth_token'] ) ? $params['auth_token'] : '';
                $from_number = isset( $params['from_number'] ) ? $params['from_number'] : '';

                try {
                    $url     = "https://api.twilio.com/2010-04-01/Accounts/" . $account_sid . ".json";
                    $headers = array(
                        'Content-Type'  => 'application/x-www-form-urlencoded',
                        'Authorization' => 'Basic ' . base64_encode( $account_sid . ':' . $auth_token ),
                    );

                    $response = MM()->wp_remote_request_handler->make_wp_requests( $url, array(), $headers, 1 );

                    if ( ! is_wp_error( $response ) ) {
                        if( 200 === $response['response'] && 'active' === $response['body']['status'] ) {
                            $settings = get_option( '_mint_integration_settings', $this->get_integration_default_settings() );
                            $twilio   = isset( $settings['twilio'] ) ? $settings['twilio'] : array();
    
                            $twilio['account_sid']   = $account_sid;
                            $twilio['auth_token']    = $auth_token;
                            $twilio['from_number']   = $from_number;
                            $twilio['is_integrated'] = true;
    
                            $settings['twilio'] = $twilio;
                            update_option( '_mint_integration_settings', $settings );
                            return $this->get_success_response( __( 'Twilio have been successfully connected.', 'mailmint-pro' ) );
                        }
                    }
                    
                } catch ( Exception $e ) {
                    return $this->get_error_response( __( 'Integration failed. Please check your details and try again.', 'mailmint-pro' ) );
                }
            }

            if( 'open_ai' === $integration ) {
                $secret_key = isset( $params['secret_key'] ) ? $params['secret_key'] : '';

                $url = "https://api.openai.com/v1/chat/completions";

                // Prepare the headers.
                $headers = array(
                    'Content-Type'  => 'application/json',
                    'Authorization' => 'Bearer ' . $secret_key,
                );

                // The data to send in the request.
                $data = array(
                    'model' => 'gpt-3.5-turbo',
                    'messages' => array(
                        array(
                            'role'    => 'system',
                            'content' => 'You are a helpful assistant.'
                        )
                    ),
                    'max_tokens'  => 150,
                    'temperature' => 0.7
                );

                $response = MM()->wp_remote_request_handler->make_wp_requests($url, wp_json_encode($data), $headers, 2);
                if (200 === $response['response']) {
                    $settings = get_option('_mint_integration_settings', $this->get_integration_default_settings());
                    $open_ai  = isset($settings['open_ai']) ? $settings['open_ai'] : array();

                    $open_ai['secret_key']    = $secret_key;
                    $open_ai['is_integrated'] = true;

                    $settings['open_ai'] = $open_ai;
                    update_option('_mint_integration_settings', $settings);
                    return $this->get_success_response(__('OpenAI has been successfully connected.', 'mailmint-pro'));
                } else {
                    return $this->get_error_response(__($response['body']['error']['message'], 'mailmint-pro'));
                }
            }
        }
        return $this->get_error_response( __( 'Integration failed. Please check your details and try again.', 'mailmint-pro' ) );
    }

    /**
     * Get General global settings from wp_option table
     *
     * @param WP_REST_Request $request Request object used to generate the response.
     * @return array|WP_REST_Response
     * @since 1.0.0
     */
    public function get( WP_REST_Request $request ) {
        $settings = get_option( '_mint_integration_settings', $this->get_integration_default_settings() );

        $old_secret_key = get_option('_mint_open_ai_key', '');

        if( empty( $settings['open_ai']['secret_key'] ) && !empty( $old_secret_key ) ){
            $settings['open_ai']['secret_key']    = $old_secret_key;
            $settings['open_ai']['is_integrated'] = true;
            update_option('_mint_integration_settings', $settings);
        }

        return $this->get_success_response_data( $settings );
    }

    public function disconnect( WP_REST_Request $request ) {
        $params      = MrmCommon::get_api_params_values( $request );
        $integration = isset( $params['integration'] ) ? $params['integration'] : '';
        $settings    = get_option( '_mint_integration_settings', $this->get_integration_default_settings() );

        if( 'zero_bounce' === $integration ) {
            $zero_bounce = isset( $settings['zero_bounce'] ) ? $settings['zero_bounce'] : array();

            $zero_bounce['is_integrated'] = false;

            $settings['zero_bounce'] = $zero_bounce;
            update_option( '_mint_integration_settings', $settings );
            return $this->get_success_response( __( 'ZeroBounce have been successfully disconnected.', 'mailmint-pro' ) );
        }

        if( 'twilio' === $integration ){
            $twilio = isset( $settings['twilio'] ) ? $settings['twilio'] : array();

            $twilio['is_integrated'] = false;

            $settings['twilio'] = $twilio;
            update_option( '_mint_integration_settings', $settings );
            return $this->get_success_response( __( 'Twilio have been successfully disconnected.', 'mailmint-pro' ) );
        }

        if ( 'openai' === $integration ) {
            $open_ai = isset( $settings['open_ai'] ) ? $settings['open_ai'] : array();

            $open_ai['is_integrated'] = false;

            $settings['open_ai'] = $open_ai;
 
            update_option('_mint_integration_settings', $settings);
            return $this->get_success_response(__('OpenAI have been successfully disconnected.', 'mailmint-pro'));
        }
        
    }

    public function apply_zero_bounce( WP_REST_Request $request ) {
        // Get API parameters from the request object and filter them.
		$params      = MrmCommon::get_api_params_values( $request );
        $settings    = get_option( '_mint_integration_settings', $this->get_integration_default_settings() );
        $zero_bounce = isset( $settings['zero_bounce'] ) ? $settings['zero_bounce'] : array();

        if( $zero_bounce && $zero_bounce['is_integrated'] ){
            $lists = $zero_bounce['lists'];
            $offset = isset( $params['offset'] ) ? $params['offset'] : 0;
            $all_contacts = ContactModel::get_all( $offset, 500, '' );
			$contacts     = isset( $all_contacts['data'] ) ? $all_contacts['data'] : array();

            foreach( $contacts as $contact ){
                $email = isset( $contact['email'] ) ? $contact['email'] : '';
                $ch = curl_init();

                curl_setopt($ch, CURLOPT_URL, 'https://api.zerobounce.net/v2/validate?api_key='.$zero_bounce['api_key'].'&email='.$email.'&ip_address=null');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

                $output = curl_exec($ch);

                curl_close($ch);

                $response = json_decode($output, true);

                if( isset( $response['status'] ) && 'valid' !== $response['status']){
                    $success = ContactGroupModel::set_lists_to_contact( $lists, $contact['id'] );
                }
            }

            return new WP_REST_Response(
                array(
                    'status'  => 'success',
                    'data'    => array( 'offset' => $offset + (int) 500 ),
                    'message' => __( 'ZeroBounce have been successfully applied.', 'mailmint-pro' ),
                )
            );
        }
        return $this->get_error_response( __( 'ZeroBounce have not been integrated yet.', 'mailmint-pro' ) );
    }

    public function send_test_message ( WP_REST_Request $request ) {
        // Get API parameters from the request object and filter them.
        $params = MrmCommon::get_api_params_values( $request );
        
        $message = isset( $params['message'] ) ? $params['message'] : array();
        $number  = isset( $message['phone_number'] ) ? $message['phone_number'] : '';
        $body    = isset( $message['body'] ) ? $message['body'] : '';
        $type    = isset( $message['type'] ) ? $message['type'] : '';

        $settings = get_option( '_mint_integration_settings', $this->get_integration_default_settings() );
        $twilio   = isset( $settings['twilio'] ) ? $settings['twilio'] : array();

        if( $twilio && $twilio['is_integrated'] ){
            $account_sid = $twilio['account_sid'];
            $auth_token  = $twilio['auth_token'];
            $from_number = $twilio['from_number'];

            try {
                $to   = $number;
                $from = $from_number;
                if ('whatsapp' === $type) {
                    $to = "whatsapp:".$number;
                    $from = "whatsapp:".$from_number;
                }

                $url = "https://api.twilio.com/2010-04-01/Accounts/$account_sid/Messages.json";

                $args = array(
                    'method' => 'POST',
                    'headers' => array(
                        'Content-Type'  => 'application/x-www-form-urlencoded',
                        'Authorization' => 'Basic ' . base64_encode($account_sid . ':' . $auth_token),
                    ),
                    'body' => array(
                        'From' => $from,
                        'To'   => $to,
                        'Body' => $body,
                    ),
                );
        
                $response = wp_remote_request( $url, $args );

                if ( !is_wp_error( $response ) ) {
                    $body = wp_remote_retrieve_body( $response );

                    if ( $this->is_json( $body ) ) {
                        $body = json_decode( $body, true );
                    }

                    $body = maybe_unserialize( $body );
                    if( 'queued' === $body['status'] ) {
                        return new WP_REST_Response(
                            array(
                                'status'  => 'success',
                                'message' => __( 'Test message have been successfully sent.', 'mailmint-pro' ),
                            )
                        );
                    } else {
                        return new WP_REST_Response(
                            array(
                                'status'  => 'failed',
                                'message' => $body['message'],
                            )
                        );
                    }
                }
            } catch ( Exception $e ) {
                return new WP_REST_Response(
                    array(
                        'status'  => 'failed',
                        'message' => $e->getMessage(),
                    )
                );
            }
        }
        return new WP_REST_Response(
            array(
                'status'  => 'failed',
                'message' => __( 'Twilio have not been integrated yet.', 'mailmint-pro' ),
            )
        );
    }

    public function is_json( $string ) {
		json_decode( $string );

		return ( json_last_error() === JSON_ERROR_NONE );
	}
}
