<?php

/**
 * Handles API requests related to form export and import.
 *
 * Provides methods to export form data in JSON format and import form data from a JSON file.
 *
 * @author   WPFunnels Team
 * @package  MailMintPro\Mint\Admin\API\Controllers
 * @since    1.17.0
 */

namespace MailMintPro\Mint\Admin\API\Controllers;

use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use Mint\MRM\DataBase\Models\FormModel;
use Mint\MRM\DataStores\FormData;
use MRM\Common\MrmCommon;
use WP_REST_Request;

/**
 * FormController Class
 *
 * Handles API requests related to form export and import.
 *
 * @since 1.17.0
 */
class FormController extends AdminBaseController{

    /**
     * Exports form data in JSON format.
     *
     * This function handles a REST API request to export form data in JSON format.
     *
     * @param WP_REST_Request $request The REST API request object.
     * @return WP_REST_Response The response object containing the list of coupons.
     * 
     * @since 1.17.0
     */
    public function export_form( WP_REST_Request $request ){
        $params  = MrmCommon::get_api_params_values( $request );
		$form_id = $params['id'] ?? 0;
		$form    = FormModel::get( $form_id );	
		if ( empty( $form ) ) {
			return $this->get_error_response( __( 'Failed to retrieve form data.', 'mrm' ), 400 );
		}

		$response = array(
            'form'    => !empty( $form ) ? $form : array(),
            'status'  => 'success',
            'message' => 'Form has been exported successfully.',
        );
        return rest_ensure_response($response);
    }

    /**
     * Imports form data from a JSON file.
     *
     * This function handles a REST API request to import form data from a JSON file.
     *
     * @param WP_REST_Request $request The REST API request object.
     * @return WP_REST_Response The response object containing the list of coupons.
     * 
     * @since 1.17.0
     */
    public function import_form( WP_REST_Request $request ){
        $params = MrmCommon::prepare_request_params( $request );
		if( empty($params['file']) || empty($params['file']['tmp_name']) || empty($params['file']['type'] || 'application/json' !== $params['file']['type']) ){
            return rest_ensure_response(
                array(
                    'status'  => 'failed',
                    'message' => __( 'Failed to import Form. JSON file is required.', 'mailmint-pro' ),
                )
            );
        }

		$temp_file_name = $params['file']['tmp_name'];
        $json_data      = file_get_contents($temp_file_name);
        $original_form  = json_decode($json_data, true);

		if(!isset($original_form['title']) || !isset($original_form['meta_fields']) || !isset($original_form['form_body'])){
            return rest_ensure_response(
                array(
                    'status'  => 'failed',
                    'message' => __( 'Failed to import Form. Invalid JSON file.', 'mailmint-pro' ),
                )
            );
        }

		$original_form['status']        = 'draft'; 
		$original_form['form_position'] = isset($original_form['form_position']) ? $original_form['form_position'] : array();
		$original_form['group_ids']     = isset($original_form['group_ids']) ? $original_form['group_ids'] : array();

		$data    = new FormData( $original_form );
		$form_id = FormModel::insert( $data, 'forms' );

		if ( !empty( $form_id )) {
			return rest_ensure_response(
                array(
                    'status'  => 'success',
                    'message' => __( 'Form has been imported successfully.', 'mailmint-pro' ),
                    'data'    => array(
                        'form_id' => $form_id
                    ),
                )
            );
		}

		return $this->get_error_response( __( 'Failed to import form data.', 'mailmint-pro' ), 400 );
    }
}
