<?php
/**
 * Mail Mint
 *
 * @author [MRM Team]
 * @email [support@rextheme.com]
 * @create date 2022-08-09 11:03:17
 * @modify date 2022-08-09 11:03:17
 * @package /app/API/Controllers
 */

namespace MintMailPro\Mint\Admin\API\Controllers;

use WP_REST_Request;
use MRM\Common\MrmCommon;
use WP_Query;
use WP_REST_Controller;

/**
 * Manages Analytics API callbacks
 *
 * @package /app/API/Controllers
 * @since 1.0.0
 */
class EmailBuilderController extends WP_REST_Controller {

	/**
	 * Get latest published product by category ID
	 *
	 * @param WP_REST_Request $request WP Rest Request object.
	 *
	 * @return WP_REST_Response
	 * @since 1.2.6
	 */
	public function get_latest_published_products_by_category( WP_REST_Request $request ) {
		$params      = MrmCommon::get_api_params_values( $request );
		$category_id = isset( $params['category_id'] ) ? $params['category_id'] : 0;
		$quantity    = isset( $params['quantity'] ) ? $params['quantity'] : 0;

		$all_products = array();

		if ( !$quantity ) {
			return rest_ensure_response( $all_products );
		}

		$product_term_ids = array( $category_id );

		$product_term_args = array(
			'taxonomy' => 'product_cat',
			'include'  => $product_term_ids,
			'orderby'  => 'include',
		);
		$product_terms     = get_terms( $product_term_args );

		$product_term_slugs = array();
		foreach ( $product_terms as $product_term ) {
			$product_term_slugs[] = $product_term->slug;
		}

		$product_args = array(
			'post_status' => 'publish',
			'limit'       => $quantity,
			'category'    => $product_term_slugs,
		);
		$products_obj = wc_get_products( $product_args );

		foreach ( $products_obj as $product_obj ) {
			$product = array(
				'title'      => $product_obj->get_title(),
				'excerpt'    => $product_obj->get_short_description(),
				'content'    => $product_obj->get_description(),
				'permalink'  => get_permalink( $product_obj->get_id() ),
				'thumbnail'  => empty( wp_get_attachment_image_src( $product_obj->get_image_id(), 'full' ) ) ? array( MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png', 150, 150, true ) : wp_get_attachment_image_src( $product_obj->get_image_id(), 'full' ),
				'price_html' => wc_price( $product_obj->get_price() ),
			);
			array_push( $all_products, $product );
		}

		return rest_ensure_response( $all_products );
	}

	/**
	 * Get latest published post by category ID
	 *
	 * @param WP_REST_Request $request WP Rest Request object.
	 *
	 * @return WP_REST_Response
	 * @since 1.0.0
	 */
	public function get_latest_published_post_by_category( WP_REST_Request $request ) {
		add_filter( 'get_the_excerpt', array( $this, 'mint_custom_excerpt_more' ), 20 );
		$params      = MrmCommon::get_api_params_values( $request );
		$category_id = isset( $params['category_id'] ) ? $params['category_id'] : '';
		$post_type   = isset( $params['post_type'] ) ? $params['post_type'] : '';
		$quantity	 = isset( $params['quantity'] ) ? $params['quantity'] : 3;

		$args = array(
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'posts_per_page' => $quantity,
			'orderby'        => 'date',
			'order'          => 'DESC',
			'category__in'   => array( $category_id ),
		);

		$latest_post = new WP_Query( $args );
		$posts       = array();

		if ( $latest_post->have_posts() ) {
			while ( $latest_post->have_posts() ) {
				$latest_post->the_post();
				$post = array(
					'title'     => get_the_title(),
					'excerpt'   => strip_shortcodes( get_the_excerpt() ),
					'content'   => strip_shortcodes( get_the_content() ),
					'permalink' => get_the_permalink(),
					'thumbnail' => empty( get_the_post_thumbnail_url(get_the_ID(), 'full') ) ? MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png' : get_the_post_thumbnail_url(get_the_ID(), 'full'),
				);
				array_push( $posts, $post );
			}
			wp_reset_postdata();
		}

		array_walk(
			$posts,
			function( &$post ) {
				$content         = preg_replace( '/<img[^>]+\>/i', ' ', $post['content'] );
				$content         = apply_filters( 'the_content', $content );
				$content         = str_replace( ']]>', ']]>', $content );
				$post['content'] = $content;
			}
		);

		return rest_ensure_response( $posts );
	}

	/**
	 * Get published post by post ID
	 *
	 * @param WP_REST_Request $request WP Rest Request object.
	 *
	 * @return WP_REST_Response
	 * @since 1.2.6
	 */
	public function get_published_product_by_id( WP_REST_Request $request ) {
		add_filter( 'get_the_excerpt', array( $this, 'mint_custom_excerpt_more' ), 20 );
		$params     = MrmCommon::get_api_params_values( $request );
		$product_id = isset( $params['product_id'] ) ? $params['product_id'] : '';

		$product_obj = wc_get_product( $product_id );
		$product     = array(
			'title'      => $product_obj->get_title(),
			'excerpt'    => $product_obj->get_short_description(),
			'content'    => $product_obj->get_description(),
			'permalink'  => get_permalink( $product_obj->get_id() ),
			'thumbnail'  => empty( wp_get_attachment_image_src( $product_obj->get_image_id(), 'full' ) ) ? array( MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png', 150, 150, true ) : wp_get_attachment_image_src( $product_obj->get_image_id(), 'full' ),
			'price_html' => wc_price( $product_obj->get_price() ),
		);

		return rest_ensure_response( $product );
	}

	/**
	 * Get all products from native WooCommerce
	 *
	 * @param WP_REST_Request $request WP Rest Request object.
	 *
	 * @return WP_REST_Response
	 * @since 1.0.8
	 */
	public function get_all_products( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$term   = isset( $params['term'] ) ? $params['term'] : '';
		if ( $term && MrmCommon::is_wc_active() ) {
			$data_store = \WC_Data_Store::load( 'product' );
			$ids        = $data_store->search_products( $term, '', true, false, 10 );

			$products = array();

			foreach ( $ids as $id ) {
				$product_object = wc_get_product( $id );

				if ( ! wc_products_array_filter_readable( $product_object ) ) {
					continue;
				}

				$formatted_name = $product_object->get_formatted_name();
				$managing_stock = $product_object->managing_stock();

				if ( $managing_stock && ! empty( $_GET['display_stock'] ) ) { //phpcs:ignore
					$stock_amount = $product_object->get_stock_quantity();
					/* Translators: %d stock amount */
					$formatted_name .= ' &ndash; ' . sprintf( __( 'Stock: %d', 'mailmint-pro' ), wc_format_stock_quantity_for_display( $stock_amount, $product_object ) );
				}

				$products[ $product_object->get_id() ] = rawurldecode( wp_strip_all_tags( $formatted_name ) );
			}
		}
		return rest_ensure_response( $products );
	}

	/**
	 * Get all posts from native WooCommerce
	 *
	 * @param WP_REST_Request $request WP Rest Request object.
	 *
	 * @return WP_REST_Response
	 * @since 1.0.0
	 */
	public function get_all_posts( WP_REST_Request $request ) {
		$params    = MrmCommon::get_api_params_values( $request );
		$term      = isset( $params['term'] ) ? $params['term'] : '';
		$post_type = isset( $params['post_type'] ) ? $params['post_type'] : '';

		$args = array(
			's'              => $term,
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

		$posts_query = new WP_Query( $args );
		$posts       = array();
		if ( $posts_query->have_posts() ) {
			while ( $posts_query->have_posts() ) {
				$posts_query->the_post();
				$post = array(
					'value' => get_the_ID(),
					'label' => get_the_title(),
				);
				array_push( $posts, $post );
			}
			wp_reset_postdata();
		}

		return rest_ensure_response( $posts );
	}

	/**
	 * Get all posts from native WooCommerce
	 *
	 * @param WP_REST_Request $request WP Rest Request object.
	 *
	 * @return WP_REST_Response
	 * @since 1.0.0
	 */
	public function get_recent_posts( WP_REST_Request $request ) {
		add_filter( 'get_the_excerpt', array( $this, 'mint_custom_excerpt_more' ), 20 );
		$params    = MrmCommon::get_api_params_values( $request );
		$post_type = isset( $params['post_type'] ) ? $params['post_type'] : '';
		$quantity  = isset( $params['quantity'] ) ? $params['quantity'] : 3;

		$args = array(
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'posts_per_page' => $quantity,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);
		$posts_query = new WP_Query( $args );
		$posts       = array();
		if ( $posts_query->have_posts() ) {
			while ( $posts_query->have_posts() ) {
				$posts_query->the_post();
				$post = array(
					'title'     => get_the_title(),
					'excerpt'   => strip_shortcodes( get_the_excerpt() ),
					'content'   => strip_shortcodes( get_the_content() ),
					'permalink' => get_the_permalink(),
					'thumbnail' => empty( get_the_post_thumbnail_url(get_the_ID(), 'full') ) ? MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png' : get_the_post_thumbnail_url(get_the_ID(), 'full'),
				);
				array_push( $posts, $post );
			}
			wp_reset_postdata();
		}
		array_walk(
			$posts,
			function( &$post ) {
				$content         = preg_replace( '/<img[^>]+\>/i', ' ', $post['content'] );
				$content         = apply_filters( 'the_content', $content );
				$content         = str_replace( ']]>', ']]>', $content );
				$post['content'] = $content;
			}
		);

		return rest_ensure_response( $posts );
	}

	/**
	 * Get published post by post ID
	 *
	 * @param WP_REST_Request $request WP Rest Request object.
	 *
	 * @return WP_REST_Response
	 * @since 1.0.0
	 */
	public function get_published_post_by_id( WP_REST_Request $request ) {
		add_filter( 'get_the_excerpt', array( $this, 'mint_custom_excerpt_more' ), 20 );
		$params    = MrmCommon::get_api_params_values( $request );
		$post_id   = isset( $params['post_id'] ) ? $params['post_id'] : '';
		$post_type = isset( $params['post_type'] ) ? $params['post_type'] : '';

		$args = array(
			'post_type'   => $post_type,
			'post_status' => 'publish',
			'p'           => $post_id,
		);

		$latest_post = new WP_Query( $args );
		$post        = array();
		if ( $latest_post->have_posts() ) {
			while ( $latest_post->have_posts() ) {
				$latest_post->the_post();
				$post = array(
					'title'     => get_the_title(),
					'excerpt'   => strip_shortcodes( get_the_excerpt() ),
					'content'   => strip_shortcodes( get_the_content() ),
					'permalink' => get_the_permalink(),
					'thumbnail' => empty( get_the_post_thumbnail_url(get_the_ID(), 'full') ) ? MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png' : get_the_post_thumbnail_url(get_the_ID(), 'full'),
				);

				$content         = preg_replace( '/<img[^>]+\>/i', ' ', $post['content'] );
				$content         = apply_filters( 'the_content', $content );
				$content         = str_replace( ']]>', ']]>', $content );
				$post['content'] = $content;
				return $post;
			}
			wp_reset_postdata();
		}

		return rest_ensure_response( $post );
	}

	/**
	 * Customize post automatic excert
	 *
	 * @param mixed $output Post content.
	 *
	 * @return string
	 * @since 1.0.7
	 */
	public function mint_custom_excerpt_more( $output ) {
		if ( ! has_excerpt() ) {
			$output = $this->add_ellipsis( $output, 50 );
		}

		return $output;
	}

	/**
	 * Add ellipsis after 50 word in post
	 *
	 * @param mixed  $string Post content.
	 * @param mixed  $word_count Post content word count.
	 * @param string $ellipsis Ellipsis type.
	 *
	 * @return string
	 * @since 1.0.7
	 */
	public function add_ellipsis( $string, $word_count, $ellipsis = '...' ) {
		// Strip tags and extra white space.
		$string = trim( strip_tags( $string ) ); //phpcs:ignore

		// Get the number of words in the string.
		$word_count = intval( $word_count );

		if ( str_word_count( $string ) > $word_count ) {
			// Find the position of the 50th word in the string.
			$pos = 0;
			for ( $i = 1; $i <= $word_count; $i++ ) {
				$pos = strpos( $string, ' ', $pos + 1 );
			}

			// Add the ellipsis and return the modified string.
			return substr( $string, 0, $pos ) . $ellipsis;
		}

		// Return the original string if it doesn't need to be shortened.
		return $string;
	}

	/**
	 * Retrieves categories for a given post type.
	 *
	 * @param WP_REST_Request $request The REST API request object.
	 *
	 * @return array Returns an array of category objects with their ID and name as properties.
	 * @since 1.0.8
	 */
	public function get_categories_by_post( WP_REST_Request $request ) {
		$params    = MrmCommon::get_api_params_values( $request );
		$post_type = isset( $params['post_type'] ) ? $params['post_type'] : '';
		$taxonomy  = 'category';
		// Get all posts by post type.
		$get_all_posts = get_posts(
			array(
				'post_type'   => esc_attr( $post_type ),
				'post_status' => 'publish',
				'numberposts' => -1,
			)
		);

		if ( !empty( $get_all_posts ) ) {

			// First Empty Array to store the terms.
			$post_terms = array();

			// Loop through the posts array and retrieve the terms attached to those posts.
			foreach ( $get_all_posts as $all_posts ) {

				/**
				 * Store the new terms objects within `$post_terms`.
				 */
				$post_terms[] = get_the_terms( $all_posts->ID, esc_attr( $taxonomy ) );
			}

			// Second Empty Array to store final term data in key, value pair.
			$post_terms_array = array();

			/**
			 * Loop through `$post_terms` as it's a array of term objects.
			 */

			foreach ( $post_terms as $new_arr ) {
				if ( !empty( $new_arr ) ) {
					foreach ( $new_arr as $arr ) {

						/**
						 * Store the terms with our desired key, value pair inside `$post_terms_array`.
						 */
						$post_terms_array[] = array(
							'name'    => $arr->name,
							'term_id' => $arr->term_id,
							'slug'    => $arr->slug,
							'url'     => get_term_link( $arr->term_id ),
						);
					}
				}
			}

			$post_categories = array_unique( $post_terms_array, SORT_REGULAR );
		}

		$categories = array();
		foreach ( $post_categories as $category ) {
			$categories[] = array(
				'value' => $category['term_id'],
				'label' => $category['name'],
			);
		}
		return $categories;
	}

	/**
	 * Retrieve an array of randomly selected products from the shop.
	 *
	 * This function queries the shop's products and returns an array of information about randomly selected products.
	 *
	 * @access public
	 *
	 * @param WP_REST_Request $request The REST request object containing the API parameters.
	 * @return WP_REST_Response The response containing an array of product information.
	 *
	 * @since 1.5.5
	 */
	public function retrieve_random_shop_products( WP_REST_Request $request ) {
		$params   = MrmCommon::prepare_request_params( $request );
		$quantity = isset( $params['quantity'] ) ? $params['quantity'] : 3;

		$args = array(
			'post_type'        => array( 'product' ),
			'post_status'      => 'publish',
			'posts_per_page'   => $quantity,
			'suppress_filters' => true,
		);

		$products_obj = wc_get_products( $args );

		if ( empty( $products_obj ) ) {
			return rest_ensure_response(
				array(
					'status'  => 'success',
					'data'    => array(),
					'message' => __( 'No products are currently available.', 'mailmint-pro' ),
				)
			);
		}

		$all_products = array();
		$total_price  = 0;

		foreach ( $products_obj as $product_obj ) {
			$product = array(
				'title'      => $product_obj->get_title(),
				'excerpt'    => $product_obj->get_short_description(),
				'content'    => $product_obj->get_description(),
				'permalink'  => get_permalink( $product_obj->get_id() ),
				'thumbnail'  => empty( wp_get_attachment_image_src( $product_obj->get_image_id(), 'full' ) ) ? array( MRM_DIR_URL . 'admin/assets/images/mint-placeholder.png', 150, 150, true ) : wp_get_attachment_image_src( $product_obj->get_image_id(), 'full' ),
				'price_html' => wc_price( $product_obj->get_price() ),
			);
			array_push( $all_products, $product );
			// Calculate and update the total price.
			$total_price += floatval( $product_obj->get_price() );
		}

		return rest_ensure_response(
			array(

				'status'  => 'success',
				'data'    => array(
					'items'       => $all_products,
					'total_price' => wc_price( $total_price ),
				),
				'message' => __( 'Products has been retrieved successfully.', 'mailmint-pro' ),
			)
		);
	}

	/**
	 * User accessability check for REST API
	 *
	 * @return bool
	 * @since 1.0.0
	 */
	public function rest_permissions_check() {
		if ( ! MrmCommon::rest_check_manager_permissions() ) {
			return new \WP_Error( 'MailMint_rest_cannot_edit', __( 'Sorry, you cannot edit this resources.', 'mailmint-pro' ), array( 'status' => rest_authorization_required_code() ) );
		}
		return true;
	}

}
