<?php

/**
 * Handles API requests related to form export and import.
 *
 * Provides methods to export form data in JSON format and import form data from a JSON file.
 *
 * @author   WPFunnels Team
 * @package  MailMintPro\Mint\Admin\API\Controllers
 * @since    1.17.0
 */

namespace MailMintPro\Mint\Admin\API\Controllers;

use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use Mint\MRM\Utilities\Helper\PermissionManager;
use Mint\Utilities\Arr;
use MRM\Common\MrmCommon;
use WP_REST_Request;
use WP_User_Query;

/**
 * RoleManagementController Class
 *
 * Handles API requests related to form export and import.
 *
 * @since 1.17.0
 */
class CustomAccessController extends AdminBaseController{

    /**
     * Retrieve and paginate WordPress roles with additional metadata.
     *
     * This function handles a REST API request to retrieve WordPress roles.
     *
     * @param WP_REST_Request $request The REST API request object.
     *
     * @return WP_REST_Response The response object containing the roles data.
     * @since 1.17.0
     */
    public function get_custom_access( WP_REST_Request $request ) {
        $params   = MrmCommon::get_api_params_values( $request );
        $page     = isset( $params['page'] ) ? $params['page'] : 1;
        $per_page = isset( $params['per-page'] ) ? $params['per-page'] : 10;

        $query = new WP_User_Query(array(
            'meta_key'     => '_mailmint_has_role',
            'meta_value'   => 1,
            'meta_compare' => '=',
            'number'       => $per_page,
            'paged'        => $page
        ));

        $managers = [];

        foreach ($query->get_results() as $user) {
            $managers[] = [
                'id'          => $user->ID,
                'first_name'  => $user->first_name,
                'last_name'   => $user->last_name,
                'email'       => $user->user_email,
                'permissions' => PermissionManager::get_user_permissions($user)
            ];
        }

        $count = $query->get_total();    
        // Prepare the response.
        $response = array(
            'status'  => 'success',
            'message' => __( 'Users have been retrieved successfully.', 'mailmint-pro' ),
            'data'    => array(
                'users'       => $managers,
                'count'       => $count,
                'total_pages' => ceil( $count / $per_page ),
            ),
        );
    
        return rest_ensure_response( $response );
    }

    /**
     * Imports form data from a JSON file.
     *
     * This function handles a REST API request to import form data from a JSON file.
     *
     * @param WP_REST_Request $request The REST API request object.
     * @return WP_REST_Response The response object containing the list of coupons.
     * 
     * @since 1.17.0
     */
    public function get_users_and_permissions(WP_REST_Request $request){
        $params  = MrmCommon::get_api_params_values($request);
        $user_id = Arr::get($params, 'id');
        $user    = get_user_by('id', $user_id);

        $response = [
            'status' => 'success',
            'message' => 'Users and capabilities has been retrieved successfully.',
            'data' => [
                'email' => $user->user_email,
                'permissions' => $user->allcaps,
            ],
        ];
        return rest_ensure_response( $response );
    }

    public function assign_custom_access( WP_REST_Request $request ){
        $params = MrmCommon::prepare_request_params( $request );
        $email  = Arr::get($params, 'email');
        $user   = get_user_by('email', $email);

        // Validate user.
        if ( ! $user ) {
            return rest_ensure_response( array(
                'status'  => 'failed',
                'message' => __( 'Associate user could not be found with this email.', 'mailmint-pro' ),
            ) );
        }

        // Filter and keep only keys with truthy values.
        $permissions = Arr::get($params, 'permissions', []);
        $permissions = array_keys(array_filter($permissions));

        $all_permissions = PermissionManager::get_readable_permissions();
        foreach ($permissions as $permission) {
            $depends = Arr::get($all_permissions, $permission . '.depends', []);
            if ($depends && !array_intersect($permissions, $depends)) {
                rest_ensure_response(array(
                    'status'  => 'failed',
                    'message' => $permission . ' require permissions: ' . implode(', ', $depends),
                ));
            }
        }

        // Assign permissions to each user if $users is not empty
        PermissionManager::attach_permissions($user, $permissions);
        update_user_meta($user->id, '_mailmint_has_role', 1);

        return rest_ensure_response( array(
            'status'  => 'success',
            'message' => __( 'Permissions added to the user successfully.', 'mailmint-pro' ),
        ) );
    }

    public function update_custom_access( WP_REST_Request $request ){
        $params = MrmCommon::prepare_request_params( $request );
        $email  = Arr::get($params, 'email');
        $user   = get_user_by('email', $email);

        // Validate user.
        if ( ! $user ) {
            return rest_ensure_response( array(
                'status'  => 'failed',
                'message' => __( 'Associate user could not be found with this email.', 'mailmint-pro' ),
            ) );
        }

        // Filter and keep only keys with truthy values.
        $permissions = Arr::get($params, 'permissions', []);
        $permissions = array_keys(array_filter($permissions));

        $all_permissions = PermissionManager::get_readable_permissions();
        foreach ($permissions as $permission) {
            $depends = Arr::get($all_permissions, $permission . '.depends', []);
            if ($depends && !array_intersect($permissions, $depends)) {
                rest_ensure_response(array(
                    'status'  => 'failed',
                    'message' => $permission . ' require permissions: ' . implode(', ', $depends),
                ));
            }
        }

        // Assign permissions to each user if $users is not empty
        PermissionManager::attach_permissions($user, $permissions);
        update_user_meta($user->id, '_mailmint_has_role', 1);

        return rest_ensure_response( array(
            'status'  => 'success',
            'message' => __( 'Permissions added to the user successfully.', 'mailmint-pro' ),
        ) );
    }

    public function delete_custom_access( WP_REST_Request $request ){
        $id   = $request->get_param('id');
        $user = get_user_by('ID', $id);

        // Validate user.
        if ( ! $user ) {
            return rest_ensure_response( array(
                'status'  => 'failed',
                'message' => __( 'Associate user could not be found with this email.', 'mailmint-pro' ),
            ) );
        }

        // Assign permissions to each user if $users is not empty
        PermissionManager::attach_permissions($user, array());
        delete_user_meta($user->id, '_mailmint_has_role');

        return rest_ensure_response( array(
            'status'  => 'success',
            'message' => __( 'Permissions has been deleted successfully.', 'mailmint-pro' ),
        ) );
    }
}
