<?php
/**
 * Mail Mint
 *
 * @author [MRM Team]
 * @email [support@getwpfunnels.com]
 * @create date 2023-07-20 11:03:17
 * @modify date 2023-07-20 11:03:17
 * @package /app/Admin/API/Controllers
 */

namespace MintMailPro\Mint\Admin\API\Controllers;

use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use MintMailPro\Mint\Admin\API\Creators\ContactExportActionCreator;
use MRM\Common\MrmCommon;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Class ContactExportController
 * Extends AdminBaseController and provides methods for handling contact export related functionalities.
 *
 * @since 1.5.0
 */
class ContactExportController extends AdminBaseController {

	/**
	 * The ContactExportActionCreator instance used to create ContactExportAction objects.
	 *
	 * @var ContactExportActionCreator
	 * @access protected
	 * @since 1.5.0
	 */
	protected $creator;

	/**
	 * The ContactExportAction instance used for performing contact export actions.
	 *
	 * @var ContactExportAction
	 * @access protected
	 * @since 1.5.0
	 */
	protected $action;

	/**
	 * ContactExportController constructor.
	 *
	 * This constructor initializes the ContactExportActionCreator and ContactExportAction objects,
	 * making them accessible within the class for further use.
	 *
	 * @access public
	 * @since 1.5.0
	 */
	public function __construct() {
		$this->creator = new ContactExportActionCreator();
		$this->action  = $this->creator->makeAction();
	}

	/**
	 * Summary: Retrieves contact export fields and segments data.
	 * Description: Retrieves contact fields and segments data using the ContactExportActionCreator class.
	 *
	 * @access public
	 *
	 * @return WP_REST_Response Returns a REST response containing the status, data, and message.
	 * @since 1.5.0
	 */
	public function get_contact_export_fields() {
		$response = $this->action->get_contact_fields_and_segments_data();

		return new WP_REST_Response(
			array(
				'status'  => 'success',
				'data'    => $response,
				'message' => __( 'Contact fields have been retrieved successfully.', 'mailmint-pro' ),
			)
		);
	}

	/**
	 * Summary: Fetches contact data for contact export preview.
	 * Description: Retrieves contact data based on the provided API parameters to generate a preview for contact export.
	 *
	 * @access public
	 *
	 * @param WP_REST_Request $request The REST request object containing the API parameters.
	 * @return WP_REST_Response Returns a REST response containing the contact data preview.
	 * @since 1.5.0
	 */
	public function get_contact_export_preview( WP_REST_Request $request ) {
		// Get API parameters from the request object and filter them.
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );

		// Get the contact data for export preview.
		$response = $this->action->get_contact_data_to_preview( $params );
		return new WP_REST_Response(
			array(
				'status'  => 'success',
				'data'    => $response,
				'message' => __( 'Contacts have been retrieved successfully.', 'mailmint-pro' ),
			)
		);
	}

	/**
	 * Exports contacts to a CSV file based on the provided API parameters.
	 *
	 * @access public
	 *
	 * @param WP_REST_Request $request The REST request object containing the API parameters.
	 * @return WP_REST_Response Returns a REST response containing the status, data, and message.
	 * @since 1.5.0
	 */
	public function export_all_contacts( WP_REST_Request $request ) {
		// Get API parameters from the request object and filter them.
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );

		// Get the contact data for export preview.
		$response = $this->action->export_contacts_to_csv( $params );
		return new WP_REST_Response(
			array(
				'status'  => 'success',
				'data'    => $response,
				'message' => __( 'Contacts have been exported successfully.', 'mailmint-pro' ),
			)
		);
	}

	/**
	 * Downloads the exported CSV file based on the provided API parameters.
	 *
	 * @access public
	 *
	 * @param WP_REST_Request $request The REST request object containing the API parameters.
	 * @return void.
	 * @since 1.5.0
	 * @since 1.10.0 Added the header information to download the CSV file.
	 */
	public function download_csv_file( WP_REST_Request $request ) {
		// Get API parameters from the request object and filter them.
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );

		$file_name = isset( $params['fileName'] ) ? $params['fileName'] : '';
		$file_name = MAIL_MINT_EXPORT_DIR . '/' . $file_name;
		if ( file_exists( $file_name ) ) {
			// Define header information.
			header( 'Content-Description: File Transfer' );
			header( 'Content-Type: application/octet-stream' );
			header( 'Cache-Control: no-cache, must-revalidate' );
			header( 'Expires: 0' );
			header( 'Content-Disposition: attachment; filename="' . basename( $file_name ) . '"' );
			header( 'Content-Length: ' . filesize( $file_name ) );
			header( 'Pragma: public' );

			// Clear system output buffer.
			flush();

			// Initialize WP_Filesystem.
			global $wp_filesystem;
			if ( empty( $wp_filesystem ) ) {
				require_once ABSPATH . '/wp-admin/includes/file.php';
				WP_Filesystem();
			}

			// Read the size of the file.
			echo esc_html( $wp_filesystem->get_contents( $file_name ) );
			exit;
		}
		wp_die();
	}
}
