<?php

/**
 * Mail Mint
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-01-20 11:03:17
 * @modify date 2024-01-20 11:03:17
 * @package /app/Admin/API/Controllers
 */

namespace MintMailPro\Mint\Admin\API\Controllers;

use MRM\Common\MrmCommon;
use Mint\MRM\Admin\API\Controllers\AdminBaseController;
use WP_REST_Request;
use MintMailPro\Mint\Admin\API\Creators\AnalyticsActionCreator;

/**
 * AnalyticsController class for handling campaign analytics requests.
 *
 * This class extends the AdminBaseController and is responsible for handling
 * REST API requests related to campaign analytics. It includes an action creator
 * and an action instance for processing analytics requests.
 *
 * @since 1.9.0
 */

class AnalyticsController extends AdminBaseController {

	/**
	 * Instance of the AnalyticsActionCreator class for creating analytics action instances.
	 *
	 * @var AnalyticsActionCreator
	 * @since 1.9.0
	 */
	protected $creator;

	/**
	 * Instance of the analytics action for handling campaign analytics requests.
	 *
	 * @var mixed
	 * @since 1.9.0
	 */
	protected $action;

	/**
	 * Constructor for the AnalyticsController class.
	 *
	 * Initializes the action creator and creates an analytics action instance.
	 *
	 * @since 1.9.0
	 */
	public function __construct() {
		$this->creator = new AnalyticsActionCreator();
		$this->action  = $this->creator->makeAction();
	}

	/**
	 * Handle campaign analytics request.
	 *
	 * Processes the incoming REST API request for retrieving campaign analytics.
	 *
	 * @param WP_REST_Request $request The REST API request object.
	 * @return WP_REST_Response The REST API response containing the retrieved analytics data.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_analytics_request( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );
		$result = $this->action->format_and_retrieve_campaign_analytics( $params );

		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Prepare analytics for regular or sequence campaign
	 *
	 * @param int $campaign_id Campaign ID.
	 * @param int $total_recipients Total number of recipients.
	 *
	 * @return array
	 * @since 1.9.0
	 */
	public function handle_campaign_analytics_overview( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );
		$result = $this->action->format_and_retrieve_campaign_overview( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Handles the retrieval of campaign email recipients.
	 *
	 * This method retrieves the parameters from the request, filters them, and then
	 * calls the `format_and_retrieve_campaign_email_recipients` method of the `action` object
	 * to get the result. It then returns a response with the status, a success message, and the result.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response The response object containing the status, message, and results.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_email_recipients( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );
		$result = $this->action->format_and_retrieve_campaign_email_recipients( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Handles the retrieval of sent email lists.
	 *
	 * This method retrieves the parameters from the request, filters them, and then
	 * calls the `format_and_retrieve_campaign_email_recipients` method of the `action` object
	 * to get the result. It then returns a response with the status, a success message, and the result.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response The response object containing the status, message, and results.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_email_sent_to_lists( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );
		$result = $this->action->format_and_retrieve_campaign_email_sent_to_lists( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Handles the retrieval of campaign email opened lists.
	 *
	 * This method retrieves the parameters from the request, filters them, and then
	 * calls the `format_and_retrieve_campaign_email_opened_lists` method of the `action` object
	 * to get the result. It then returns a response with the status, a success message, and the result.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response The response object containing the status, message, and results.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_email_opened_lists( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );
		$result = $this->action->format_and_retrieve_campaign_email_opened_lists( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Handles the retrieval of campaign email clicked lists.
	 *
	 * This method retrieves the parameters from the request, filters them, and then
	 * calls the `format_and_retrieve_campaign_email_clicked_lists` method of the `action` object
	 * to get the result. It then returns a response with the status, a success message, and the result.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response The response object containing the status, message, and results.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_email_clicked_lists( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );
		$result = $this->action->format_and_retrieve_campaign_email_clicked_lists( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Handles the retrieval of campaign email bounced lists.
	 *
	 * This method retrieves the parameters from the request, filters them, and then
	 * calls the `format_and_retrieve_campaign_email_bounced_lists` method of the `action` object
	 * to get the result. It then returns a response with the status, a success message, and the result.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response The response object containing the status, message, and results.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_email_bounced_lists( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );
		$result = $this->action->format_and_retrieve_campaign_email_bounced_lists( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Handles the retrieval of campaign email unsubscribed lists.
	 *
	 * This method retrieves the parameters from the request, filters them, and then
	 * calls the `format_and_retrieve_campaign_email_unsubscribed_lists` method of the `action` object
	 * to get the result. It then returns a response with the status, a success message, and the result.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response The response object containing the status, message, and results.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_email_unsubscribed_lists( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );

		$result = $this->action->format_and_retrieve_campaign_email_unsubscribed_lists( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}

	/**
	 * Handles the retrieval of campaign email click performance lists.
	 *
	 * This method retrieves the parameters from the request, filters them, and then
	 * calls the `format_and_retrieve_campaign_email_click_performance_lists` method of the `action` object
	 * to get the result. It then returns a response with the status, a success message, and the result.
	 *
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response The response object containing the status, message, and results.
	 *
	 * @since 1.9.0
	 */
	public function handle_campaign_email_click_performance_lists( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$params = filter_var_array( $params );

		$result = $this->action->format_and_retrieve_campaign_email_click_performance_lists( $params );
		return rest_ensure_response(
			array(
				'status'  => 'success',
				'message' => __( 'Data has been retrieved successfully.', 'mailmint-pro' ),
				'results' => $result,
			)
		);
	}
}
