<?php
/**
 * Mail Mint
 *
 * @package /app/Admin/API/Controllers
 *
 * @since 1.5.0
 *
 * @author [MailMint Team]
 * @email [support@getwpfunnels.com]
 */

namespace MintMailPro\Mint\Admin\API\Controllers;

use MailMintPro\Mint\Internal\AbandonedCart\Helper\Common;
use MailMintPro\Mint\Internal\AbandonedCart\Helper\Model;
use MRM\Common\MrmCommon;
use WP_REST_Request;
use Mint\MRM\API\Controllers\BaseController;
use WC_Session_Handler;

/**
 * Manages  AbandonedCart API callbacks
 *
 * @package /app/API/Controllers
 * @since 1.5.0
 */
class AbandonedCartFrontendController extends BaseController {

	/**
	 * Summary: Updates the checkout data for the cart.
	 *
	 * Description: Updates the checkout data for the cart based on the API parameters received in the request.
	 *
	 * @access public
	 *
	 * @param WP_REST_Request $request The REST request object.
	 *
	 * @return WP_REST_Response Returns a REST response with the success status and message.
	 * @since 1.5.0
	 */
	public function update_checkout( WP_REST_Request $request ) {
		// Get the API parameters from the request.
		$params               = MrmCommon::get_api_params_values( $request );
		$woocommerce          = WC();
		$woocommerce->session = new WC_Session_Handler();
		$woocommerce->session->init();

		$cart_totals  = $woocommerce->session->cart_totals;
		$meta_details = Common::get_current_cart_details( $woocommerce->session );

		$cart_email    = isset( $params['email'] ) ? $params['email'] : '';
		$settings      = Common::get_abandoned_cart_settings();
		$disable_roles = isset( $settings['disable_roles'] ) ? $settings['disable_roles'] : array();

		// Check if the user role is in the disabled roles list.
		if ( ! empty( $disable_roles ) ) {
			$user = get_user_by( 'email', $cart_email );
			if ( $user ) {
				$user_roles = $user->roles;
				foreach ( $user_roles as $role ) {
					if ( in_array( $role, $disable_roles, true ) ) {
						return rest_ensure_response(
							array(
								'success' => true,
								'message' => __( 'User role is disabled to track abandoned cart.', 'mailmint-pro' ),
							)
						);
					}
				}
			}
		}

		if (isset($_COOKIE['mint_ab_cart_skip_track']) && $_COOKIE['mint_ab_cart_skip_track'] == 'yes') {
			return rest_ensure_response(
				array(
					'success' => true,
					'message' => __( 'Cart tracking has been opted out successfully.', 'mailmint-pro' ),
				)
			);
		}

		$cart_details = Model::get_cart_details_by_key_and_status( 'email', $cart_email, array( 'pending' ) );

		if ( ! empty( $cart_details ) ) {
			$session_key = $woocommerce->session->get_customer_id();
			$cart_id     = isset( $cart_details['id'] ) ? $cart_details['id'] : null;
		} else {
			$session_key = $woocommerce->session->get_customer_id();
			// Get the cart details by session_key.
			$cart_details = Model::get_cart_details_by_key_and_status( 'session_key', $session_key, array( 'pending' ) );
			$cart_id      = isset( $cart_details['id'] ) ? $cart_details['id'] : null;
		}

		if ( !$cart_id ) {
			// Check if the user ID is not 0 (indicating a valid user).
			$user    = wp_get_current_user();
			$user_id = (isset($user->ID) ? (int) $user->ID : 0);
			if ($user_id) {
				$user_data  = get_userdata($user_id);
				$cart_email = $user_data->user_email;
			}

			$data = array(
				'user_id'  => $user_id,
				'email'    => $cart_email,
				'items'    => maybe_serialize($woocommerce->session->cart),
				'provider' => 'WC',
				'status'   => 'pending',
			);

			$data                  = Common::get_abandoned_cart_totals($data);
			$data['token']         = Common::create_abandoned_cart_token(32);
			$data['session_key']   = $session_key;
			$data['created_at']    = current_time('mysql');
			$cart_details['total'] = isset($cart_totals['total']) ? $cart_totals['total'] : 0;
			$data['checkout_data'] = maybe_serialize(array('fields' => json_decode($params['checkout_fields_data'], true)));

			$abandoned_cart_id = Model::insert($data);
			Model::update_cart_meta($abandoned_cart_id, 'abandoned_cart_meta', maybe_serialize($meta_details));

			if (isset($_COOKIE['mint_checkout_page_id'])) {
				$checkout_page_id = $_COOKIE['mint_checkout_page_id'];
				Model::update_cart_meta( $abandoned_cart_id, 'checkout_page_id', $checkout_page_id );
			} else {
				Model::update_cart_meta( $abandoned_cart_id, 'checkout_page_id', wc_get_page_id('checkout') );
			}

			// set user data in action Scheduler.
			$enqueue_data = array(
				'data' => array(
					'id'          => $abandoned_cart_id,
					'email'       => $cart_email,
					'session_key' => $session_key,
				),
			);
			Common::enqueue_user_after_add_to_cart( $enqueue_data, $abandoned_cart_id );

			// set cart creation data in action Scheduler.
			$enqueue_data = array(
					'data' => array(
						'id'          => $abandoned_cart_id,
						'email'       => $cart_email,
						'session_key' => $session_key,
						'created_at'  => current_time('mysql'),
					),
				);
			Common::enqueue_cart_creation_time( $enqueue_data, $abandoned_cart_id );

			return rest_ensure_response(
				array(
					'success' => false,
					'message' => __( 'Cart has been tracking successfully.', 'mailmint-pro' ),
				)
			);
		}

		$cart_details['session_key']   = $session_key;
		$cart_details['total']         = isset( $cart_totals['total'] ) ? $cart_totals['total'] : 0;
		$cart_details['updated_at']    = current_time( 'mysql' );
		$cart_details['email']         = isset( $cart_details['email'] ) ? sanitize_email( $params['email'] ) : '';
		$cart_details['checkout_data'] = isset( $cart_details['checkout_data'] ) ? maybe_serialize( array( 'fields' => json_decode( $params['checkout_fields_data'], true ) ) ) : '';
		// Update the cart details in the database.
		Model::update( $cart_details, $cart_id );
		Model::update_cart_meta( $cart_id, 'abandoned_cart_meta', maybe_serialize( $meta_details ) );

		if (isset($_COOKIE['mint_checkout_page_id'])) {
			$checkout_page_id = $_COOKIE['mint_checkout_page_id'];
			Model::update_cart_meta( $cart_id, 'checkout_page_id', $checkout_page_id );
		} else {
			Model::update_cart_meta( $cart_id, 'checkout_page_id', wc_get_page_id('checkout') );
		}

		// Return success response indicating that checkout data has been updated.
		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Checkout data has been updated.', 'mailmint-pro' ),
			)
		);
	}

	/**
	 * Request for  AbandonedCart update chkout page
	 *
	 * @param WP_REST_Request $request equest object used to generate the response.
	 * @return array|\WP_Error|\WP_HTTP_Response|\WP_REST_Response
	 * @since 1.5.0
	 */
	public function delete_abandoned_cart( WP_REST_Request $request ) {
		$woocommerce          = WC();
		$woocommerce->session = new WC_Session_Handler();
		$woocommerce->session->init();
		$session_key           = WC()->session->get_customer_id();

		$cart_details = Model::get_cart_details_by_key('session_key', $session_key, 'pending');
		// If the cart details are empty, return.
		if (empty($cart_details)) {
			return;
		}

		// Update the status of the cart to 'opt-out'.
		$cart_id   = isset($cart_details['id']) ? $cart_details['id'] : null;
		$is_delete = Model::delete($cart_id);

		$cookie_days = (int)apply_filters('mailmint_ab_cart_opt_out_cookie_validity', 7);
		// set cookie for 30 days
		setcookie('mint_ab_cart_skip_track', 'yes', time() + (86400 * $cookie_days), COOKIEPATH, COOKIE_DOMAIN);
		
		return rest_ensure_response(
			array(
				'success' => true,
				'message' => __( 'Cart has been opted out successfully.', 'mailmint-pro' ),
			)
		);
	}

	/**
	 * Default permission_callback
	 *
	 * @return bool
	 */
	public function rest_permissions_check() {
		return true;
	}
}
