<?php
/**
 * Mail Mint
 *
 * @package /app/Admin/API/Controllers
 *
 * @since 1.5.0
 *
 * @author [MailMint Team]
 * @email [support@getwpfunnels.com]
 */

namespace MintMailPro\Mint\Admin\API\Controllers;

use MailMintPro\Mint\Internal\AbandonedCart\Helper\Common;
use MailMintPro\Mint\Internal\AbandonedCart\Helper\Model;
use MRM\Common\MrmCommon;
use WP_Error;
use WP_REST_Controller;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Manages  AbandonedCart API callbacks
 *
 * @package /app/API/Controllers
 * @since 1.5.0
 */
class AbandonedCartAdminController extends WP_REST_Controller {

	/**
	 * Retrieves the recoverable data from the abandoned cart.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 *
	 * @return WP_REST_Response The REST response containing the recoverable data.
	 *
	 * @since 1.5.0
	 */
	public function get_recoverable_carts( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$status = 'pending,abandoned';
		$result = Model::get_all_abandoned_cart( $params, $status );

		if ( isset( $result ) ) {
			return $this->get_success_response( __( 'Query Successful.', 'mailmint-pro' ), 200, $result );
		}

		return $this->get_error_response( __( 'Failed to get data.', 'mailmint-pro' ), 400 );
	}

	/**
	 * Retrieves the recovered data from the abandoned cart.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 *
	 * @return WP_REST_Response The REST response containing the recovered data.
	 *
	 * @since 1.5.0
	 */
	public function get_recovered_carts( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$status = 'recovered';
		$result = Model::get_all_abandoned_cart( $params, $status );

		if ( isset( $result ) ) {
			return $this->get_success_response( __( 'Query Successful.', 'mailmint-pro' ), 200, $result );
		}

		return $this->get_error_response( __( 'Failed to get data.', 'mailmint-pro' ), 400 );
	}

	/**
	 * Retrieves the lost abandoned data.
	 *
	 * @param WP_REST_Request $request The REST request object.
	 *
	 * @return WP_REST_Response The REST response containing the lost abandoned data.
	 *
	 * @since 1.5.0
	 */
	public function get_losts_carts( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$status = 'lost';
		$result = Model::get_all_abandoned_cart( $params, $status );

		if ( isset( $result ) ) {
			return $this->get_success_response( __( 'Query Successful.', 'mailmint-pro' ), 200, $result );
		}

		return $this->get_error_response( __( 'Failed to get data.', 'mailmint-pro' ), 400 );
	}



	/**
	 * Function used to handle delete single form requests.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 * @return WP_REST_Response
	 * @since 1.5.0
	 */
	public function delete_single( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		if ( isset( $params['abandoned_id'] ) ) {
			$success = Model::delete( $params['abandoned_id'] );
			if ( $success ) {
				return $this->get_success_response( __( 'Abandoned Cart has been deleted successfully.', 'mailmint-pro' ), 200 );
			}
		}

		return $this->get_error_response( __( 'Failed to delete.', 'mailmint-pro' ), 400 );
	}


	/**
	 * Function used to handle delete requests.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 * @return WP_REST_Response
	 * @since 1.5.0
	 */
	public function delete_all( WP_REST_Request $request ) {
		// Get values from API.
		$params = MrmCommon::get_api_params_values( $request );

		if ( isset( $params['abandoned_ids'] ) ) {
			$success = Model::destroy_all( $params['abandoned_ids'] );
			if ( $success ) {
				return $this->get_success_response( __( 'Abandoned carts has been deleted successfully.', 'mailmint-pro' ), 200 );
			}
		}

		return $this->get_error_response( __( 'Failed to delete.', 'mailmint-pro' ), 400 );
	}

	/**
	 * Function used to handle Get abandoned automation data requests.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 * @return WP_REST_Response
	 * @since 1.5.0
	 */
	public function get_abandoned_automations( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		$result = array();
		if ( !isset( $params['abandoned_id'] ) ) {
			return $this->get_error_response( __( 'Failed to get automation data.', 'mailmint-pro' ), 400 );
		}
		$abandoned_data          = Model::get_cart_by_id( $params['abandoned_id'] );
		$params['automation_id'] = !empty( $abandoned_data['automation_id'] ) ? $abandoned_data['automation_id'] : 0;
		$params['email']         = !empty( $abandoned_data['email'] ) ? $abandoned_data['email'] : '';
		$result                  = Model::get_all_abandoned_automations( $params );
		if ( !empty( $result ) ) {
			return $this->get_success_response( __( 'Abandoned automation data.', 'mailmint-pro' ), 200, $result );
		}
		return $this->get_success_response( __( 'Abandoned automation data.', 'mailmint-pro' ), 200, $result );
	}
	/**
	 * Function used to handle Get abandoned automation data requests.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 * @return WP_REST_Response
	 * @since 1.5.0
	 */
	public function get_reports( WP_REST_Request $request ) {
		// Get query params from the API.
		$params = MrmCommon::get_api_params_values( $request );
		$filter = ! empty( $params[ 'filter' ] ) ? $params[ 'filter' ] : 'all';
		$result = Model::get_abandoned_cart_report( $filter );
		if ( !empty( $result ) ) {
			return $this->get_success_response( __( 'Get Report Successfully.', 'mailmint-pro' ), 200, $result );
		}
		return $this->get_success_response( __( 'Abandoned cart analytics Report.', 'mailmint-pro' ), 200, $result );
	}

	/**
	 * Function used to handle Get abandoned automation where trigger is "Abandoned Cart" data requests.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 * @return WP_REST_Response
	 * @since 1.5.5
	 */
	public function get_all_abandoned_automations( WP_REST_Request $request ) {
		$result = Model::get_all_abandoned_cart_automation();
		if ( !empty( $result ) ) {
			return $this->get_success_response( __( 'Abandoned automation data.', 'mailmint-pro' ), 200, $result );
		}
		return $this->get_success_response( __( 'Abandoned automation data.', 'mailmint-pro' ), 200, $result );
	}

	/**
	 * Function used to handle to run abandoned cart manually.
	 *
	 * @param WP_REST_Request $request Request object used to generate the response.
	 * @return WP_REST_Response
	 * @since 1.5.5
	 */
	public function run_abandoned_automation_manually( WP_REST_Request $request ) {
		$params = MrmCommon::get_api_params_values( $request );
		if ( !isset( $params['abandoned_id'], $params['automation_id'] ) ) {
			return $this->get_error_response( __( 'Failed to run automation data.', 'mailmint-pro' ), 400 );
		}
		$result = Model::run_automation_manually( $params['abandoned_id'], $params['automation_id'] );
		if ( $result ) {
			return $this->get_success_response( __( 'Run automation successfully.', 'mailmint-pro' ), 200, $result );
		}
		return $this->get_success_response( __( 'Run abandoned successfully.', 'mailmint-pro' ), 200, $result );
	}



	/**
	 * Prepare success response for REST API.
	 *
	 * @param string $message Response success message.
	 * @param int    $code Response success code.
	 * @param mixed  $data Response data on success.
	 *
	 * @return array
	 * @since 1.5.0
	 */
	public function get_success_response( $message = '', $code = 0, $data = null ) {
		$response = array(
			'code'    => $code,
			'message' => $message,
			'data'    => $data,
		);
		return rest_ensure_response( $response );
	}
	/**
	 * Prepare error response for REST API.
	 *
	 * @param string        $message   Response error message.
	 * @param int           $code      Response error code.
	 * @param WP_Error|null $wp_error WP_Error object for additional error details.
	 *
	 * @return WP_Error The prepared error response.
	 *
	 * @since 1.5.0
	 */
	public function get_error_response( $message = '', $code = 0, $wp_error = null ) {
		if ( 0 !== absint( $code ) ) {
			$this->response_code = $code;
		} elseif ( empty( $code ) ) {
			$this->response_code = 500;
		}

		$data = array();
		if ( $wp_error instanceof \WP_Error ) {
			$message = $wp_error->get_error_message();
			$data    = $wp_error->get_error_data();
		}

		return new \WP_Error(
			$this->response_code,
			$message,
			array(
				'status'     => $this->response_code,
				'error_data' => $data,
			)
		);
	}
}
