<?php
/**
 * Mail Mint
 *
 * @author [getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2023-07-20 11:03:17
 * @modify date 2023-07-20 11:03:17
 * @package /app/Admin/API/Actions
 */

namespace MintMailPro\Mint\Admin\API\Actions;

use MailMintPro\App\Utilities\Helper\MintEDD;
use Mint\MRM\API\Actions\Action;
use Mint\MRM\DataBase\Models\ContactModel;
use MRM\Common\MrmCommon;
use Mint\MRM\DataBase\Models\EmailModel;

/**
 * Class ContactProfileAction
 * Implements the Action interface and provides methods for contact profile functionality.
 *
 * @since 1.6.1
 */
class ContactProfileAction implements Action {

	/**
	 * Get the WooCommerce purchase history for a specific contact.
	 *
	 * @param array $params Parameters passed to customize the query (e.g., page, per-page, contact_id).
	 *
	 * @return array An array containing the formatted WooCommerce purchase history data, total pages, and total count.
	 * @since 1.6.1
	 */
	public function get_contact_woocommerce_purchase_history( $params ) {
		// Extract parameters or use default values.
		$page       = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page   = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$contact_id = isset( $params['contact_id'] ) ? $params['contact_id'] : '';

		$email = ContactModel::get_contact_email_by_id( $contact_id );

		$args = array(
			'limit'    => $per_page,
			'offset'   => $per_page * ( $page - 1 ),
			'paginate' => true,
			'customer' => $email,
		);

		$customer_orders  = wc_get_orders( $args );
		$formatted_orders = array();

		if ( empty( $customer_orders->orders ) ) {
			return array(
				'data' => $formatted_orders,
			);
		}

		foreach ( $customer_orders->orders as $order ) {
			// Initialize an array to store product information.
			$products = array();

			// Get items in the order.
			$order_items = $order->get_items();

			foreach ( $order_items as $item_id => $item ) {
				// Get product ID and title for each item.
				$product_id    = $item->get_product_id();
				$product_title = $item->get_name();

				// Store product information in the $products array.
				$products[] = array(
					'product_id'    => $product_id,
					'product_title' => $product_title,
				);
			}

			$formatted_orders[] = array(
				'id'       => $order->get_order_number(),
				'date'     => esc_html( wc_format_datetime( $order->get_date_created() ) ),
				'status'   => $order->get_status(),
				'total'    => MrmCommon::price_format_with_wc_currency( $order->get_total() ),
				'action'   => $order->get_edit_order_url(),
				'products' => $products,
			);
		}

		return array(
			'data'        => $formatted_orders,
			'total_pages' => $customer_orders->max_num_pages,
			'total_count' => $customer_orders->total,
		);
	}

	/**
	 * Get the EDD purchase history for a specific contact.
	 *
	 * @param array $params Parameters passed to customize the query (e.g., page, per-page, contact_id).
	 *
	 * @return array An array containing the formatted EDD purchase history data, total pages, and total count.
	 * @since 1.6.1
	 */
	public function get_contact_edd_purchase_history( $params ) {
		$formatted_orders = array();

		if ( !class_exists( '\Easy_Digital_Downloads' ) ) {
			return array(
				'data' => $formatted_orders,
			);
		}
		// Extract parameters or use default values.
		$page       = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page   = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset     = ( $page - 1 ) * $per_page;
		$contact_id = isset( $params['contact_id'] ) ? $params['contact_id'] : '';

		$email    = ContactModel::get_contact_email_by_id( $contact_id );
		$customer = new \EDD_Customer( $email );

		if ( !$customer || !$customer->id ) {
			return array(
				'data' => $formatted_orders,
			);
		}

		$total_count = MintEDD::customer_total_order_count( $customer->id );

		if ( !$total_count ) {
			return array(
				'data' => $formatted_orders,
			);
		}

		$orders = MintEDD::get_customer_orders( $customer->id, $offset, $per_page );

		foreach ( $orders as $order ) {
			// Get products for the current order.
			$order_products = edd_get_payment_meta_cart_details( $order->id, true );

			// Initialize an array to store product information.
			$products = array();

			foreach ( $order_products as $product ) {
				// Add product information to the $products array.
				$products[] = array(
					'product_id'    => $product['id'],
					'product_title' => $product['name'],
				);
			}

			$formatted_orders[] = array(
				'id'       => $order->id,
				'date'     => date_i18n( get_option( 'date_format' ), strtotime( $order->date_created ) ),
				'status'   => $order->status,
				'total'    => html_entity_decode( edd_currency_filter( edd_format_amount( $order->total ) ) ),
				'action'   => add_query_arg( 'id', $order->id, admin_url( 'edit.php?post_type=download&page=edd-payment-history&view=view-order-details' ) ),
				'products' => $products,
			);
		}

		return array(
			'data'        => $formatted_orders,
			'total_pages' => ( 0 !== $per_page ) ? ceil( $total_count / $per_page ) : 0,
			'total_count' => (int) $total_count,
		);
	}

	/**
	 * Summary: Retrieves contact overview data.
	 *
	 * Description: This method compiles various metrics for a given contact based on specified filters.
	 *
	 * @access public
	 *
	 * @param array $params An array containing parameters for retrieving contact overview data.
	 *
	 * @return array Returns an array containing contact overview metrics.
	 *
	 * @since 1.7.0
	 */
	public function get_contact_overview( $params ) {
		// Extract relevant parameters from the provided array.
		$contact_id = isset( $params['contact_id'] ) ? $params['contact_id'] : '';
		$filter     = isset( $params['filter'] ) ? $params['filter'] : 'lifetime';

		// Get additional customer information from the ContactModel and MrmCommon classes.
		$email = ContactModel::get_contact_email_by_id( $contact_id );

		// Retrieve email delivery metrics.
		$total_delivered = EmailModel::count_delivered_status_single_contact( $contact_id, 'sent', $filter );
		$total_bounced   = EmailModel::count_delivered_status_single_contact( $contact_id, 'failed', $filter );

		// Retrieve email open and click metrics.
		$total_opened  = EmailModel::count_email_open_click_on_contact( $contact_id, 'is_open', $filter );
		$total_clicked = EmailModel::count_email_open_click_on_contact( $contact_id, 'is_click', $filter );

		$last_opened = EmailModel::last_opened_email_single_contact( $contact_id, 'is_open' );
		$last_opened = !empty( $last_opened ) ? human_time_diff( strtotime( $last_opened ), current_time( 'timestamp' ) ) : 'N/A';

		// Calculate open rate, click rate, and time since the last email was opened.
		$divide_by        = $total_delivered - $total_bounced;
		$divide_by        = 0 === $divide_by ? 1 : $divide_by;
		$open_percentage  = number_format( (float) ( $total_opened / $divide_by ) * 100, 2, '.', '' );
		$click_percentage = number_format( (float) ( $total_clicked / $divide_by ) * 100, 2, '.', '' );

		// Initialize variables for total spent and total orders.
		$total_spent  = 0;
		$total_orders = 0;

		// Retrieve orders associated with the customer.
		$orders = MrmCommon::get_orders_by_customer_email( $email, $filter );

		// If no orders are found, return default customer summary with zeros.
		if ( empty( $orders ) ) {
			return array(
				'total_orders'  => $total_orders,
				'total_spent'   => MrmCommon::price_format_with_wc_currency( $total_spent ),
				'total_sent'    => $total_bounced + $total_delivered,
				'total_opened'  => $total_opened,
				'total_clicked' => $total_clicked,
				'open_rate'     => $open_percentage,
				'click_rate'    => $click_percentage,
				'last_opened'   => $last_opened,
			);
		}

		// Iterate through orders to calculate total spent and total orders.
		foreach ( $orders as $order ) {
			if ( $order ) {
				$total_spent += $order['total_amount'];
			}
		}
		// Return updated customer summary with calculated metrics.
		return array(
			'total_orders'  => count( $orders ),
			'total_spent'   => MrmCommon::price_format_with_wc_currency( $total_spent ),
			'total_sent'    => $total_bounced + $total_delivered,
			'total_opened'  => $total_opened,
			'total_clicked' => $total_clicked,
			'open_rate'     => $open_percentage,
			'click_rate'    => $click_percentage,
			'last_opened'   => $last_opened,
		);
	}
}
