<?php
/**
 * Mail Mint
 *
 * @author [MRM Team]
 * @email [support@getwpfunnels.com]
 * @create date 2023-07-20 11:03:17
 * @modify date 2023-07-20 11:03:17
 * @package /app/Admin/API/Actions
 */

namespace MintMailPro\Mint\Admin\API\Actions;

use MailMintPro\Mint\Internal\Admin\Segmentation\FilterSegmentContacts;
use Mint\MRM\API\Actions\Action;
use Mint\MRM\DataBase\Models\ContactGroupModel;
use Mint\MRM\DataBase\Models\ContactModel;
use MRM\Common\MrmCommon;

/**
 * Class ContactExportAction
 * Implements the Action interface and provides methods for contact export functionality.
 *
 * @since 1.5.0
 */
class ContactExportAction implements Action {

	/**
	 * File handle used for reading or writing data to the CSV file during contact export.
	 *
	 * @var resource|null The file handle, or null if no file is currently opened.
	 * @access private
	 * @since 1.5.0
	 */
	private $file_handle;

	/**
	 * Flag indicating whether the CSV file header has been added.
	 *
	 * @var bool True if the header is already added, false otherwise.
	 * @access private
	 * @since 1.5.0
	 */
	private $header_added = false;

	/**
	 * Summary: Retrieves contact fields and segments data for contact export.
	 * Description: Retrieves contact fields and segments data used for contact export.
	 *
	 * @access public
	 *
	 * @return array Returns an array containing contact fields and segments data.
	 * @since 1.5.0
	 */
	public function get_contact_fields_and_segments_data() {
		// Retrieve contact fields.
		$fields = MrmCommon::import_contacts_map_attrs();

		// Perform array check to ensure $fields is an array before merging.
		if ( !is_array( $fields ) ) {
			$fields = array();
		}

		// Add additional fields.
		$fields = array_merge(
			$fields,
			array(
				array(
					'name' => 'Status',
					'slug' => 'status',
				),
				array(
					'name' => 'Source',
					'slug' => 'source',
				),
			)
		);

		// Retrieve contact segments.
		$segments = ContactGroupModel::get_groups_to_contact_export();

		return array(
			'fields'   => $fields,
			'segments' => $segments,
		);
	}

	/**
	 * Summary: Retrieves contact data for preview based on the provided parameters.
	 * Description: Retrieves contact data for preview based on the provided parameters such as fields and segment ID.
	 *
	 * @access public
	 *
	 * @param array $params An array containing the parameters for contact data preview.
	 * @return array Returns an array containing the contact data, total count, and total batch count for pagination.
	 * @since 1.5.0
	 */
	public function get_contact_data_to_preview( $params ) {
		$fields     = isset( $params['fields'] ) ? $params['fields'] : array();
		$segment_id = isset( $params['segmentID'] ) ? $params['segmentID'] : 0;

		if ( $segment_id ) {
			$segment_contacts = FilterSegmentContacts::get_segment( $segment_id, '', 0, 5 );
			$contacts         = isset( $segment_contacts['contacts'][ 'data' ] ) ? $segment_contacts['contacts'][ 'data' ] : array();
			$total            = isset( $segment_contacts['contacts'][ 'total_count' ] ) ? $segment_contacts['contacts'][ 'total_count' ] : 0;
		} else {
			$all_contacts = ContactModel::get_all( 0, 5, '' );
			$contacts     = isset( $all_contacts['data'] ) ? $all_contacts['data'] : array();
			$total        = isset( $all_contacts['total_count'] ) ? $all_contacts['total_count'] : 0;
		}

		// Check if $contacts is empty after preparing the contact data.
		if ( empty( $contacts ) ) {
			// No data to export or preview, return an empty array or suitable response.
			return array(
				'contacts'    => 0,
				'total'       => 0,
				'total_batch' => 0,
			);
		}

		// Prepare the contact data.
		$contacts = $this->prepare_contact_data( $contacts );

		// Extract required fields and meta_fields from contacts.
		$contacts = $this->extract_fields_and_meta_fields( $contacts, $fields );

		/**
		 * Get the export batch limit per operation.
		 *
		 * @param int $per_batch The default export batch limit per operation.
		 * @return int The modified export batch limit per operation.
		 *
		 * @since 1.5.0
		 */
		$per_batch = apply_filters( 'mint_export_batch_limit', 500 );

		return array(
			'contacts'    => $contacts,
			'total'       => $total,
			'total_batch' => ceil( $total / (int) $per_batch ),
		);
	}

	/**
	 * Summary: Prepares contact data by fetching additional information from the database.
	 * Description: Prepares the contact data by fetching additional information from the database,
	 * such as tags and lists associated with the contact.
	 *
	 * @access private
	 *
	 * @param array $contacts An array containing the contacts.
	 * @return array Returns the prepared contact data with additional information.
	 * @since 1.5.0
	 */
	private function prepare_contact_data( $contacts ) {
		// Check if $contacts is an array and not empty.
		if ( is_array( $contacts ) && !empty( $contacts ) ) {
			foreach ( $contacts as $key => $contact ) {
				// Get the contact details from the database based on the contact ID.
				$contact = ContactModel::get( $contact['id'] );
				$contact = ContactGroupModel::get_tags_to_contact( $contact );
				$contact = ContactGroupModel::get_lists_to_contact( $contact );
				// Update the contacts array with the prepared contact data.
				$contacts[ $key ] = $contact;
			}
		}
		return $contacts;
	}

	/**
	 * Summary: Extracts specific fields and their corresponding meta fields from an array of contacts.
	 * Description: Extracts the specified fields and their corresponding meta fields from the array of contacts.
	 *
	 * @access private
	 *
	 * @param array $contacts An array containing the contacts.
	 * @param array $fields An array containing the fields to extract.
	 * @return array Returns an array of extracted data containing the specified fields and meta fields.
	 * @since 1.5.0
	 */
	private function extract_fields_and_meta_fields( $contacts, $fields ) {
		$results = array();

		// Check if $contacts is an array and not empty.
		if ( is_array( $contacts ) && !empty( $contacts ) ) {
			foreach ( $contacts as $contact ) {
				// Merge the specified fields with the contact data and filter by $fields.
				$extracted_data = array_intersect_key( array_merge( array_fill_keys( $fields, '' ), $contact ), array_flip( $fields ) );

				// Check if $contact has 'meta_fields' and it is an array.
				if ( isset( $contact['meta_fields'] ) && is_array( $contact['meta_fields'] ) ) {
					foreach ( $fields as $field ) {
						if ( isset( $contact['meta_fields'][ $field ] ) ) {
							$extracted_data[ $field ] = $contact['meta_fields'][ $field ];
						}
					}
				}

				// Add the extracted data to the results array.
				$results[] = $extracted_data;
			}
		}
		return $results;
	}

	/**
	 * Exports contacts to a CSV file based on the provided parameters.
	 *
	 * @access public
	 *
	 * @param array $params An array containing the parameters for contact export.
	 *                      - 'selectedFields' (array): The selected fields to be exported.
	 *                      - 'segmentID' (int): The ID of the contact segment to export, if any.
	 *                      - 'fileName' (string): The name of the CSV file to be created for export.
	 *                      - 'offset' (int): The offset for pagination during export.
	 *                      - 'contactFields' (array): An array of available contact fields.
	 * @return array Returns an array containing the updated offset value for pagination.
	 * @since 1.5.0
	 */
	public function export_contacts_to_csv( $params ) {
		$selected_fields   = isset( $params['selectedFields'] ) ? $params['selectedFields'] : array();
		$segment_id        = isset( $params['segmentID'] ) ? $params['segmentID'] : 0;
		$file_name         = isset( $params['fileName'] ) ? $params['fileName'] : uniqid();
		$offset            = isset( $params['offset'] ) ? $params['offset'] : 0;
		$contact_fields    = isset( $params['contactFields'] ) ? $params['contactFields'] : array();
		$delimiter         = isset( $params['delimiter'] ) && 'comma' === $params['delimiter'] ? ',' : ';';
		$selected_contacts = isset( $params['selectedContacts'] ) ? $params['selectedContacts'] : array();

		/**
		 * Get the export batch limit per operation.
		 *
		 * @param int $per_batch The default export batch limit per operation.
		 * @return int The modified export batch limit per operation.
		 *
		 * @since 1.5.0
		 */
		$per_batch = apply_filters( 'mint_export_batch_limit', 500 );

		// Extracts headers for CSV file based on selected fields from the list of contact fields.
		$header_fields = $this->get_headers_to_csv( $contact_fields, $selected_fields );

		// Prepares the CSV file for export and writes the header row.
		$file_name = $this->prepare_csv_file( $file_name, $offset, $header_fields, $delimiter );

		if ( $segment_id ) {
			$segment_contacts = FilterSegmentContacts::get_segment( $segment_id, '', $offset, $per_batch );
			$contacts         = isset( $segment_contacts['contacts'][ 'data' ] ) ? $segment_contacts['contacts'][ 'data' ] : array();
		} elseif ( !empty( $selected_contacts ) ) {
			$all_contacts = ContactModel::get_all( $offset, $per_batch, '', $selected_contacts );
			$contacts     = isset( $all_contacts['data'] ) ? $all_contacts['data'] : array();
		} else {
			$all_contacts = ContactModel::get_all( $offset, $per_batch, '' );
			$contacts     = isset( $all_contacts['data'] ) ? $all_contacts['data'] : array();
		}

		// Prepare the contact data.
		$contacts = $this->prepare_contact_data( $contacts );

		// Extract required fields and meta_fields from contacts.
		$contacts = $this->extract_fields_and_meta_fields( $contacts, $selected_fields );

		// Write each data row to the CSV file.
		foreach ( $contacts as $contact ) {
			$csv_row = array_map( array( $this, 'convert_to_csv_string' ), $contact );
			$csv_row = $this->reorder_csv_row( $csv_row, $header_fields );
			fputcsv( $this->file_handle, $csv_row, $delimiter );
		}

		$file_path  = false;
		$upload_dir = wp_upload_dir();
		$baseurl    = ! empty( $upload_dir['baseurl'] ) ? $upload_dir['baseurl'] : '';

		if ( $baseurl ) {
			$file_path = $baseurl . '/mailmint-pro/exports/' . $file_name;
		}

		fclose( $this->file_handle ); // phpcs:ignore
		return array(
			'offset'    => $offset + (int) $per_batch,
			'file_path' => file_exists( MAIL_MINT_EXPORT_DIR . '/' . $file_name ) ? $file_path : false,
			'file_name' => $file_name,
		);
	}

	/**
	 * Reorders a CSV row based on the specified header fields.
	 *
	 * This function reorders the given CSV row according to the order of the header fields. It ensures
	 * that the CSV row data aligns with the header fields for consistent data export.
	 *
	 * @param array $csv_row An associative array representing a CSV row data.
	 * @param array $header_fields An associative array containing the header fields' slugs in the desired order.
	 * @return array The reordered CSV row data.
	 * @access private
	 * @since 1.5.0
	 */
	private function reorder_csv_row( $csv_row, $header_fields ) {
		$order  = isset( $header_fields['slug'] ) ? $header_fields['slug'] : array();
		$result = array();
		foreach ( $order as $key ) {
			if ( array_key_exists( $key, $csv_row ) ) {
				$result[ $key ] = $csv_row[ $key ];
			}
		}
		return $result;
	}

	/**
	 * Convert a value to a CSV-compatible string representation.
	 *
	 * @access private
	 *
	 * @param mixed $value The input value to convert.
	 * @return string The CSV-compatible string representation of the input value.
	 * @since 1.5.0
	 */
	private function convert_to_csv_string( $value ) {
		// Check if the value is an array.
		if ( is_array( $value ) ) {
			// If the array is empty, return an empty string.
			if ( empty( $value ) ) {
				return '';
			}
			// If the value is an array (tags/lists), get the titles and join them with commas.
			$titles = array_map(
				function( $item ) {
					// Check if $item is an object and has a 'title' property.
					if ( is_object( $item ) && property_exists( $item, 'title' ) ) {
						return $item->title;
					} else {
						// If not an object or 'title' property doesn't exist, return $item as is.
						return $item;
					}
				},
				$value
			);
			// Return the CSV-compatible string representation of the array.
			return implode( ',', $titles );
		} else {
			return $value;
		}
	}

	/**
	 * Extracts headers for CSV file based on selected fields from the list of contact fields.
	 *
	 * @access private
	 *
	 * @param array $contact_fields An array containing all contact fields.
	 * @param array $selected_fields An array containing the slugs of selected fields.
	 * @return array Modified $matched_fields array.
	 * @since 1.5.0
	 */
	private function get_headers_to_csv( $contact_fields, $selected_fields ) {
		if ( ! is_array( $contact_fields ) || ! is_array( $selected_fields ) ) {
			// Return an empty array or throw an exception, depending on your use case.
			return array(
				'slug' => array(),
				'name' => array(),
			);
		}

		// Initialize an array to store the matched fields' slugs and names.
		$matched_fields = array(
			'slug' => array(),
			'name' => array(),
		);

		// Loop through each contact field to check if it is selected.
		foreach ( $contact_fields as $field ) {
			if ( in_array( $field['slug'], $selected_fields, true ) ) {
				$matched_fields['slug'][] = $field['slug'];
				$matched_fields['name'][] = $field['name'];
			}
		}

		return $matched_fields;
	}

	/**
	 * Prepares the CSV file for export and writes the header row.
	 *
	 * @access private
	 *
	 * @param string $file_name The name of the CSV file.
	 * @param int    $offset    The offset for pagination.
	 * @param array  $header_fields An array containing the header fields.
	 * @param string $delimiter The delimiter used in the CSV file.
	 *
	 * @return array|int Returns an array with status 404 if there was an error opening the file, or nothing if the operation is successful.
	 * @since 1.5.0
	 */
	private function prepare_csv_file( $file_name, $offset, $header_fields, $delimiter ) {
		// Create the export directory if it doesn't exist.
		if ( ! file_exists( MAIL_MINT_EXPORT_DIR . '/' ) ) {
			wp_mkdir_p( MAIL_MINT_EXPORT_DIR );
		}

		// Remove any special characters from the filename and replace spaces with underscores.
		$file_name = preg_replace( '/[^A-Za-z0-9_\- ]/', '', $file_name );
		$file_name = str_replace( ' ', '_', $file_name );

		// Append ".csv" extension to the filename if it doesn't already have it.
		if ( pathinfo( $file_name, PATHINFO_EXTENSION ) !== 'csv' ) {
			$file_name .= '.csv';
		}

		if ( !file_exists( MAIL_MINT_EXPORT_DIR . '/' . $file_name ) && ! $offset && !$this->header_added ) {
			// Create the CSV file and write the header on the first call (offset is 0) and only if header not already added.
			$this->file_handle = fopen(MAIL_MINT_EXPORT_DIR . '/' . $file_name, 'wb'); // phpcs:ignore
			if ( empty( $this->file_handle ) ) {
				return array(
					'status' => 404,
				);
			}
			fputcsv( $this->file_handle, $header_fields['name'], $delimiter );
			$this->header_added = true;
		} elseif ( !file_exists( MAIL_MINT_EXPORT_DIR . '/' . $file_name ) && 0 !== $offset && !$this->header_added ) {
			// Open the same CSV file in append mode for subsequent calls.
			$this->file_handle = fopen(MAIL_MINT_EXPORT_DIR . '/' . $file_name, 'ab'); // phpcs:ignore
			fputcsv( $this->file_handle, $header_fields['name'], $delimiter );
			$this->header_added = true;
		} else {
			// Open the same CSV file in append mode for subsequent calls without rewriting the header.
			$this->file_handle = fopen(MAIL_MINT_EXPORT_DIR . '/' . $file_name, 'ab'); // phpcs:ignore
		}

		return $file_name;
	}

	/**
	 * Deletes the exported CSV file from the specified directory based on the provided API parameters.
	 *
	 * @access public
	 *
	 * @param array $params An array containing the API parameters for deleting the CSV file.
	 *                      - 'fileName' (string) The name of the CSV file to be deleted.
	 *
	 * @return bool Returns true if the file is deleted successfully, false otherwise.
	 * @since 1.5.0
	 */
	public function delete_csv_file( $params ) {
		$file_name = isset( $params['fileName'] ) ? $params['fileName'] : '';

		if ( file_exists( MAIL_MINT_EXPORT_DIR . '/' . $file_name ) ) {
			wp_delete_file( MAIL_MINT_EXPORT_DIR . '/' . $file_name );
		}

		return true;
	}
}
